; Copyright (c) 1998-2008 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	 AX_WRITE_XAS
;
;LAST CHANGED: ----------------------------------- 04-jun-08
;
;PURPOSE:
;	This procedure writes a 1d data structure (spectrum) as a
; nsls X1A *.xas format, annotated ascii file.
; The user is prompted for header information and can choose from
; nsls or gas-phase-database format for entries
;
;CATEGORY:
;	AXIS: utility
;
;CALLING SEQUENCE:
;	 AX_WRITE_XAS, tmp, file = file, nof = nof
;
;CALLED FROM AXIS:
;	->WRite->XAS
;
;INPUTS:
;	TMP - AXIS 1d-structure
;
;KEYWORDS:
;	FILE 	filename. If not supplied the user is prompted for the filename.
;	NOF		no filter (default filter is *.xas)
;	MULTI	write multiple spectra with user selectable point spacing
;
;OUTPUTS:
;	An XAS-format file is written.
;
;COMMON BLOCKS:
;	@AXIS_COM	standard set of common blocks
;
;MODIFICATION HISTORY:
; (31-jan-00 aph) first written to allow use of axis data as Io for Zimba stack_analyze.
; (27-feb-00 aph) AXIS standard documentation; groupID for get_text
; (04-Jun-08 aph) incorporate multi-spectral write (what is displayed)
;-
PRO ax_write_xas, tmp, file = file, multi=multi, nof = nof

@axis_com
on_error,2

if not keyword_set(file) then begin
	fltr='*.xas'
	if keyword_set(nof) then fltr='*'
	file = pickfile2(/write, FILTER=fltr, /LPATH, DefPath=WritePath) ;, get_path=WritePath)
	if strlen(file) EQ 0 THEN RETURN  ; bail-out if no filename
	t=ax_name(file)
    fileshort = t(1)
endif

IF keyword_set(multi) then begin

; write all spectra plotted on screen as a multi-column file
; filter each spectrum to isolate content between displayed x-limits
; isolate each spectrum within displayed range
; determine if this spectrum has more data points than currently save multi-array
; interpolate to same mesh and insert into multi-array
	xmin = Xrng(0)
	xmax = Xrng(1)
	print, 'Extracting data from ', xmin, ' to ', xmax
	HANDLE_value, Data(pBufs(0)), tmp
	e_pts = where(tmp.x GE xmin AND tmp.x LE xmax, count)
	if count EQ 0 then begin
		axis_log, 'First file out of current x-data range. Cannot write'
		return
	endif
	e=tmp.x(e_pts)
	d_tmp=tmp.d(e_pts)
	dx_min=ax_deltamin(e)
	FOR i=1, pNum-1 DO BEGIN
		test=dx_min
		HANDLE_value, Data(pBufs(i)), tmp
		e_pts = where(tmp.x GE xmin AND tmp.x LE xmax, count)
		if count GT 0 then test=ax_deltamin(tmp.x(e_pts))
		if test LT dx_min then dx_min = test
	ENDFOR
; identify dx_min
	axis_log,  'Minimum x-spacing ' + strtrim(string(dx_min),2)
	t = get_num(Prompt = 'X-axis spacing', val=dx_min, group=Axis_ID)
	print, 'new spacing', t
	if t(0) GE 0. then dx_min = t(0)
; generate mesh using dx_min
	npts=fix((xmax-xmin)/dx_min)+1
	axis_log,  '#-points ' + strtrim(string(npts),2)
	e = findgen(npts)
	e = xmin + e*dx_min
	print, 'E-axis" ', npts, ' from ',  e(0), ' to ', e(npts-1)
	d = fltarr(pNum,npts)
	FOR i=0, pNum-1 DO BEGIN
		HANDLE_value, Data(pBufs(i)), tmp
		e_pts = where(tmp.x GE xmin AND tmp.x LE xmax, count)
		if count EQ 0 then goto, skip_data
		e_tmp=tmp.x(e_pts)
		d_tmp=tmp.d(e_pts)
		if e_tmp(0) GT xmin then begin		;if missing low points, put in first point
			 e_tmp = [xmin, e_tmp]
			 d_tmp = [d_tmp(0),d_tmp]
		endif
		if e_tmp(n_elements(e_tmp)-1) LT xmax then begin	;if missing high points, put in last point
			 e_tmp = [e_tmp, xmax]
			 d_tmp = [d_tmp,d_tmp(n_elements(d_tmp)-1)]
		endif
		d(i,*) = Interpol(d_tmp, e_tmp, e)
	skip_data:	; in case there is not data in the range
	ENDFOR
; write multi-column file
;	file=pickfile2(/write, fltr='*.xas', TITLE='name for multi-column file')
	write_xas, file, e, d
ENDIF ELSE BEGIN

; ---- allow user to change label -----
 	text = Get_text(PROMPT = 'Label', val = tmp.dl, group = axis_ID)
	tmp.dl = text  		; update and save buffer label
	if widget_info(/active) EQ 1 then begin
		Label(CurBuf) = tmp.dl
		HANDLE_value, Data(CurBuf), tmp, /set
		BLbl_upd
	endif
	e = tmp.x
	d = tmp.d
ENDELSE

; - set header if desired ------------------------------
text = DIALOG_MESSAGE(/Question, 'Set header(Y/N)?',/DEFAULT_NO)
IF text  EQ 'Yes' then begin
	iseels=0
	; - select format of header -------
	text2 = DIALOG_MESSAGE(/Question,'Gas database format ? (No = NSLS format header)')
	if text  EQ 'Yes' then begin
		iseels=1
		t = ax_name(file)
		file = t(0)+t(1)+'.os'
		ax_header = {Formula: '', Name: '', Edge: '',$
	         Alpha_Formula: '', Data_Source: '',Filename: '',$
	         Ref_Code: '', Write_Date: '', Reference: '',$
	         Source_and_Purity: '', Acquisition_mode: '', File_names: ''}
	    ax_header.formula = get_text(prompt='Formula', val='', group = axis_ID)
	    ax_header.Name = get_text(prompt='Name', val='', group = axis_ID)
	    ax_header.Edge = get_text(prompt='Edge', val='', group = axis_ID)
	    ax_header.Alpha_Formula = get_text(prompt='Alpha_Formula',val='', group = axis_ID)
	    ax_header.Data_Source = get_text(prompt='Data_Source', val='', group = axis_ID)
	    filename = fileshort +'.os'
	    ax_header.Filename = get_text(prompt='Filename', val=filename, group = axis_ID)
	    ax_header.Ref_Code = get_text(prompt='Ref_Code', val='', group = axis_ID)
	    ax_header.Write_Date = get_text(prompt='Write_Date',val='', group = axis_ID)
	    ax_header.Reference = get_text(prompt='Reference', val='', group = axis_ID)
	    ax_header.Source_and_Purity = get_text(prompt='Source_&_Purity', val='', group = axis_ID)
	    ax_header.Acquisition_mode = get_text(prompt='Acquisition_mode', val='', group = axis_ID)
	    ax_header.Acquisition_mode = get_text(prompt='File_names', val='', group = axis_ID)
 	endif else begin
		ax_header = {header,$
	         formula: '',common_name: '',edge: '',$
	         acquisition_mode: '',source_purity: '',comments: '',$
	         delta_ev: '',min_ev: '',max_ev: '',yaxis: '', $
	         contact_person: '',write_date: '',journal: '',$
	         authors: '',title: '',volume: '',$
	         issue_number: '',year: '',pages: '',$
	         booktitle: '',editors: '',publisher: '',$
	         address: ''}
; ------ allow user to define all values for header
		ax_header.formula = get_text(prompt='formula', val=tmp.dl, group = axis_ID)
		ax_header.common_name = get_text(prompt='common_name', val='', group = axis_ID)
		ax_header.edge = get_text(prompt='edge', val='', group = axis_ID)
		ax_header.acquisition_mode = get_text(prompt='acquisition_mode', val='', group = axis_ID)
		ax_header.source_purity = get_text(prompt='source_purity', val='', group = axis_ID)
		ax_header.comments = get_text(prompt='comments', val=tmp.dl, group = axis_ID)
		ax_header.delta_ev = get_text(prompt='delta_ev', val='', group = axis_ID)
		ax_header.min_ev = min(tmp.x, max = max_ev)
		ax_header.max_ev = max_ev
		ax_header.yaxis = get_text(prompt='yaxis', val='', group = axis_ID)
		ax_header.contact_person = get_text(prompt='contact_person', val='', group = axis_ID)
		ax_header.write_date = ''
		ax_header.journal = get_text(prompt='journal', val='', group = axis_ID)
		ax_header.authors = get_text(prompt='authors', val='', group = axis_ID)
		ax_header.title = get_text(prompt='title', val='', group = axis_ID)
		ax_header.volume = get_text(prompt='volume', val='', group = axis_ID)
		ax_header.issue_number = get_text(prompt='issue_number', val='', group = axis_ID)
		ax_header.year = get_text(prompt='year', val='', group = axis_ID)
		ax_header.pages = get_text(prompt='pages', val='', group = axis_ID)
		ax_header.booktitle = get_text(prompt='booktitle', val='', group = axis_ID)
		ax_header.editors = get_text(prompt='editors', val='', group = axis_ID)
		ax_header.publisher = get_text(prompt='publisher', val='', group = axis_ID)
		ax_header.address = get_text(prompt='address', val='', group = axis_ID)
	endelse
; -- write single or multi-column data to ascii file ---
	write_xas, file, e,d, ax_header, iseels=iseels
endif else 	write_xas, file, e,d, formula = tmp.dl

axis_log, 'Spectrum written to XAS file: ' + fileshort

END
