; Copyright (c) 1998-2014 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	PLOT2D
;
;LAST CHANGED: ----------------------------------- 23 July 2014
;
;PURPOSE:
;	This generates a 2d image plot with labels and user controllable limits.
; It is intermediate between splot2d and the IDL plot routine.
;
;CATEGORY:
;	AXIS: utility; also STAND ALONE: image processing
;
; CALLING SEQUENCE
;  	PLOT2D, G_data, xx, yy, labels, /LOG, autosize=as, wsize=s, /NORESIZE, _extra=e
;
;CALLED FROM AXIS:
;	->via splot2d routine exclusively
;
;INPUTS:
;	G_data	(z) data to plot
;	xx		(X) - values
;	yy		(Y) - values
;	labels  labels for x & y axes

;KEYWORDS:
;	LOG		- plot log(G-data)
;	WSIZE	- specify IDL window size
;	AUTOSIZE - uses dimensions to specify IDL window size
;	NORESIZE - ue current window size
;
;OUTPUTS: none
;
;COMMON BLOCKS:
;	@AXIS_COM	standard set of common blocks
;
; PROCEDURE
; generalized 2D intensity plot with x and y axes
; assumes equal pixel unit increments to determine window aspect ratio
;
;MODIFICATION HISTORY:
; 2/13/95  er add keyword /NORESIZE to PLOT2D
; 6/15/96  jdd move SPLOT stuff to separate procedure
;
; ----------- AXIS modifications -------
; (25-jun-99 aph) plot2d: allow co-existing image/spectra color tables (ax_imscl, ax_color)
; ( 6-jul-99 aph) plot2d: allow negative-going plots; try to avoid funny colored backgrounds
; ( 3-oct-99 aph) plot2d: removed explicit color, back statements !
; (15-apr-00 aph) AXIS standard documentation
; (14-nov-00 cgz) Replaced calls to GET_RANGE function with ecplicit MIN and MAX calls
; (15-nov-00 cgz) Replaced call of NCOLS function by explicit (SIZE(var))(1)
; (15-nov-00 cgz) Replaced call of NROWS function by explicit (SIZE(var))(2)
; (30-nov-00 cgz) Replaced call of nwin with explicit window statement
; (19-dec-00 cgz) substituted ax_white_color_index with ax_plot_axes_color_index
;			substituted ax_black_color_index with ax_plot_bkgd_color_index
; (30-dec-00 aph) set XSTYLE=9, YSTYLE=9 for frame (1-sided axes)
;				  set Xmargin=[8,0], Ymargin = [4,2]
; (22-mar-06 aph) try to increase size of labels; allow room for Y-axis label
; (23-jul-14 aph)  remove ticks & increase size of main window; adjust margins so side windows line up
;                 which required changes to axis_Dialog, axis_c; implement Plot_Axes_Flag switch
;-

pro PLOT2D, G_data, xarr, yarr, labels, log=il, $
         autosize=as, wsize=ws, noresize=nores, _extra=e
;************************************
; generalized 2D intensity plot with x and y axes
; assumes equal pixel unit increments to determine window aspect ratio

@axis_com
on_error,2

xt=' '
yt=' '
CASE n_elements( labels ) OF
1: begin
     xt=labels(0) & yt=labels(0)
   end
2: begin
     xt=labels(0) & yt=labels(1)
   end
ELSE:
ENDCASE

;xr=get_range(xarr)		; this forces low-to-high plotting
;yr=get_range(yarr)		; DOES NOT ALLOW for reversed axes
xr=[min(xarr),max(xarr)]		; this forces low-to-high plotting
yr=[min(yarr),max(yarr)]		; DOES NOT ALLOW for reversed axes
;print, 'xyr', xr, yr

if keyword_set(il) then d=alog(d>0.001)

;nc=ncols(G_data)
nc=(size(G_data))(1)
;nr=nrows(G_data)
nr=(size(G_data))(2)
resize=0	;default to no window resize
if !d.window eq -1 then resize=1
if keyword_set( as ) then resize=1
if keyword_set( nores) then resize=0
if keyword_set( ws ) then resize=1

if resize then begin
   aspect=nr/float(nc)
   ;print, 'size', nc, nr, aspect
   if keyword_set(ws) eq 0 then ws=max(300, nc)
   wsx=ws*1.13                   ; reduced from 1.24 (22-mar-06 aph)
   wsy=ws*aspect*1.05            ; reduced from 1.15 (22-mar-06 aph)
   ; print, 'wsxy', wsx, wsy

  if (!D.NAME eq 'X') or (!D.NAME eq 'MAC')or (!D.NAME eq 'WIN') then $
;     nwin, (!d.window-(!d.window<0)), wsx, wsy
	window, !d.window>0, xsize=wsx, ysize=wsy
endif

; ------------- plot with or without axes & tics, depending on plot_axes_flag
; plot frame
if Axes EQ 1 then begin		; plot with labelled x,y axes
plot, xr, yr, /nodata, color=ax_plot_bkgd_color_index, back=ax_plot_axes_color_index, $
     xtitle=xt, ytitle=yt, xstyle=9, ystyle=9,_extra=e, $;
	 Xmargin=[8,0], Ymargin = [4,2], charsize = 0.8, ticklen=-!p.ticklen
endif else begin			; plot without any axes
	plot, xr, yr, /nodata, color=ax_plot_bkgd_color_index, back=ax_plot_axes_color_index, $
	     charsize = 0.8, xtitle=xt, ytitle=yt, xstyle=5, ystyle=5, Xmargin=[6,2], Ymargin = [2,2],  _extra=e ;
endelse

pxy=convert_coord( xr, yr, /data, /to_device)
nx=pxy(0,1)-pxy(0,0)
ny=pxy(1,1)-pxy(1,0)
; ------------- reverse image elements if x or y axes are reversed (assumes ORDERED xarr, yarr)
; if xarr(0) GT xr(0) then G_data = reverse(G_data)
; if yarr(0) GT yr(0) then G_data = reverse(G_data,2)
; ------------------------------------ modified for reduced color index range
zdata = ax_imscl(G_data)
zdata =  congrid(zdata, nx, ny, /INTERP)
tv, zdata, pxy(0,0)+1, pxy(1,0)+1

end
