 ; Copyright (c) 1998-2001 A.P. Hitchcock  All rights reserved
;+
;NAME: READ_SDF
;
;LAST CHANGED: ----------------------------------- 17-aug-01
;
; PURPOSE:
;	This function reads self-defining format ascii files used in ALS Beamline 5.3.2 STXM.
; Spectra, images, stacks and linescans have *.hdr and a set of *.xim files
; Image modes have separate file for each {energies.channels.regions}
; Spectra are a *#.xsp file for each region (#) containing header and multi-column spectra.
;
; CATEGORY:
;	Input / output utilty
;
; CALLING SEQUENCE:
;	Result = READ_SDF(file, nof=nof, filter=filter, verbose=verbose, groupID=groupID, header_only=header_only, _extra=e)
;
; INPUTS:
; 	FILE	optional file name
;
; KEYWORDS:
;	FILTER	   user defined filter
;	VERBOSE		print out details of parsing structure (testing)
;	CHANNEL		preselected data channel
;	REGION		preselected region
;	GROUPID		group leader (Axis_ID if called from aXis2000 and axis)
;	HEADER_ONLY extract and return header contents only
;	_EXTRA		other passed through parameters
;
; OUTPUTS:
;	structure variable; either 1d (spectra) or 2d (image, linescan)
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;	SDFORMAT	reference number for naming muti-element arrays of structures
;
; MODIFICATION HISTORY:
; (11-feb-01 aph) first version
; (23-feb-01 aph) arrays of structures implemented
; (25-feb-01 aph) option to return structure only implemented
; (27-jun-01 aph) adapt to read focus and OSA scans
; (12-jul-01 aph) enable read-in of spectral modified format
; (29-jul-01 aph) retired read_bl5 spectral read-in; new format with *.hdr is in place
;               regions implemented for spectra
; (14-aug-01 aph) fixed errors found by Ivo in spectra & linescan read-in
; (17-aug-01 aph) adapt to REGIONS (kluge needed !)
;-

function read_sdf, file, filter=filter, channel=channel, region = region, $
         verbose=verbose, groupID=groupID, header_only=header_only, _extra=e
on_error,2
@axis_com
common sdformat, starnum

t = size(starnum)
; print, t
if t(1) EQ 0 then starnum = 0   ; start of unique names - SHOULD USE A CHECK for existence
; print, 'starnum ',  starnum

if n_elements(file) eq 0 then begin  ;popup file dialog box
   if not keyword_set(filter) then fltr='*.hdr' else fltr = filter
   file=PICKFILE2(/Read, FILTER=fltr, /LPATH, DEFPATH=defpath)
endif
s = 0
if strlen(file) LE 0 THEN return, s  ; bail-out if no filename
; ------- Open the header to be read -------
; first force to be a header
t = ax_name(file)
fileshort = t(1)
file_hdr = t(0) + t(1) + '.hdr'
t = findfile(file_hdr)
;IF strlen(t(0)) EQ 0 then begin
;	s = read_bl5(file, group = Axis_ID)		; kluge until spectra format split
;	return, s
;ENDIF
; -----------
on_ioerror,nofile
get_lun,lun
openr,lun,file_hdr
; ------- Read header file into a single string (max = 32600 characters)  --------
line = ' '
all = ''
count = 0
while not  eof(lun) do begin
	readf,lun, line
;		print, line				; debug to ensure full file read-in
	all = all + line
	count = count + 1			; the number of lines is used in spectral readin
endwhile
close, lun
on_ioerror,null

; format for recursive structure parsing
all = '{'+all+'};'
; return, all

; --------- parse the structure --------------
if keyword_set(verbose) then verbose = 1 else verbose = 0

reg = strpos(all,'Regions')
if reg GT 0 then begin
	b = strmid(all,0, reg-1) + '}'
	ScanDef = parse_sdf(b,verbose=verbose)
;	return, ScanDef
;	help, ScanDef.SCANDEFINITION, /struct
	ScanDef = ScanDef.ScanDefinition
	stax = strpos(all,'StackAxis')
; -- isolate regions information
	regions = strmid(all, reg, stax-reg)
	t = strpos(regions,'(') & tc = strpos(regions,',')
	n_regions = 	fix(strmid(regions,t+1,tc-t-1))
	for i = 0, n_regions-1 do begin
		pax = strpos(regions,'PAxis') & paxe = strpos(regions,'};',pax)
		paxis = strmid(regions,pax,paxe-pax)
		p = parse_sdf('{'+paxis,verbose=verbose)
		qax = strpos(regions,'QAxis') & qaxe = strpos(regions,'};',qax)
		qaxis = strmid(regions,qax,qaxe-qax)
		q = parse_sdf('{'+qaxis,verbose=verbose)
		pn = 'PAxis_'+strtrim(string(i),2)
		ScanDef = create_struct(ScanDef,pn,p.paxis)
		qn = 'QAxis_'+strtrim(string(i),2)
		ScanDef = create_struct(ScanDef,qn,q.qaxis)
	endfor
	stax = strpos(all,'StackAxis') & end_part = '{' + strmid(all,stax-1)
	ScanDef_end1 = parse_sdf(end_part,verbose=verbose)
	end1 = tag_names(ScanDef_end1)
	q = where(end1 EQ 'CHANNELS', count)
	if count GT 0 then begin
		print, ' Channels replaced by Channel_labels'
		end1(q(0)) = 'Channel_labels'
		for i = 0, n_elements(end1)-1 do begin
			tl = end1(i)
			ScanDef = create_struct(ScanDef,tl, scandef_end1.(i))
		endfor
	endif else begin
		for i = 0, n_tags(ScanDef_end1)-1 do $
		ScanDef = create_Struct(ScanDef, end1(i), ScanDef_end1.(i))
	endelse
	SD_names = tag_names(ScanDef)
	for i = 0, n_tags(ScanDef)-1 do print, SD_names(i)
	stax = strpos(all,'StackAxis') & end_part = '{END= {' + strmid(all,stax-1)
	ScanDef_end2 = parse_sdf(end_part,verbose=verbose)
	end2 = tag_names(ScanDef_end2)
	q = where(end2 EQ 'BEAMFEEDBACK', count)
	if count GT 0 then for i = q(0), n_elements(end2)-1 do print, end2(i)
	if count GT 0 then begin
		for i = q(0)+1, n_tags(ScanDef_end2)-1 do $
		ScanDef = create_Struct(ScanDef, end2(i), ScanDef_end2.(i))
	end
;	return, ScanDef
; ------ handle old type files
endif else  ScanDef = parse_sdf(all)
	return, ScanDef
; help, /struct, ScanDef

if keyword_set(header_only) then begin
; ------- print header information
;	XDisplayFile,file_hdr
	return, ScanDef
endif

; --------- Read in data according to the type of scan -----

n_channels = n_elements(ScanDef.Channel_labels)
if n_channels EQ 0 then n_channels = n_elements(ScanDef.ScanDefinition.channels)
n_data =-1
if NOT keyword_set(channel) then begin
	if keyword_Set(groupID) then Widget_CONTROL, Uprompt, Set_Value = '5.3.2 data from '+ fileshort
	Print, file, '  has ', strtrim(string(n_channels),2), ' data channels'
		for i = 0, n_channels-1 do begin
		text = string(i+1) + '  ' + ScanDef.ScanDefinition.channels[i].name
		print, text
		if keyword_Set(groupID) then Widget_CONTROL, Uprompt, Set_Value = text, /append
	endfor
	while n_data LE 0 OR n_data GT n_channels do begin
		if keyword_set(groupID) then $
		  n_data = get_num(prompt='Choose data channel', val=2, group=groupID) $
		   else n_data = get_num(prompt='Choose data channel', val=2)
	endwhile
endif else 	n_data = channel
case n_data of
	1 : let = 'a'
	2 : let = 'b'
	3 : let = 'c'
	4 : let = 'd'
	5 : let = 'e'
else : print, 'Only set up for 5 channels max'
endcase

; -------  identify number of regions and allow user to select a region
t = where(tag_names(ScanDef.ScanDefinition) EQ 'REGIONS', count)
if count GT 0 then begin		; only execute region dialog if scandefinition contains regions
	n_regions = n_elements(ScanDef.ScanDefinition.regions)
	n_region = -1
	print, 'data file has ',strtrim(string(n_regions),2), ' regions'
	if  NOT keyword_set(region) then begin
		while n_region LE 0 OR n_region GT n_regions do begin
			if keyword_set(groupID) then $
			  n_region = get_num(prompt='Choose region', val=2, group=groupID) $
			   else n_region = get_num(prompt='Choose region', val=2)
		endwhile
	endif else n_region = region
	n_region = fix(n_region)
endif else n_region = 1

TYPE = ScanDef.ScanDefinition.type
print, 'reading Self Defining Data file of type ', type
CASE TYPE of
	'Image Scan' : BEGIN
		x = float(ScanDef.ScanDefinition.PAxis.points)
		nx = n_elements(x)
		y = float(ScanDef.ScanDefinition.QAxis.points)
		ny = n_elements(y)
		t = ax_name(file)
		img_file = t(0) + t(1) + '_' + let +'.xim'
		d = text_read1(img_file)
;	------ pack into AXIS 2d structure -----------------
		energy = ScanDef.ScanDefinition.StackAxis.min
		dwell = ScanDef.ImageScan.EnergyRegions[0].DwellTime
		xl = string(FORMAT='("x (um)     E = ",f8.3," eV     dwell = ",f5.2," ms")', $
     			 energy, dwell)
		t = ax_name(file)
		fileshort = t(1)
	s = {t:'2d', x:x, y:y, d:d, e: energy, xl:xl, yl:'um  ch '+let, dl: fileshort}
		text = string(format= $
	  '(i4," x",i4," image read from ",A,". E =",f7.2," eV. Dwell =",f5.2," ms.")', $
	   nx, ny, file, energy, dwell)
	print, text
	if keyword_Set(groupID) then Widget_CONTROL, Uprompt, Set_Value = text, /append
	return, s
	END

	'Focus Scan' : BEGIN
		x = float(ScanDef.ScanDefinition.PAxis.points)
		nx = n_elements(x)
		y = float(ScanDef.ScanDefinition.QAxis.points)
		ny = n_elements(y)
		t = ax_name(file)
		img_file = t(0) + t(1) + '_' + let +'.xim'
		d = text_read1(img_file)
;	------ pack into AXIS 2d structure -----------------
		energy = ScanDef.ScanDefinition.StackAxis.min
		dwell = ScanDef.FocusScan.DwellTime
		xl = string(FORMAT='("(um)     E = ",f8.3," eV     dwell = ",f5.2," ms")', $
     			 energy, dwell)
		t = ax_name(file)
		fileshort = t(1)
	s = {t:'2d', x:x, y:y, d:d, e: energy, xl:xl, yl:'Zone Plate Z (um)  ch '+let, dl: fileshort}
		text = string(format= $
	  '(i4," x",i4," focus scan read from ",A,". E =",f7.2," eV. Dwell =",f5.2," ms.")', $
	   nx, ny, file, energy, dwell)
	print, text
	if keyword_Set(groupID) then Widget_CONTROL, Uprompt, Set_Value = text, /append
	return, s
	END

	'OSA Scan' : BEGIN
		x = float(ScanDef.ScanDefinition.PAxis.points)
		nx = n_elements(x)
		y = float(ScanDef.ScanDefinition.QAxis.points)
		ny = n_elements(y)
		t = ax_name(file)
		img_file = t(0) + t(1) + '_' + let +'.xim'
		d = text_read1(img_file)
;	------ pack into AXIS 2d structure -----------------
		energy = ScanDef.ScanDefinition.StackAxis.min
		dwell = ScanDef.OSAScan.DwellTime
		xl = string(FORMAT='("OSA x (um)     E = ",f8.3," eV     dwell = ",f5.2," ms")', $
     			 energy, dwell)
		t = ax_name(file)
		fileshort = t(1)
	s = {t:'2d', x:x, y:y, d:d, e: energy, xl:xl, yl:'OSA y (um)  ch '+let, dl: fileshort}
		text = string(format= $
	  '(i4," x",i4," OSA scan read from ",A,". E =",f7.2," eV. Dwell =",f5.2," ms.")', $
	   nx, ny, file, energy, dwell)
	print, text
	if keyword_Set(groupID) then Widget_CONTROL, Uprompt, Set_Value = text, /append
	return, s
	END

	'NEXAFS Point Scan': BEGIN
	; ------- open file for selected region - each region is in separate file
		t = ax_name(file)
		file = t(0) + t(1) + '_' + strtrim((string(n_region-1, format='(A)')),2) + '.xsp'
;		print, 'opening data file: ', file
		openr,lun,file, /get_lun			; re-open to get data
;		for i = 0, (count - npts - 1) do readf,lun,test            ; ONLY when header at top
; ------ now positioned at start of multi-column data
		test = ''
		tr = tag_names(ScanDef.ScanDefinition)
;		print, 'tags', tr
		t = where(tr EQ 'Regions', tc)
		if tc GT 0 then  energy = float(ScanDef.ScanDefinition.Regions.(0).points) $
		  else energy = float(ScanDef.ScanDefinition.PAxis.points)
		energy = reform(energy(*,n_region-1))
		npts = n_elements(energy)
;		print, 'E-axis', npts, ' points from ', energy(0), ' to ', energy (n_elements(energy)-1), ' eV.'
		d = fltarr(npts)  & i =0
		ldata = fltarr(n_channels+1)
;		Print, file, '       # points: ', strtrim(string(npts),2),'      # data channels: ', strtrim(string(n_channels),2)
		for i = 0, npts-1 do begin
			readf,lun, line
;					print, 'line ', i,': ', line
			reads, line, ldata
			energy(i) = ldata(0)
			d(i)  = ldata(n_data)
		endfor
		close, lun
;		plot, energy,d

; --------------- generate axis 1d structure -----
		dwell_time = ScanDef.ImageScan.EnergyRegions[0].DwellTime
		t = ax_name(file)
		fileshort = t(1)
		xl = string(FORMAT='("Energy(eV)    dwell = ",f7.2," ms")', dwell_time)
		tmp = {t:'1d', x:energy, d:d, dn:d, xl:xl, dl:fileshort}
		print, 'read ALS BL5.3.1 NEXAFS spectrum: ', file
		npts = n_elements(tmp.x)
		text = string(format='(i4," pts. ",f9.4," to ",f9.4," eV. Dwell= ",f7.2," ms.")', $
		    npts, tmp.x(0), tmp.x(npts-1), dwell_time)
		print, text
		if keyword_Set(groupID) then Widget_CONTROL, Uprompt, Set_Value = text, /append
		return, tmp
	END

	'NEXAFS Image Scan': BEGIN
		Print, 'Reading in ALS Beamline 5.3.1 Stack'
		text = 'ALS 5.3 stack read in: not yet developed'
		print, text
		if keyword_Set(groupID) then Widget_CONTROL, Uprompt, Set_Value = text
		s = 0
		return,s
	END

	'NEXAFS Line Scan': BEGIN
		x = float(ScanDef.ScanDefinition.PAxis.points)
		y = float(ScanDef.ScanDefinition.QAxis.points)
		dwell = ScanDef.ImageScan.EnergyRegions[0].DwellTime
		t = ax_name(file)
		img_file = t(0) + t(1) + '_' + let +'.xim'
		d = float(text_read1(img_file))
		help, d
		xl = string(FORMAT='("x (um)   dwell = ",f5.2," ms")', $
     			 dwell)
		s = {t:'2d', x:x, y:y, d:d, e: 0, xl:xl, yl:'um  ch '+let, dl: fileshort}
		text = string(format= $
	  	    '(" Linescan read from ",A,". Emin =",f7.2," eV. Emax =",f7.2," eV.  Dwell =",f5.2," ms.")', $
	   		   file, x(0), x(n_elements(x)-1), dwell)
		print, text
		if keyword_Set(groupID) then Widget_CONTROL, Uprompt, Set_Value = text, /append
		return,s
	END
	; -------- bailout -------------------------
	ELSE: BEGIN
		print, 'Unknown scan type'
	ENDELSE
ENDCASE
return, scandef

nofile:
print, 'Open or read error on ', file
close,/all
return, s
end

