;+
; NAME:
;	ZSTACK_READLIST.PRO
; AUTHORS:
;	Carl G. Zimba (NIST), Chris Jacobsen (SUNY - Stonybrook)
; PURPOSE:
;	Forms list of image filenames from file containing such a list
;	Called by ZSTACK_ANALYZE.PRO
; CATEGORY:
;	Data analysis.
; CALLING SEQUENCE:
;	zstack_readlist, saved_list_filename
; INPUTS:
;	saved_list_filename = name of file containing a list of image files
;		(previously developed by zstack_buildlist within zstack_analyze)
; KEYWORD PARAMETERS:
;
; OUTPUTS:
;	filename_list = list of image filenames
; COMMON BLOCKS:
;	zstack_common	:				
;		image_stack			:	3-D matrix of multiple x-ray microscope images
;		filename_list			:	list of filename corresponding to images in image_stack
;		ev_list				:	list of x-ray energies corresponding to images in image_stack
;		msec_list				:	list of dwell times corresponding to images in image_stack
;		filename_display_list		:	list of filename, x-ray energy, and dwell time corresponding to images in image_stack
;		displayed_file_index		:	index in image_stack, filename_list, msec_list, and filename_display_list currently being displayed or processed
;		data_source			:	type of x-ray microscopy data: STXM:1, SXM:2, SM:3, ALS:4, POLY:5
;		image_zoom				:	zoom factor for displaying images
;		init_zoom				:	initial zoom factor for displaying images (used in ZSTACK_SAVE and ZSTACK_TUNE)
;		disp_min				:	minimum percentage intensity for display of images
;		disp_max				:	maximum percentage intensity for display of images
;		disp_gamma				:	gamma factor for display of images
;		movie_delay			:	delay used to display movie images of data stacks
;		n_files				:	number of images
;		list_filename			:	filename of image list
;		shift_filename			:	filename of alignment shifts
;		x_shift				:	array of x-coordinate alignment shifts
;		y_shift				:	array of y-coordinate alignment shifts
; SIDE EFFECTS:
;
; RESTRICTIONS:
;
; PROCEDURE:
;	Called by ZSTACK_ANALYZE.PRO
;	Also called by ZSTACK_MOVIE.PRO and ZSTACK_PSPRINT.PRO
; EXAMPLE:
;       
; MODIFICATION HISTORY:
; 20feb99, CGZ
;	Renamed ZSTACK_READLIST.PRO for use in ZSTACK_ANALYZE.PRO
;	Adjusted to accommodate directory header in *.sl file
;	Removed contitional test for valid saved_list_filename
;		This is done is ZSTACK_ANALYZE.PRO - no need for it here
;		Eliminated call to ZSTACK_BUILDLIST.PRO which was used if saved_list_filename was not valid
;		Eliminated stxm1_sxm2_sm3_als4 variable which was used only by zstack_buildlist call
;		Modified ZSTACK_MOVIE.PRO and ZSTACK_PSPRINT.PRO to reflect this change
;	Can this be used to read saved_list_filename from within STACK Buildlist dialog window ?
;	Modified COMMON block structure so that common variables are shared with all zstack routines
;		i.e., zstack_common contains variables used by 
;		ZSTACK_ANALYZE.PRO, ZSTACK_READLIST.PRO, ZSTACK_BUILDLIST.PRO, and ZSTACK_ALIGN.PRO
;	Common variables for data are now determined within ZSTACK_READLIST.PRO:
;		image_stack, filename_list, ev_list, msec_list, filename_display_list
;
; (13nov00 cgz)  Added calculation of initial zoom factor, based upon image size
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO zstack_readlist, saved_list_filename

;print,'zstack_readlist'

COMMON zstack_common, $
	data_source, data_directory, image_stack, $
	filename_list, ev_list, msec_list, filename_display_list, displayed_file_index, $
	n_files, list_filename, shift_filename, x_shift, y_shift, $
	data_shifted, n_clipped_cols, n_clipped_rows, clipbox, $
	reference_spectrum, reference_image, reference_stack, axis_call

COMMON zstack_display_common, $
	zstack_display_par, zstack_plot_par, zstack_pickcolor_par, $
	image_zoom, subimage_zoom, movie_delay, disp_min, disp_max, disp_gamma, $
	spectrum_display, spectrum_offset, init_zoom, movie_active, profile_zoom, $
	image_range, image_ratio, ratio_image_index, image_scale, image_invert, $
	temp_old_display, temp_new_display, zstack_subroutine, $
	plot_x_min, plot_x_max, plot_y_min, plot_y_max, x_autoscale, y_autoscale

@bsif_common

get_lun,lun
on_ioerror,stack_readlist_bailout
openr,lun,saved_list_filename
got_started = -1
keep_going = 1

; Get directory from first lines of file
; then second line starts filename_list
; and all subsequent lines are appended to filename_list
WHILE (keep_going eq 1) DO BEGIN
	this_filename = ''
	readf,lun,this_filename
	IF (got_started eq -1) THEN BEGIN
;		print,3,'  ',got_started,'  ',this_filename
		cd,current=current_directory
		data_directory = this_filename
		IF (current_directory NE data_directory) THEN BEGIN
			; test to see if the first line of saved_list_filename, 
			;	i.e., first value of this_filename, is directory path
			;	if it has the proper structure for a directory path, 
			;	firstpos will be greater than 0
			osname = strupcase(strmid(!Version.os, 0, 3))
			CASE osname OF
				'VMS' : BEGIN
					firstpos = strpos(data_directory, ']')+1
				END
				'WIN' : BEGIN
					firstpos = rstrpos(data_directory, '\')+1
				END
				'MAC' : BEGIN
					firstpos = rstrpos(data_directory, ':')+1
				END
				'UNIX' : BEGIN
					firstpos = rstrpos(data_directory, '/')+1
				END
				ELSE : BEGIN
					print,'Unrecognized operating system'
					firstpos = -1
				END
			ENDCASE
			IF (firstpos GT 0) THEN BEGIN
				; change directory if first line appears to be a directory
				dummy = findfile(data_directory+'*.*',count=count)
				IF (count NE 0) THEN BEGIN
					cd,data_directory
				ENDIF ELSE BEGIN
					print,'Find alternative path for data files'
					
		dummy = dialog_pickfile( /read, /must_exist, $
			path=data_directory, get_path=data_directory, /directory, $
			title='Select desired directory :')
        		; /directory option will display only directories, not individual files
        IF (dummy EQ '') THEN data_directory = current_directory
			; sets data_directory to initial directory if CANCEL is chosen during dialog
			; without this, data_directory becomes null string

        cd,data_directory	; change directory to data_directory, CGZ
        last_char = strmid(data_directory,strlen(data_directory)-1,1)

		CASE strupcase(!version.os_family) OF
			'VMS' : BEGIN
				IF (last_char NE ']') THEN BEGIN
					data_directory = data_directory+']'
				ENDIF
			END
			'WINDOWS' : BEGIN
				IF (last_char NE '\') THEN BEGIN
					data_directory = data_directory+'\'
				ENDIF
			END
			'MACOS' : BEGIN
				IF (last_char NE ':') THEN BEGIN
					data_directory = data_directory+':'
				ENDIF
			END
			'UNIX' : BEGIN
				IF (last_char NE '/') THEN BEGIN
					data_directory = data_directory+'/'
				ENDIF
			END
			ELSE : BEGIN
				print,'Unrecognized operating system'
			END
		ENDCASE



				ENDELSE
				print,'Changed directory to ',data_directory
				got_started = 0
			ENDIF ELSE BEGIN
				; start making filename_list
				filename_list = [this_filename]
				got_started = 1
			ENDELSE
		ENDIF
	ENDIF ELSE BEGIN
		IF (got_started EQ 0) THEN BEGIN
;			print,1,'  ',got_started,'  ',this_filename
			filename_list = [this_filename]
			got_started = 1
		ENDIF ELSE BEGIN
;			print,2,'  ',got_started,'  ',this_filename
			filename_list = [temporary(filename_list),this_filename]
		ENDELSE
	ENDELSE
ENDWHILE

stack_readlist_bailout:
close,lun
free_lun,lun

print,'Reading in images from '+saved_list_filename+'. Please be patient...'
print,"... or I suppose that I'll have to use force !"
start_time = systime(1.)

n_files = n_elements(filename_list)
ev_list = 0
msec_list = 0
filename_display_list = 0
FOR i=0,(n_files-1) DO BEGIN
	this_filename = filename_list(i)
	CASE data_source OF
		1 : BEGIN	; X1A STXM data
			read_stxm,this_filename,sd,khz
			this_ev = 12398.52/sd.wavelength
			this_msec = sd.dwell_time
			this_file_text_label = this_filename+': '+$
					strtrim(string(this_ev, format = '(f10.2)'), 2)+' eV, '+$
					strtrim(string(sd.wavelength, format = '(f10.3)'), 2)+' A'
;					strtrim(string(sd.dwell_time, format='(i6)'), 2)+' msec, '+$
;					strtrim(string(n_cols),2)+'x'+strtrim(string(n_rows),2)
		END
		2 : BEGIN	; X1A cryo-STXM data
			read_sxm,this_filename
			khz = float(image_data(*,*,0))
			this_ev = 1239.852/sxm_par.now_nm
			this_msec = sd.dwell_time
			this_file_text_label = this_filename+': '+$
					strtrim(string(this_ev, format = '(f10.2)'), 2)+' eV, '+$
					strtrim(string(sd.wavelength, format = '(f10.3)'), 2)+' A'
;					strtrim(string(sxm_par.msec_dwell, format='(i6)'), 2)+' msec, '+$
;					strtrim(string(n_cols),2)+'x'+strtrim(string(n_rows),2)
		END
		3 : BEGIN	; sm data
		END
		4 : BEGIN	; Beamline 7 (ALS) data
		END
		5 : BEGIN	; Polymer STXM data
		END
	ENDCASE

	IF (i EQ 0) THEN BEGIN
		n_cols_expected = n_cols
		n_rows_expected = n_rows
		image = fltarr(n_cols,n_rows)
		image = khz
		image_stack = image
		ev_list = [this_ev]
		msec_list = [this_msec]
		filename_display_list = [this_file_text_label]
	ENDIF ELSE BEGIN
		IF (n_cols NE n_cols_expected) OR $
				(n_rows NE n_rows_expected) THEN BEGIN
			print,'Skipping "'+filename_list(i)+'" because it has '+$
					strtrim(string(n_cols),2)+'x'+$
					strtrim(string(n_rows),2)+' rather than '+$
					strtrim(string(n_cols_expected),2)+'x'+$
					strtrim(string(n_rows_expected),2)+' pixels'
		ENDIF ELSE BEGIN
			image = khz
			image_stack = [[[temporary(image_stack)]],[[image]]]
			ev_list = [temporary(ev_list), this_ev]
			msec_list = [temporary(msec_list), this_msec]
			filename_display_list = $
					[temporary(filename_display_list), this_file_text_label]
		ENDELSE
	ENDELSE

ENDFOR

; Elapsed time for reading in image data files
end_time = systime(1.)
total_time = end_time - start_time
minutes = fix(total_time/60)
seconds = total_time-60.*minutes
IF (minutes GT 0) THEN BEGIN
	print,'Elapsed time for reading in images: '+$
			strtrim(string(minutes),2)+' minutes '+$
			strtrim(string(seconds,format='(f10.1)'),2)+' seconds.
ENDIF ELSE BEGIN
	print,'Elapsed time for reading in images: '+$
			strtrim(string(seconds,format='(f10.1)'),2)+' seconds.
ENDELSE

	    	IF (max([n_cols,n_rows]) GT 150) THEN BEGIN
			image_zoom = 1./round(max([n_cols,n_rows])/150.)
			print,'readlist : image_zoom :',image_zoom
		ENDIF ;ELSE BEGIN
;;			inv_zoom = 1
;;		ENDELSE

		n_files = n_elements(filename_list)
		IF n_elements(axis_call) NE 0 THEN BEGIN	; make sure axis_call is defined
			IF ((axis_call EQ 1) AND (n_files NE 0)) THEN BEGIN
				zstack_spectra
			ENDIF
		ENDIF

return
END

