; Copyright(c) 1998-2008 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		AX_READ_ROI
;
;LAST CHANGED: ----------------------------------- 	29-Dec-08
;
; PURPOSE:
;	This function reads a roi from different formats and saves it as a 0/1 mask-image
;
; CATEGORY:
;	Image display.
;
; CALLING SEQUENCE:

; for functions:
;	Result = AX_READ_ROI(file-file, filter=filter, roi_type=roi_type, template=template)
;
; INPUTS:
;	All input parameters are passed as keywords.
;
; KEYWORDS:
;	FILE:  name of file ; if not supplied, program requests it
; 	FILTER:   extension
;	ROI_TYPE: type of region of interest (stack, peem2, lox)
;   TEMPLATE:   AXB image defining the x, y scales and pixel numbers (*.axb)
;
; OUTPUTS:
;	internal axis format structure is generated
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;
; RESTRICTIONS:
;	only reads the I_roi portion; format supports storage of both Io_roi and I_roi
;
; MODIFICATION HISTORY:
; (28-jan-06 aph) first version; STACK ROI read-in taken from stack_analyze
; (09-Feb-06 aph) extend to PEEM2 AOI files
; (12-jun-06 aph) extend to Lox ROI files
; (29-Dec-08 aph) add title to guide user when need ot get template image from disk
;-

function ax_read_roi, file=file, filter=filter, roi_type=roi_type, template=template
@axis_com
@bsif_com
@analcom
COMMON volume_data, image_stack

s = 0
;------- determine type of ROI (ALS peem2, CLS Lox, or aXis2000 (jacobsen) stack) --------
if not keyword_set(roi_type) then begin
	test = dialog_message(/question, 'Axis2000 stack ROI ? (No = PEEM (Lox or ALS)')
	if test EQ 'Yes' then roi_type='stack' else begin
		test = dialog_message(/question, 'Lox (*.roi)? (No = ALS-PEEM2 (*.aoi)')
		if test EQ 'Yes' then roi_type='lox' else roi_type = 'peem2'
	endelse
endif

; ---- set file name filter ------------
if not keyword_set(filter) then begin
	if roi_type EQ 'stack' then filter = '*.roi'
	if roi_type EQ 'peem2' then filter = '*.aoi'
	if roi_type EQ 'lox'   then filter = '*.png'
endif

; --------- get filename and confirm existence --------
if not keyword_set(file) then $
   roi_filename = pickfile2(/read,FILTER=filter, /LPATH, DEFPATH=defpath) else $
   roi_filename = file
t = findfile(roi_filename, count=tcount)
if tcount LE 0 then begin
	axis_log, 'cannot find file ' + roi_filename
	return, s
endif
t = ax_name(roi_filename)
fileshort = t(1)

; ---------------- get template image -----------------
if not keyword_set(template) then begin
	tfile=pickfile2(/read,FILTER='*.axb', TITLE='Select template image (same dimensions as ROI)', /LPATH, DEFPATH=defpath)
	template = axb_load(file=tfile)
endif
IF n_tags(template) EQ 0 THEN  begin
	axis_log, 'missing template file'
	return, s
ENDIF

; ---------------- get stack roi data -----------------
print, 'Type is ', roi_type

CASE roi_type OF
	'stack': BEGIN
		get_lun,lun
		openr,lun,roi_filename,/xdr
		n_i_roi = long(0)
		n_i0_roi = long(0)
		i_roi = 0 & i0_roi = 0
		readu,lun,n_i_roi,n_i0_roi
		IF (n_i_roi NE 0) THEN BEGIN
		    i_roi = lonarr(n_i_roi)
		    readu,lun,long(i_roi)
		ENDIF ELSE BEGIN
		    i_roi = 0
		ENDELSE
		IF (n_i0_roi NE 0) THEN BEGIN
		    i0_roi = lonarr(n_i0_roi)
		    readu,lun,long(i0_roi)
		ENDIF ELSE BEGIN
		    i0_roi = 0
		ENDELSE
		close,lun
		free_lun,lun
		template.d = 0
		template.d(i_roi) = 1
		template.dl = fileshort + ' roi'
		axis_log,'Read aXis2000 ROI from ' + fileshort
		return, template
	END

	'lox': BEGIN
		n_cols = n_elements(template.x)
		n_rows = n_elements(template.y)
		roi_img = make_array(n_cols,n_rows,/integer,value = 0)
   		img_ROI_PNG = read_png(roi_filename)
		for i = 0, n_cols-1 do begin
		   for j = 0, n_rows-1 do begin
				if img_ROI_PNG(2,i,j) EQ 1 then begin		;if blue component of pixel is equal to 1 then
					roi_img(i, j) = 1						;point on the image is also equal to 1
				endif
			endfor
		endfor
		i_roi = where(roi_img EQ 1)		;JSO transfer image data into i_roi pixel list
		template.d = roi_img
		template.dl = fileshort + ' roi'
		axis_log,'Read Lox ROI from ' + fileshort
		return, template
	END

	'peem2': BEGIN
; -------- read in definition of AOI - assumes RECTANGULAR with one line only
		text = ' '
		get_lun,lun
		openr,lun,roi_filename
		readf, lun, text
		close,lun
		free_lun,lun
;		print, text
		t=0
		while size(t,/type) EQ 2 do begin
;			print, text
			t = parsestring(text)
			text=strmid(text, strpos(text,' '))
		endwhile
		ts=n_elements(t)-4
		aoi_xmin=t(ts)
		aoi_xmax=t(ts+1)
		aoi_ymin=t(ts+2)
		aoi_ymax=t(ts+3)
		print, 'aoi values', aoi_xmin, aoi_xmax,  aoi_ymin, aoi_ymax
		axis_log,'Read Peem2 AOI from ' + fileshort
;	--------- mark template image with this structure

		return, t
	END

	ELSE: BEGIN
		print, 'Not a recognized type of ROI'
		return, s
	ENDELSE
ENDCASE


end
