; Copyright (c) 1998-2008 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		AX_RGB_MAP
;
;LAST CHANGED: ----------------------------------- 	29-Dec-07
;
; PURPOSE:
;	This set of procedures is a widget which generates R_G-B composite maps
;  It allows mage selection and ; adjustment of min/max scales for each color.
; A color circle and separate R, G, B color scales can be optionally selected
;
; CATEGORY:
;	aXis2000 capitve. Image display.
;
; CALLING SEQUENCE:
;	AX_RGB_MAP
;
; INPUTS: default set of buffers are R=4, G=5, B=6
;
; KEYWORDS: none
;
; OUTPUTS:
; (optional)  a TIF image of the RGB composite map can be stored
;
; AX_RGB_MAP_EVENT ROUTINES:
;
; NUM_FORMAT		standard number format for limits
; MAKERGB			generate composite image
; LIST_RGB_RED		select buffer for RED
; LIST_RGB_GREEN	select buffer for GREEN
; LIST_RGB_BLUE		select buffer for BLUE
; BUT_RGB_CIRCLE	on/off color circle display
; BUT_RGB_SCALES	on/off R, G, B scales
; SLIDER_R			adjust red min / max value
; SLIDER_G			adjust green max value
; SLIDER_B			adjust blue max value
; TEXT_RGB_RMAX		adjust red max value
; TEXT_RGB_GMAX		adjust green max value
; TEXT_RGB_BMAX		adjust blue max value
; TEXT_RGB_RMIN		adjust red min value
; TEXT_RGB_GMIN		adjust green min value
; TEXT_RGB_BMIN		adjust blue min value
; ABSOLUTE_SCALE	set all colors to global limits
; RGB_RESCALE		set each color to limits of its data
; BUT_RGB_CANCEL	cancel
; BUT_RGB_SAVE		save composite map
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;	AX_RGB_COM  parameters specifc to ax_rgb_map
;
; SIDE EFFECTS:
;	A window is created/destroyed.
;
; RESTRICTIONS:
;	Any known ?
;
; MODIFICATION HISTORY:
; (16-Jun-03 aph) first verasion started
; (10-Mar-06 aph) widget structure generated from AX_RGB_WIDGET.PRC
; (29-Dec-07 aph) coded
;-

FUNCTION num_format, value
num_string = strtrim(string(value,format = '(g9.2)'))
return, num_string
END

PRO MakeRGB
@axis_com
@ax_rgb_com
on_error,2

;  ************ Generate an RBG composite from 3 images (do not use buffer 0) *******
; ------- select all data from images within slider ranges
R_exist = 0 & G_exist = 0 & B_exist = 0
Rimg_size=[0,0] &  Gimg_size=[0,0] &  Bimg_size=[0,0]
if red_index GT 0 AND red_index LE 9 then begin
	HANDLE_VALUE, Data(red_index), tmpR
	IF n_tags(tmpR) NE 0 THEN BEGIN
		IF tmpR.t EQ '2d' THEN BEGIN
			ri=tmpR.d
			tmp_index = where(ri LT r_min, count)
			if count GT 0 then ri(tmp_index)=r_min
			tmp_index = where(ri GT r_max, count)
			if count GT 0 then ri(tmp_index)=r_max
			Rimg_size=[n_elements(tmpR.x), n_elements(tmpR.y)]
			R_exist = 1
		ENDIF
	ENDIF
endif
if green_index GT 0 AND green_index LE 9 then begin
	HANDLE_VALUE, Data(green_index), tmpG
	IF n_tags(tmpG) NE 0 THEN BEGIN
		IF tmpG.t EQ '2d' THEN BEGIN
			gi=tmpG.d
			tmp_index = where(gi LT g_min, count)
			if count GT 0 then gi(tmp_index)=g_min
			tmp_index = where(gi GT g_max, count)
			if count GT 0 then gi(tmp_index)=g_max
			Gimg_size=[n_elements(tmpG.x), n_elements(tmpG.y)]
			G_exist = 1
		ENDIF
	ENDIF
endif
if blue_index GT 0 AND blue_index LE 9 then begin
	HANDLE_VALUE, Data(blue_index), tmpB
	IF n_tags(tmpB) NE 0 THEN BEGIN
		IF tmpB.t EQ '2d' THEN BEGIN
			bi=tmpB.d
			tmp_index = where(bi LT b_min, count)
			if count GT 0 then bi(tmp_index)=b_min
			tmp_index = where(bi GT b_max, count)
			if count GT 0 then bi(tmp_index)=b_max
			Bimg_size=[n_elements(tmpB.x), n_elements(tmpB.y)]
			B_exist = 1
		ENDIF
	ENDIF
endif

 IF R_exist OR  G_exist OR B_exist THEN BEGIN
	x_size = max([Rimg_size(0),Gimg_size(0),Bimg_size(0)])
	y_size = max([Rimg_size(1),Gimg_size(1),Bimg_size(1)])
	img_size=[x_size, y_size]
	image = bytarr(3,img_size(0), img_size(1))
; ---- deal with cases where R, G, B might be missing ---------
	if n_elements(ri) EQ 0 then begin
		ri=fltarr(img_size)
		r_min= 0. & r_max = 0.
	endif
	if n_elements(gi) EQ 0 then begin
		gi=fltarr(img_size)
		g_min= 0. & g_max = 0.
	endif
	if n_elements(bi) EQ 0 then begin
		bi=fltarr(img_size)
		b_min= 0. & b_max = 0.
	endif

	if r_max NE r_min then begin
		rslope = 255./(r_max-r_min)
		image(0,*,*) = byte(rslope*(ri-r_min))
	endif else image(0,*,*) = 0.
	if g_max NE g_min then begin
		gslope = 255./(g_max-g_min)
		image(1,*,*) = byte(gslope*(gi-g_min))
	endif else image(1,*,*) = 0.
	if b_max NE b_min then begin
		bslope = 255./(b_max-b_min)
		image(2,*,*) = byte(bslope*(bi-b_min))
	endif else image(2,*,*) = 0.

	wset, RGB_display
; ------ set up color scale (use default if it is not already set)
	if n_elements(ax_top_color_index) EQ 0 then ax_top_color_index = 256
	r = indgen(ax_top_color_index) & g = indgen(ax_top_color_index) & b = indgen(ax_top_color_index) ; make linear color tables for R G B
 	tvlct, r,g,b
; ------- generate RGB image
	main = 360*gr_scale
 	t_scl = main/max([img_size(0), img_size(1)])
 	nx = fix(t_scl*img_size(0))
 	ny = fix(t_scl*img_size(1))
 	t_image = intarr(3, nx, ny)
 	for i=0,2 do $
 		t_image(i,*,*) = congrid(reform(image(i,*,*)),nx, ny, /interp)
 	tv, t_image, /true

 ENDIF
END

;-----------------------------------------------------------------
PRO Red_histogram_update ;, new=new
@axis_com
@ax_rgb_com
on_error,2

wset, Red_hist_ID
if n_elements(r_hist) GT 1 then begin
;	if keyword_Set(new) then $
	        plot, r_hist(0,*),r_hist(1,*), xstyle=5, ystyle=5, xmargin=[0,0],ymargin=[0,0]
    y_min = min(r_hist(1,*), max=y_max)
	oplot, [r_min, r_min], [y_min, y_max]
	oplot, [r_max, r_max], [y_min, y_max]
endif else erase
END

;-----------------------------------------------------------------
PRO Green_histogram_update  ;, new=new
@axis_com
@ax_rgb_com
on_error,2
if n_elements(g_hist) GT 1 then begin
	wset, Green_hist_ID
;	if keyword_Set(new) then $
	  plot, g_hist(0,*),g_hist(1,*), xstyle=1, ystyle=5, xmargin=[0,0],ymargin=[0,0]
	y_min = min(g_hist(1,*), max=y_max)
	oplot, [g_min, g_min], [y_min, y_max]
	oplot, [g_max, g_max], [y_min, y_max]
endif
END

;-----------------------------------------------------------------
PRO Blue_histogram_update ;, new=new
@axis_com
@ax_rgb_com
on_error,2
if n_elements(b_hist) GT 1 then begin
	wset, Blue_hist_ID
;	if keyword_Set(new) then $
	  plot, b_hist(0,*),b_hist(1,*), xstyle=1, ystyle=5, xmargin=[0,0],ymargin=[0,0]
	y_min = min(b_hist(1,*), max=y_max)
	oplot, [b_min, b_min], [y_min, y_max]
	oplot, [b_max, b_max], [y_min, y_max]
endif
END

;-----------------------------------------------------------------
pro TEXT_RGB_RMAX, Event
@axis_com
@ax_rgb_com
on_error,2
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Rmax')
WIDGET_CONTROL,id, Get_value = r_max
r_max = float(r_max(0))
WIDGET_CONTROL,id, Set_value = num_format(r_max)
Rvalue=[double(r_min), double(r_max)]
id = widget_info(Event.top, FIND_BY_UNAME = 'SLIDER_R')
CW_RSLIDER_SET_VALUE, id, Rvalue
rescale_flag = 0
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_rescale')
WIDGET_CONTROL,id, Set_button = rescale_flag
;print, 'Rmax changed to ', r_max
Red_histogram_update
makeRGB
end

;-----------------------------------------------------------------
pro TEXT_RGB_Gmax, Event
@axis_com
@ax_rgb_com
on_error,2
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Gmax')
WIDGET_CONTROL,id, Get_value = g_max
g_max = float(g_max(0))
WIDGET_CONTROL,id, Set_value = num_format(g_max)
Gvalue=[double(g_min), double(g_max)]
id = widget_info(Event.top, FIND_BY_UNAME = 'SLIDER_G')
CW_RSLIDER_SET_VALUE, id, Gvalue
rescale_flag = 0
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_rescale')
WIDGET_CONTROL,id, Set_button = rescale_flag
Green_histogram_update
makeRGB
end

;-----------------------------------------------------------------
pro TEXT_RGB_Bmax, Event
@axis_com
@ax_rgb_com
on_error,2
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Bmax')
WIDGET_CONTROL,id, Get_value = b_max
b_max = float(b_max)
WIDGET_CONTROL,id, Set_value = num_format(b_max)
Bvalue=[double(b_min), double(b_max)]
id = widget_info(Event.top, FIND_BY_UNAME = 'SLIDER_B')
CW_RSLIDER_SET_VALUE, id, Bvalue
rescale_flag = 0
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_rescale')
WIDGET_CONTROL,id, Set_button = rescale_flag
Blue_histogram_update
makeRGB
end
;-----------------------------------------------------------------
pro TEXT_RGB_Rmin, Event
@axis_com
@ax_rgb_com
on_error,2
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Rmin')
WIDGET_CONTROL,id, Get_value = r_min
r_min = float(r_min)
WIDGET_CONTROL,id, Set_value = num_format(r_min)
Rvalue=[double(r_min), double(r_max)]
id = widget_info(Event.top, FIND_BY_UNAME = 'SLIDER_R')
CW_RSLIDER_SET_VALUE, id, Rvalue
rescale_flag = 0
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_rescale')
WIDGET_CONTROL,id, Set_button = rescale_flag
Red_histogram_update
makeRGB
end

;-----------------------------------------------------------------
pro TEXT_RGB_Gmin, Event
@axis_com
@ax_rgb_com
on_error,2
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Gmin')
WIDGET_CONTROL,id, Get_value = g_min
g_min = float(g_min)
WIDGET_CONTROL,id, Set_value = num_format(g_min)
Gvalue=[double(g_min), double(g_max)]
id = widget_info(Event.top, FIND_BY_UNAME = 'SLIDER_G')
CW_RSLIDER_SET_VALUE, id, Gvalue
rescale_flag = 0
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_rescale')
WIDGET_CONTROL,id, Set_button = rescale_flag
Green_histogram_update
makeRGB
end

;-----------------------------------------------------------------
pro TEXT_RGB_Bmin, Event
@axis_com
@ax_rgb_com
on_error,2
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Bmin')
WIDGET_CONTROL,id, Get_value = b_min
b_min = float(b_min)
WIDGET_CONTROL,id, Set_value = num_format(b_min)
Bvalue=[double(b_min), double(b_max)]
id = widget_info(Event.top, FIND_BY_UNAME = 'SLIDER_B')
CW_RSLIDER_SET_VALUE, id, Bvalue
rescale_flag = 0
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_rescale')
WIDGET_CONTROL,id, Set_button = rescale_flag
Blue_histogram_update
makeRGB
end

; ***********************SLIDER EVENTS *************
;   -----------------------------------------------
pro SLIDER_Red, Event
@axis_com
@ax_rgb_com
on_error,2

if n_elements(r_hist) LE 1 then return	; in case user changes slider while there is no image
r_min =  event.values(0)
r_max =  event.values(1)
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Rmin')
WIDGET_CONTROL,id, Set_value = num_format(r_min)
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Rmax')
WIDGET_CONTROL,id, Set_value = num_format(r_max)
rescale_flag = 0
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_rescale')
WIDGET_CONTROL,id, Set_button = rescale_flag
Red_histogram_update
makeRGB
end

;   -----------------------------------------------
pro SLIDER_Green, Event
@axis_com
@ax_rgb_com
on_error,2

if n_elements(g_hist) LE 1 then return	; in case user changes slider while there is no image
g_min =  event.values(0)
g_max =  event.values(1)
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Gmin')
WIDGET_CONTROL,id, Set_value = num_format(g_min)
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Gmax')
WIDGET_CONTROL,id, Set_value = num_format(g_max)
rescale_flag = 0
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_rescale')
WIDGET_CONTROL,id, Set_button = rescale_flag
Green_histogram_update
makeRGB
end

;   -----------------------------------------------
pro SLIDER_Blue, Event
@axis_com
@ax_rgb_com
on_error,2

if n_elements(b_hist) LE 1 then return	; in case user changes slider while there is no image
b_min =  event.values(0)
b_max =  event.values(1)
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Bmin')
WIDGET_CONTROL,id, Set_value = num_format(b_min)
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Bmax')
WIDGET_CONTROL,id, Set_value = num_format(b_max)
rescale_flag = 0
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_rescale')
WIDGET_CONTROL,id, Set_button = rescale_flag
Blue_histogram_update
makeRGB
end

; ------------------ Auto rescale ------------------
pro RGB_Rescale, Event
@axis_com
@ax_rgb_com
on_error,2
if rescale_flag EQ 1 then rescale_flag = 0 else  rescale_flag = 1
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_rescale')
WIDGET_CONTROL,id, Set_button = rescale_flag
absolute_flag = 0
id = widget_info(Event.top, FIND_BY_UNAME = 'Absolute scale')
WIDGET_CONTROL,id, Set_button = absolute_flag
if rescale_flag EQ 1 then begin
	r_min = Ar_min & r_max = Ar_max
	g_min = Ag_min & g_max = Ag_max
	b_min = Ab_min & b_max = Ab_max
	id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Rmin')
	widget_control, id, set_value=num_format(r_min), sens = r_sens
	id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Rmax')
	widget_control, id, set_value=num_format(r_max), sens = r_sens
	if r_min NE r_max then begin
		Rvalue=[double(r_min), double(r_max)]
		id = widget_info(Event.top, FIND_BY_UNAME = 'SLIDER_R')
		CW_RSLIDER_SET_VALUE, id, Rvalue
	endif
	id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Gmin')
	widget_control, id, set_value=num_format(g_min), sens = g_sens
	id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Gmax')
	widget_control, id, set_value=num_format(g_max), sens = g_sens
	if g_min NE g_max then begin
		Gvalue=[double(g_min), double(g_max)]
		id = widget_info(Event.top, FIND_BY_UNAME = 'SLIDER_G')
		CW_RSLIDER_SET_VALUE, id, Gvalue
	endif
	id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Bmin')
	widget_control, id, set_value=num_format(b_min), sens = b_sens
	id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Bmax')
	widget_control, id, set_value=num_format(b_max), sens = b_sens
	if b_min NE b_max then begin
		Bvalue=[double(b_min), double(b_max)]
		id = widget_info(Event.top, FIND_BY_UNAME = 'SLIDER_B')
		CW_RSLIDER_SET_VALUE, id, Bvalue
	endif
	Red_histogram_update
	Green_histogram_update
	Blue_histogram_update
	makeRGB
endif
end

; ------------------ Auto rescale ------------------
pro Absolute_scale, Event
@axis_com
@ax_rgb_com
on_error,2

if absolute_flag EQ 1 then absolute_flag = 0 else  absolute_flag = 1
id = widget_info(Event.top, FIND_BY_UNAME = 'Absolute scale')
WIDGET_CONTROL,id, Set_button = absolute_flag
rescale_flag = 0
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_rescale')
WIDGET_CONTROL,id, Set_button = rescale_flag

if absolute_flag EQ 1 then begin
	r_min = min([Ar_min, Ag_min, Ab_min])
	r_max = max([Ar_max, Ag_max, Ab_max])
	g_min = r_min & g_max = r_max
	b_min = r_min & b_max = r_max
	id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Rmin')
	widget_control, id, set_value=num_format(r_min)
	id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Rmax')
	widget_control, id, set_value=num_format(r_max)
	Rvalue=[double(r_min), double(r_max)]
	id = widget_info(Event.top, FIND_BY_UNAME = 'SLIDER_R')
	CW_RSLIDER_SET_MINMAX, id, Rvalue
	CW_RSLIDER_SET_VALUE, id, Rvalue
	id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Gmin')
	widget_control, id, set_value=num_format(g_min)
	id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Gmax')
	widget_control, id, set_value=num_format(g_max)
	Gvalue=[double(g_min), double(g_max)]
	id = widget_info(Event.top, FIND_BY_UNAME = 'SLIDER_G')
	CW_RSLIDER_SET_MINMAX, id, Gvalue
	CW_RSLIDER_SET_VALUE, id, Gvalue
	id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Bmin')
	widget_control, id, set_value=num_format(b_min)
	id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Bmax')
	widget_control, id, set_value=num_format(b_max)
	Bvalue=[double(b_min), double(b_max)]
	id = widget_info(Event.top, FIND_BY_UNAME = 'SLIDER_B')
	CW_RSLIDER_SET_MINMAX, id, Bvalue
	CW_RSLIDER_SET_VALUE, id, Bvalue
	Red_histogram_update
	Green_histogram_update
	Blue_histogram_update
	makeRGB
endif
end

; ---------------------------------------------------
pro LIST_RGB_RED, Event ;, index=index
@axis_com
@ax_rgb_com
on_error,2
;if not keyword_set(index) then begin
	id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_DROPLIST_red')
	red_index = WIDGET_INFO(id, /Droplist_select)
;endif
HANDLE_VALUE, Data(red_index), tmp
IF n_tags(tmp) NE 0 THEN BEGIN
	if tmp.t EQ '2d' then begin
		rgb_label(red_index) = label(red_index)
		r_min = min(tmp.d, max=r_max)
	endif else begin
		rgb_label(red_index)= 'not an image'
		r_min = 0 & r_max = 0
	endelse
ENDIF ELSE BEGIN
	if red_index NE 0 then rgb_label(red_index)= 'empty' else rgb_label(red_index)= 'none'
	r_min = 0. & r_max = 0
ENDELSE
Ar_min = r_min & Ar_max = r_max
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_DROPLIST_red')
widget_control, id, set_value = rgb_label, SET_DROPLIST_SELECT = red_index
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Index_Red')
widget_control, id, set_value = strtrim(string(red_index),2)
; ---- if there is an image reset min/max and values of the color adjuster
if r_min NE r_max then begin
	r_sens = 1
	r_hist = ax_histogram(tmp.d, n_bins=hist_bins)
	RS_min = double(r_min - s_margin*(r_max-r_min))
	RS_max = double(r_max + s_margin*(r_max-r_min))
	Rvalue = [double(r_min), double(r_max)]
	id = widget_info(Event.top, FIND_BY_UNAME = 'SLIDER_R')
	CW_RSLIDER_SET_MINMAX, id, [RS_min, RS_max]
	CW_RSLIDER_SET_VALUE, id, Rvalue
endif else begin
	r_hist = 0
	r_sens = 0
endelse
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Rmin')
WIDGET_CONTROL,id, Set_value = num_format(r_min), sensitive = R_sens
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Rmax')
WIDGET_CONTROL,id,  Set_value = num_format(r_max), sensitive = R_sens
Red_histogram_update
makeRGB
end

pro LIST_RGB_GREEN, Event
@axis_com
@ax_rgb_com
on_error,2
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_DROPLIST_green')
green_index = WIDGET_INFO(id, /Droplist_select)
HANDLE_VALUE, Data(green_index), tmp
IF n_tags(tmp) NE 0 THEN BEGIN
	if tmp.t EQ '2d' then begin
		rgb_label(green_index) = label(green_index)
		g_min = min(tmp.d, max=g_max)
	endif else begin
		rgb_label(green_index)= 'not an image'
		g_min = 0 & g_max = 0
	endelse
ENDIF ELSE BEGIN
	if green_index NE 0 then rgb_label(green_index)= 'empty' else rgb_label(green_index)= 'none'
	g_min = 0 & g_max = 0
ENDELSE
Ag_min = g_min & Ag_max = g_max
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_DROPLIST_green')
widget_control, id, set_value = rgb_label, SET_DROPLIST_SELECT = green_index
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Index_Green')
widget_control, id, set_value = strtrim(string(green_index),2)
; ---- if there is an image reset min/max and values of the color adjuster
IF g_min NE g_max then begin
	g_sens = 1
	g_hist = ax_histogram(tmp.d, n_bins=hist_bins)
	GS_min = double(g_min - s_margin*(g_max-g_min))
	GS_max = double(g_max + s_margin*(g_max-g_min))
	Gvalue = [double(g_min), double(g_max)]
	id = widget_info(Event.top, FIND_BY_UNAME = 'SLIDER_G')
	CW_RSLIDER_SET_MINMAX, id, Gvalue
ENDIF else begin
	g_hist = 0
	g_sens = 0
ENDELSE
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Gmin')
WIDGET_CONTROL,id, Set_value = num_format(g_min), sensitive = g_sens
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Gmax')
WIDGET_CONTROL,id, Set_value =  num_format(g_max), sensitive = g_sens
Green_histogram_update
makeRGB
end

pro LIST_RGB_BLUE, Event
@axis_com
@ax_rgb_com
on_error,2
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_DROPLIST_blue')
blue_index = WIDGET_INFO(id, /Droplist_select)
blue_label = rgb_label(blue_index)
HANDLE_VALUE, Data(blue_index), tmp
IF n_tags(tmp) NE 0 THEN BEGIN
	if tmp.t EQ '2d' then begin
		rgb_label(blue_index) = label(blue_index)
		b_min = min(tmp.d, max=b_max)
	endif else begin
		rgb_label(blue_index)= 'not an image'
		b_min = 0 & b_max = 0
	endelse
ENDIF ELSE BEGIN
	if blue_index NE 0 then rgb_label(blue_index)= 'empty' else rgb_label(blue_index)= 'none'
	b_min = 0 & b_max = 0
ENDELSE
Ab_min = b_min & Ab_max = b_max
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_DROPLIST_blue')
widget_control, id, set_value = rgb_label, SET_DROPLIST_SELECT = blue_index
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Index_Blue')
widget_control, id, set_value = strtrim(string(blue_index),2)
; ---- if there is an image reset min/max and values of the color adjuster
IF b_min NE b_max then begin
	b_sens = 1
	b_hist = ax_histogram(tmp.d, n_bins=hist_bins)
	BS_min = double(b_min - s_margin*(b_max-b_min))
	BS_max = double(b_max + s_margin*(b_max-b_min))
	Bvalue = [double(b_min), double(b_max)]
	id = widget_info(Event.top, FIND_BY_UNAME = 'SLIDER_B')
	CW_RSLIDER_SET_MINMAX, id, Bvalue
ENDIF else begin
	b_hist = 0
	b_sens = 0
ENDELSE
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Bmin')
WIDGET_CONTROL,id, Set_value =num_format(b_min), sensitive = b_sens
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Bmax')
WIDGET_CONTROL,id, Set_value = num_format(b_max), sensitive = b_sens
Blue_histogram_update
makeRGB
end

;-----------------------------------------------------------------
pro TEXT_RGB_Index_Red, Event
@axis_com
@ax_rgb_com
on_error,2
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Index_Red')
widget_control, id, get_value = red_index
; print, 'changed red to buffer ', red_index
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_DROPLIST_red')
widget_control, id, SET_DROPLIST_SELECT = red_index
LIST_RGB_Red, event

END

;-----------------------------------------------------------------
pro TEXT_RGB_Index_Green, Event
@axis_com
@ax_rgb_com
on_error,2
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Index_Green')
widget_control, id, get_value = green_index
; print, 'changed green to buffer ', green_index
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_DROPLIST_green')
widget_control, id, SET_DROPLIST_SELECT = green_index
LIST_RGB_Green, event
end

;-----------------------------------------------------------------
pro TEXT_RGB_Index_Blue, Event
@axis_com
@ax_rgb_com
on_error,2
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_TEXT_Index_Blue')
widget_control, id, get_value = blue_index
; print, 'changed blue to buffer ', blue_index
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_DROPLIST_blue')
widget_control, id, SET_DROPLIST_SELECT = blue_index
LIST_RGB_Blue, event
end

;-----------------------------------------------------------------
pro BUT_RGB_SCALES, Event
@axis_com
@ax_rgb_com
on_error,2
print, 'scales_flag ', scales_flag
if scales_flag EQ 1 then scales_flag = 0 else scales_flag = 1
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_BUTTON_scales')
WIDGET_CONTROL,id, Set_button = scales_flag
print, ' Add scales routine not written'
end

;-----------------------------------------------------------------
pro BUT_RGB_CIRCLE, Event
@axis_com
@ax_rgb_com
on_error,2
print, 'circle_flag ', circle_flag
if circle_flag EQ 1 then circle_flag = 0 else circle_flag = 1
id = widget_info(Event.top, FIND_BY_UNAME = 'RGB_BUTTON_circle')
WIDGET_CONTROL,id, Set_button = circle_flag
print, ' Color circle routine not written'
end

;-----------------------------------------------------------------
pro BUT_RGB_SAVE, Event
@axis_com
@ax_rgb_com
on_error,2

file = pickfile2(/write, FILTER='*.tif',/lpath, DefPath=DefPath)
if file NE '' then begin
	wset, RGB_display
	Win2Tif, file
	t=ax_name(file)
	axis_log, 'RBG composite stored as ' + t(1)
endif
end

;-----------------------------------------------------------------
pro  WIDGET_KILL_REQUEST, Event
@axis_com
@ax_rgb_com
on_error,2
WIDGET_CONTROL, Event.Top, /DESTROY
end

;-----------------------------------------------------------------
pro BUT_RGB_CANCEL, Event
@axis_com
@ax_rgb_com
on_error,2
WIDGET_CONTROL, Event.Top, /DESTROY
end

; **************************************************************************
; ---------------------------------  PROCESS CLICKS etc -------------------
pro RBG_BASE_event, Event
@axis_com
@ax_rgb_com
on_error,2

wWidget =  Event.top

  case Event.id of

; --------- kill ax_rgb_widget if use X in upper right corner --------------
    Widget_Info(wWidget, FIND_BY_UNAME='RBG_BASE'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_KILL_REQUEST' )then $
       WIDGET_KILL_REQUEST, Event
    end

 ; ------------ select buffer  --------------
    Widget_Info(wWidget, FIND_BY_UNAME='RGB_DROPLIST_red'): begin
    if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_DROPLIST' )then $
        LIST_RGB_RED, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='RGB_DROPLIST_green'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_DROPLIST' )then $
        LIST_RGB_GREEN, Event
    end
     Widget_Info(wWidget, FIND_BY_UNAME='RGB_DROPLIST_blue'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_DROPLIST' )then $
        LIST_RGB_BLUE, Event
    end

; -------------- adjust maxima and minima with compound slider
    Widget_Info(wWidget, FIND_BY_UNAME='SLIDER_R'): SLIDER_Red, Event
    Widget_Info(wWidget, FIND_BY_UNAME='SLIDER_G'): SLIDER_Green, Event
    Widget_Info(wWidget, FIND_BY_UNAME='SLIDER_B'): SLIDER_Blue, Event


; ------ adjust minima & maxima numerically
    Widget_Info(wWidget, FIND_BY_UNAME='RGB_TEXT_Rmax'): begin
;    	print, tag_names(Event, /structure_name)
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_CH' )then $
        TEXT_RGB_RMAX, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='RGB_TEXT_Gmax'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_CH' )then $
        TEXT_RGB_GMAX, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='RGB_TEXT_Bmax'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_CH' )then $
        TEXT_RGB_BMAX, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='RGB_TEXT_Rmin'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_CH' )then $
        TEXT_RGB_RMIN, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='RGB_TEXT_Gmin'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_CH' )then $
        TEXT_RGB_GMIN, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='RGB_TEXT_Bmin'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_CH' )then $
        TEXT_RGB_BMIN, Event
    end

    Widget_Info(wWidget, FIND_BY_UNAME='RGB_rescale'): RGB_Rescale, Event
    Widget_Info(wWidget, FIND_BY_UNAME='Absolute scale'): Absolute_scale, Event

    Widget_Info(wWidget, FIND_BY_UNAME='RGB_TEXT_Index_Red'):   TEXT_RGB_Index_Red, Event
    Widget_Info(wWidget, FIND_BY_UNAME='RGB_TEXT_Index_Green'): TEXT_RGB_Index_Green, Event
    Widget_Info(wWidget, FIND_BY_UNAME='RGB_TEXT_Index_Blue'):  TEXT_RGB_Index_Blue, Event

    Widget_Info(wWidget, FIND_BY_UNAME='RGB_BUTTON_circle'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_RGB_CIRCLE, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='RGB_BUTTON_scales'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_RGB_SCALES, Event
    end

    Widget_Info(wWidget, FIND_BY_UNAME='RGB_BUTTON_cancel'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_RGB_CANCEL, Event
    end

    Widget_Info(wWidget, FIND_BY_UNAME='RGB_BUTTON_save'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_RGB_SAVE, Event
    end
    else: print, 'Undefined action'
  endcase

end

; --------------------------------------------------------------------------------------
; **************************************************************************************
; --------------------------------------------------------------------------------------


pro ax_rgb_map, GROUP_LEADER=wGroup, _EXTRA=_VWBExtra_
@axis_com
@ax_rgb_com
on_error,2

tx = size(wGroup)
IF tx(1) EQ 0 THEN modal_val=0 else modal_val=1

  RBG_BASE = Widget_Base( GROUP_LEADER=wGroup, UNAME='RBG_BASE',  $
      modal = modal_val, TITLE='RGB composite map', $
     /TLB_KILL_REQUEST_EVENTS, SPACE=3 ,XPAD=3 ,YPAD=3)

; ------------------ GUI parameterization
number_size = 9
hist_offset = 78
hist_Vsize = 100
hist_Hsize = 200
T_size = 78
Tmin_x_offset =  0
Tmax_x_offset =  hist_Hsize  + hist_offset
color_size  = hist_Vsize + 30
R_y_offset = hist_Vsize
G_y_offset = R_y_offset + color_size
B_y_offset = G_y_offset + color_size
main = fix(360*gr_scale)
main_offset = 360
  RGB_image = Widget_Draw(RBG_BASE, UNAME='RGB_image' ,XOFFSET=main_offset  $
      ,SCR_XSIZE=main ,SCR_YSIZE=main)

  RGB_TEXT_Rmax = Widget_Text(RBG_BASE, UNAME='RGB_TEXT_Rmax'  $
      ,XOFFSET=Tmax_x_offset, YOFFSET=R_y_offset ,/EDITABLE, XSIZE=number_size ,YSIZE=1)
  RGB_TEXT_Gmax = Widget_Text(RBG_BASE, UNAME='RGB_TEXT_Gmax'  $
      ,XOFFSET=Tmax_x_offset, YOFFSET=G_y_offset ,/EDITABLE,XSIZE=number_size ,YSIZE=1)
  RGB_TEXT_Bmax = Widget_Text(RBG_BASE, UNAME='RGB_TEXT_Bmax'  $
      ,XOFFSET=Tmax_x_offset, YOFFSET=B_y_offset ,/EDITABLE, XSIZE=number_size ,YSIZE=1)

  RGB_TEXT_Rmin = Widget_Text(RBG_BASE, UNAME='RGB_TEXT_Rmin'  $
      ,XOFFSET=Tmin_x_offset ,YOFFSET=R_y_offset, /EDITABLE,XSIZE=number_size ,YSIZE=1)
  RGB_TEXT_Gmin = Widget_Text(RBG_BASE, UNAME='RGB_TEXT_Gmin'  $
      ,XOFFSET=Tmin_x_offset ,YOFFSET=G_y_offset, /EDITABLE ,XSIZE=number_size ,YSIZE=1)
  RGB_TEXT_Bmin = Widget_Text(RBG_BASE, UNAME='RGB_TEXT_Bmin'  $
      ,XOFFSET=Tmin_x_offset ,YOFFSET=B_y_offset,/EDITABLE ,XSIZE=number_size ,YSIZE=1)

  yoffset_red = main + 7
  yoffset_green = main + 33
  yoffset_blue = main + 60
  xoffset_lists = main_offset
  label_size = 60
  Index_size = label_size + 30
  tlabel = widget_label(RBG_BASE, XOFFSET=xoffset_lists ,YOFFSET= yoffset_red,value='Red')
  tlabel = widget_label(RBG_BASE, XOFFSET=xoffset_lists ,YOFFSET=yoffset_green,value='Green')
  tlabel = widget_label(RBG_BASE, XOFFSET=xoffset_lists ,YOFFSET=yoffset_blue,value='Blue')
  RGB_TEXT_INDEX_Red = Widget_Text(RBG_BASE, UNAME='RGB_TEXT_Index_Red', sensitive = 1  $
      ,XOFFSET= xoffset_lists + label_size,YOFFSET=yoffset_red, /EDITABLE,XSIZE=1 ,YSIZE=1)
  RGB_TEXT_INDEX_Green = Widget_Text(RBG_BASE, UNAME='RGB_TEXT_Index_Green', sensitive = 1  $
      ,XOFFSET= xoffset_lists + label_size,YOFFSET=yoffset_green, /EDITABLE,XSIZE=1 ,YSIZE=1)
  RGB_TEXT_INDEX_Blue = Widget_Text(RBG_BASE, UNAME='RGB_TEXT_Index_Blue', sensitive = 1  $
      ,XOFFSET= xoffset_lists + label_size,YOFFSET=yoffset_blue, /EDITABLE,XSIZE=1 ,YSIZE=1)

  RGB_DROPLIST_red = Widget_Droplist(RBG_BASE,  $
      UNAME='RGB_DROPLIST_red' ,XOFFSET= xoffset_lists + index_size ,YOFFSET = yoffset_red  $
      ,SCR_XSIZE=240  ,SCR_YSIZE=26)
  RGB_DROPLIST_green = Widget_Droplist(RBG_BASE,  $
      UNAME='RGB_DROPLIST_green' ,XOFFSET=xoffset_lists + index_size ,YOFFSET =  yoffset_green  $
      ,SCR_XSIZE=240 ,SCR_YSIZE=26)
  RGB_DROPLIST_blue = Widget_Droplist(RBG_BASE,  $
      UNAME='RGB_DROPLIST_blue' ,XOFFSET=xoffset_lists + index_size ,YOFFSET = yoffset_blue  $
      ,SCR_XSIZE=240,SCR_YSIZE=26)

; ------------------------- histogram plots
  Red_histogram = Widget_Draw(RBG_BASE, UNAME='Red_histogram' ,XOFFSET=hist_offset, YOFFSET=0  $
      ,SCR_XSIZE=hist_Hsize ,SCR_YSIZE=hist_Vsize)
  Green_histogram = Widget_Draw(RBG_BASE, UNAME='Green_histogram' ,XOFFSET=hist_offset, YOFFSET=color_size  $
      ,SCR_XSIZE=hist_Hsize ,SCR_YSIZE=hist_Vsize)
  Blue_histogram = Widget_Draw(RBG_BASE, UNAME='Blue_histogram' ,XOFFSET=hist_offset, YOFFSET=2*color_size  $
      ,SCR_XSIZE=hist_Hsize ,SCR_YSIZE=hist_Vsize)

; --------- control parameters
param_offset = 390
  Rescale_base = WIDGET_BASE(RBG_Base, UNAME='RESCALE_BASE', XOFFSET=10  $
      ,YOFFSET=param_offset ,SCR_XSIZE=120 ,SCR_YSIZE=26 ,COLUMN=1  ,/EXCLUSIVE)
  RGB_AUTO_SCALE = Widget_Button(Rescale_BASE,  $
      UNAME='RGB_rescale' ,XOFFSET=50 ,YOFFSET=14 ,/ALIGN_LEFT  $
      ,VALUE='Rescale RGB')
  Autoscale_base = WIDGET_BASE(RBG_Base, UNAME='AUTOSCALE_BASE', XOFFSET=128  $
      ,YOFFSET=param_offset ,SCR_XSIZE=120 ,SCR_YSIZE=26 ,COLUMN=1  ,/EXCLUSIVE)
  RGB_ABS_SCALE = Widget_Button(AUTOSCALE_BASE,  $
      UNAME='Absolute scale' ,XOFFSET=50 ,YOFFSET=14 ,/ALIGN_LEFT  $
      ,VALUE='Absolute scale')

  circle_base = WIDGET_BASE(RBG_Base, UNAME='CIRCLE_BASE', XOFFSET=10  $
      ,YOFFSET=param_offset + 26 ,SCR_XSIZE=90 ,SCR_YSIZE=26 ,COLUMN=1  ,/NONEXCLUSIVE)
  RGB_BUTTON_circle = Widget_Button(CIRCLE_BASE,  sensitive = 0, $
      UNAME='RGB_BUTTON_circle' ,XOFFSET=50 ,YOFFSET=14 ,/ALIGN_LEFT  $
      ,VALUE='color circle')
  scale_base = Widget_Base(RBG_BASE, UNAME='SCALE_BASE' ,XOFFSET=130  $
      ,YOFFSET=param_offset + 26 ,COLUMN=1, /NONEXCLUSIVE)
  RGB_BUTTON_scales = Widget_Button(SCALE_BASE, sensitive = 0,  $
     UNAME='RGB_BUTTON_scales', /ALIGN_LEFT ,VALUE='scales')

  RGB_BUTTON_cancel = Widget_Button(RBG_BASE,  $
      UNAME='RGB_BUTTON_cancel' ,XOFFSET=5 ,YOFFSET=param_offset + 60  $
      ,SCR_XSIZE=100 ,SCR_YSIZE=40 ,/ALIGN_CENTER ,VALUE='Dismiss')
  RGB_BUTTON_save = Widget_Button(RBG_BASE, UNAME='RGB_BUTTON_save'  $
      ,XOFFSET=120 ,YOFFSET=param_offset + 60 ,SCR_XSIZE=100 ,SCR_YSIZE=40  $
      ,/ALIGN_CENTER ,VALUE='Save')

  text = 'Left button - lower limit      Right button - upper limit      Both or middle button = move both'
  tlabel = widget_label(RBG_BASE, XOFFSET=10,YOFFSET= param_offset + 120,value=text)

; populate the Widget parameters

; -------- list of image files
rgb_label = strarr(10)
rgb_label(0) = 'none'
FOR I = 1,9 DO BEGIN
	HANDLE_VALUE, Data(i), tmp
	IF n_tags(tmp) NE 0 THEN BEGIN
		if tmp.t EQ '2d' then rgb_label(i) = label(i) else rgb_label(i)= 'not an image'
	ENDIF ELSE rgb_label(i)= 'empty'
ENDFOR
red_index   = 4
green_index = 5
blue_index  = 6
widget_control, RGB_TEXT_INDEX_Red, set_value = strtrim(string(red_index),2)
widget_control, RGB_DROPLIST_red, set_value = rgb_label, SET_DROPLIST_SELECT = red_index
widget_control, RGB_TEXT_INDEX_Green, set_value = strtrim(string(green_index),2)
widget_control, RGB_DROPLIST_green, set_value = rgb_label, SET_DROPLIST_SELECT = green_index
widget_control, RGB_TEXT_INDEX_Blue, set_value = strtrim(string(blue_index),2)
widget_control, RGB_DROPLIST_blue, set_value = rgb_label, SET_DROPLIST_SELECT = blue_index

; -------- set initial values of min/max ---------------------
r_min = 0 & r_max = 0
g_min = 0 & g_max = 0
b_min = 0 & b_max = 0
hist_bins = 50
r_hist = fltarr(2,hist_bins)
g_hist = r_hist & b_hist = r_hist

HANDLE_VALUE, Data(red_index), tmp
IF n_tags(tmp) NE 0 THEN BEGIN
	if tmp.t EQ '2d' then begin
		r_min=min(tmp.d,max=r_max)
		r_hist = ax_histogram(tmp.d, n_bins=hist_bins)
	endif
ENDIF
HANDLE_VALUE, Data(green_index), tmp
IF n_tags(tmp) NE 0 THEN BEGIN
	if tmp.t EQ '2d' then begin
		g_min=min(tmp.d,max=g_max)
		g_hist = ax_histogram(tmp.d, n_bins=hist_bins)
	endif
ENDIF
HANDLE_VALUE, Data(blue_index), tmp
IF n_tags(tmp) NE 0 THEN BEGIN
	if tmp.t EQ '2d' then begin
		b_min=min(tmp.d,max=b_max)
		b_hist = ax_histogram(tmp.d, n_bins=hist_bins)
	endif
ENDIF
; --------- store the absolute limits of images
Ar_min = r_min & Ar_max = r_max
Ag_min = g_min & Ag_max = g_max
Ab_min = b_min & Ab_max = b_max

; ----- set up sliders and values
S_margin = 0.05
if r_min EQ r_max then r_sens = 0 else r_sens = 1
widget_control, RGB_TEXT_Rmin, set_value=num_format(r_min), sensitive=r_sens
widget_control, RGB_TEXT_Rmax, set_value=num_format(r_max), sensitive=r_sens
RS_min = double(r_min - S_margin*(r_max-r_min))
RS_max = double(r_max + S_margin*(r_max-r_min))
Rvalue = [double(r_min), double(r_max)]
RS_base = WIDGET_BASE(RBG_Base, UNAME='WID_BASE_R', TITLE="Red", XPAD = 5, YPAD = 5, $
           XOFFSET=T_size, YOFFSET=hist_Vsize, SCR_YSIZE=70,SCR_XSIZE=200)
sliderR = CW_RSLIDER(RS_BASE, UNAME='SLIDER_R', color=[255,0,0],  $
            XSIZE=196, YSIZE = 20, $
            MINIMUM=RS_min, MAXIMUM=RS_max, VALUE=Rvalue)

if g_min EQ g_max then g_sens = 0 else g_sens = 1
widget_control, RGB_TEXT_Gmin, set_value=num_format(g_min), sensitive=g_sens
widget_control, RGB_TEXT_Gmax, set_value=num_format(g_max), sensitive=g_sens
GS_min = double(g_min - S_margin*(g_max-g_min))
GS_max = double(g_max + S_margin*(g_max-g_min))
Gvalue = [double(g_min), double(g_max)]
GS_base = WIDGET_BASE(RBG_Base, UNAME='WID_BASE_G', TITLE="Green", XPAD = 5, YPAD = 5, $
           XOFFSET=T_size, YOFFSET=hist_Vsize + color_size, SCR_YSIZE=70,SCR_XSIZE=200)
sliderG = CW_RSLIDER(GS_BASE, UNAME='SLIDER_G', color=[0,255,0],  $
            XSIZE=196, YSIZE = 20, $
            MINIMUM=GS_min, MAXIMUM=GS_max, VALUE=Gvalue)

if b_min EQ b_max then b_sens = 0 else b_sens = 1
widget_control, RGB_TEXT_Bmin, set_value=num_format(b_min), sensitive=b_sens
widget_control, RGB_TEXT_Bmax, set_value=num_format(b_max), sensitive=b_sens
BS_min = double(b_min - S_margin*(b_max-b_min))
BS_max = double(b_max + S_margin*(b_max-b_min))
Bvalue = [double(b_min), double(b_max)]
BS_base = WIDGET_BASE(RBG_Base, UNAME='WID_BASE_B', TITLE="Blue", XPAD = 5, YPAD = 5, $
           XOFFSET=T_size, YOFFSET=hist_Vsize + 2*color_size, SCR_YSIZE=70,SCR_XSIZE=200)
sliderB = CW_RSLIDER(BS_BASE, UNAME='SLIDER_B', color=[0,0,255],  $
            XSIZE=196, YSIZE = 20, $
            MINIMUM=BS_min, MAXIMUM=BS_max, VALUE=Bvalue )

; ------- indicate start condition is auto rescaled, wihout circle or scales
rescale_flag = 1
WIDGET_CONTROL, RGB_AUTO_SCALE, Set_button = rescale_flag
absolute_flag = 0
WIDGET_CONTROL,RGB_ABS_SCALE, Set_button = absolute_flag
circle_flag = 0
scales_flag = 0

; --------- realize and load the widget --------
Widget_Control, /REALIZE, RBG_BASE
; ---- get index of the draw and histogram windows
WIDGET_CONTROL, RGB_image, GET_VALUE = RGB_display
WIDGET_CONTROL, Red_histogram, GET_VALUE = Red_hist_ID
WIDGET_CONTROL, Green_histogram, GET_VALUE = Green_hist_ID
WIDGET_CONTROL, Blue_histogram, GET_VALUE = Blue_hist_ID

; --------- generate histogram plots
Red_histogram_update
Green_histogram_update
Blue_histogram_update

; generate first RGB image (if there are 3 valid images)
MakeRGB

XManager, 'RBG_BASE', RBG_BASE

end

