; Copyright (c) 1998-2010 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	READ_DM_DATACUBE
;
;LAST CHANGED: ----------------------------------- 23-Feb-10 (aph)
;
;PURPOSE:
;	This function reads in a binary datacuve I(x,y,E) file
; exported from Digital Micrograph and writes out an axis format ncb file
; The user must supply the image and spectral dimensions
; data is 4-bytes

;
;CATEGORY:
;	STAND ALONE: read in utility
;
;CALLING SEQUENCE:
;	Result = READ_DM_DATACUBE([file=file, NOF=nof, group=group)
;
;CALLED FROM AXIS:
;	Read->Stacks
;
;INPUTS: none. All input parameters are passed as keywords.
;
;KEYWORDS:
;	FILE 	filename to read
;	GROUP	group ID (aXisID)
;	VERBOSE print log information about processing
;	EXTRA 	passed on parameters
;
;OUTPUTS:
;	RESULT = AXIS 1d  structure
;
;COMMON BLOCKS: none
;
;MODIFICATION HISTORY:
; (14-Feb-10 aph) first version
; (20-Feb-10 aph) change name to avoid overwriting *.dat files
; (23-feb-10 aph) remove real for now as it gave problems with other stacks

Function read_dm_datacube, file=file, NOF=nof, verbose=verbose, group=group
@axis_com
@stack_process_com
COMMON volume_data, image_stack
@bsif_com
on_error,2

if NOT keyword_set(group) then group = 0 else begin
	if n_elements(axis_ID) EQ 1 then group = axis_ID
 endelse

IF NOT keyword_set(file) then begin
	if keyword_set(nof) then fltr='*' else  fltr='*.dat'
	file = pickfile2(/READ, FILTER=fltr, /LPATH, DEFPATH=defpath)
ENDIF
s = ''
if file EQ '' THEN RETURN, s  ; bail-out if no filename
check = findfile(file)		 ; bail-out if non-existent file
if strlen(check(0)) EQ 0 then begin
	axis_log, 'Cannot find ' + file
	return, s
endif
t = ax_name(file)
file_path = t(0)
fileshort = t(1)

; ---------- define size of the datacube
nx =24 & ny = 24 & nen = 1340
if keyword_set(group) then begin
	nx = get_num(prompt='Number of columns (x)', val = nx, group = axis_ID)
endif else	nx = get_num(prompt='Number of columns (x)', val = nx)
if keyword_set(group) then begin
	ny = get_num(prompt='Number of rows (y)', val = ny, group = axis_ID)
endif else	ny = get_num(prompt='Number of rows (y)', val = ny)
if keyword_set(group) then begin
	nen = get_num(prompt='Number of energy channels', val = nen, group = axis_ID)
endif else	nen = get_num(prompt='Number of energy channels', val = nen)

dc = fltarr(nx,ny,nen)

on_ioerror, read_error
get_lun, iunit
openu, iunit, file
readu, iunit, dc
close, iunit & free_lun, iunit

; return, dc
ncols = nx
nrows = ny
npts = nen
print, 'nx, ny, #channels ', ncols, nrows, npts
evc = 1
if keyword_set(group) then begin
	evc = get_num(prompt='Energy (eV) per channel', val = evc, group = axis_ID)
endif else	evc = get_num(prompt='Energy (eV) per channel', val = evc)


; -------- Create aXis2000 stack
image_stack=dc              ;reform(dc, ncols, nrows, npts)
ev_max= npts*evc & ev_min = 0
slope = (ev_max - ev_min)/npts
ev = ev_min + slope*findgen(npts)
x_start = 0.
x_stop = float(ncols-1.)
x_step = 1.
y_start = 0.
y_stop = float(nrows-1.)
y_step = 1.
filename_ev_msec_list = strarr(npts)
for i = 0, npts-1 do filename_ev_msec_list(i) = string(ev(i))

; --------- ask user what file name they want - default name is same as files without counter
fileout = file_path + fileshort + '_axis2000.ncb'
sname=pickfile2(/write, file= fileout, filter='*.ncb', title = 'name of binary stack file')
stack_wb, sname    ;, /real		; use REAL keyword to avoid loss of z-resolution

axis_log, 'DM data cube read from ' + fileshort
return, s

read_error :
axis_log, 'read_dm_datacube: read error - perhaps data cube dimensions are wrong'
end

