; Copyright (c) 1998-2002 A.P. Hitchcock  All rights reserved
;+
;NAME: LN_LOAD
;
;LAST CHANGED: ----------------------------------- 	12-may-02
;
; PURPOSE:
;	This function loads a Beamline 7 STXM Line scan array
; from a text file (*,xa*, *,dat) and maps line-scan to (y)
; Energy-scan to (x)
;
; CATEGORY:
;	data read-in. stand-alone, or from axis.
; READ~ALS_STXM7.0.1 linescan
;
; CALLING SEQUENCE:
;	Result = LN_LOAD(file, FILTER=fltr, plot=pl, _extra=e)
;
; INPUTS:
; FILE	name of file
;
; KEYWORDS:
;	FILTER - extensiob
;	PLOT - if set, plot data
;
; OUTPUTS:
;	No explicit outputs.   A new window is created if necessary
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;
; MODIFICATION HISTORY:
; (14-may-98 aph) first developed for line scan mode
; (14-jun-98 aph) axis_com
; (07-aug-98 aph) indicate extension
; (18-jan-99 aph) CHANGE name to ln_load (to group with other LS functions)
;                 include dwell on X-axis label
; (25-jun-99 aph) use ax_name to parse file name
; ( 7-jul-99 aph) FREE_LUN as well as close !
; (15-nov-00 cgz) Replaced call of NCOLS function by explicit (SIZE(var))(1)
; (15-nov-00 cgz) Replaced call of NROWS function by explicit (SIZE(var))(2)
; (13-jun-01 aph) add E to the structure generated
; (31 oct-01 aph) work around for wrong 'npts' in new linescan format; (12-may-02 aph) AXIS standard header added
;-

function ln_load, file, FILTER=fltr, plot=pl, _extra=e
@axis_com
on_error,2

if n_elements(file) eq 0 then begin  ;popup file dialog box
   if not keyword_set(fltr) then fltr='*.xa*'
   file=PICKFILE2(/Read, FILTER=fltr, /LPATH, DEFPATH=defpath)
endif

if strlen(file) GT 0 THEN BEGIN
	name_bits = ax_name(file)
	fileshort= name_bits(1)
	ext = name_bits(2)
	file = name_bits(0) + fileshort ; take all but the extension
; (aph 2-mar-98 - converted to jacobsen's readin as it does not assume SQUARE input data !)
; puts energy and dwell parameters into the nsls SCANDATA (SD) structure
openr,lun,(file+'.dat'), /get_lun
input_string = ''
readf,lun,input_string
;print, '1:  ', input_string
reads,input_string, ndark, nsect, ptdelay, drk_dwell, dwell
dwell = 0.001*dwell  	;(convert to msec)
;print, strcompress(string(fix(nsect))),' sect. Dwell (ms)= ',strcompress(string(dwell))
readf,lun,input_string
; print, '2:  ', input_string

; from Steele - 4-may-98
;The coordinates on line 2 are given in mm. The numbers on line two are:
;(1)  starting X position
;(2)  X step size
;(3)  starting Y position
;(4)  Y step size
;(5)  number of points measured along line
; So in the example you show, the starting position of the line is, in
; microns, (-5.8104, -3.4569) and the endpoint of the line is
; (-5.8104+200*.045021,-3.4569+200*.020232) = (3.1938, 0.5895)

reads,input_string, x1,xstep,y1,ystep,npts
x1 = 1e3*x1   & xstep = 1e3*xstep  &  y1 = 1e3*y1 & ystep = 1e3*ystep
step = sqrt(xstep^2 + ystep^2)
; print, 'line (um) 0 - ', strcompress(string(line(npts-1)))
Edef=fltarr(nsect,4)
nenergy=0  & energy=0   & demin = 500.
for i = 0, nsect-1 do begin
	readf,lun,input_string
;	print,  input_string
	reads,input_string, a0,a1,a2,a3
	Edef(i,0)=a0 & Edef(i,1)=a1 & Edef(i,2)=a2 & Edef(i,3)=a3
    Esect=findgen(Edef(i,2))*Edef(i,1) + Edef(i,0)
    energy=[energy,Esect]
	demin = min([demin,a1])
	nenergy = fix(Edef(i,2)+nenergy)
;    print, 'sect ',strcompress(string(i+1)),' :  ', Esect(0),' - ', $
;       Esect(Edef(i,2)-1),' .', Edef(i,2),' points'
endfor
energy=energy(1:*)   ; remove starter point
;print, 'npts, Escale', nenergy, energy(0), energy(nenergy-1)
sd.dwell_time = dwell       ; report dwell in msec
close,lun & free_lun,lun

sd.wavelength=12398./Edef(0,0)    ; save FIRST photon energy as wavelength

a=text_read1(file+'.'+ext)
a=rotate(a,4)		; swap x,y axes to put E-scale horizontal
t = size(a) & npts = t(2)
line = findgen(npts)*step             ; generate line

;text = string(format='(a12," : Line scan:  ",i3," pts x ",f4.1," ms/pixel. ",i3," energies, ",f6.2," - ",f6.2," eV.")', $
;	    fileshort,nrows(a), dwell, ncols(a)-1 , min(energy), max(energy))
text = string(format='(a12," : Line scan:  ",i3," pts x ",f4.1," ms/pixel. ",i3," energies, ",f6.2," - ",f6.2," eV.")', $
	    fileshort,(size(a))(2), dwell, (size(a))(1)-1 , min(energy), max(energy))
print, text
WIDGET_CONTROL, Uprompt, Bad_ID=BadID, SET_VALUE = text

; determine minimum x,y -spacing to establish #  points
espan = energy(nenergy-1) - energy(0)
ten= fix(espan/demin(0))
;  make a linearised interpolated square array
ai = fltarr(1,(size(a))(2))
hi = 0
for i = 0, nsect-1 do begin
	lo = hi
	hi = fix(lo + edef(i,2))
	ni = fix(ten*(edef(i,1)*edef(i,2))/espan)+1
;	print, 'sect, original #, interp #, lo, hi', i+1,fix(hi-lo),ni,lo,(hi-1)
	asect= congrid(a(lo:hi-1,*),ni,(size(a))(2),/interp)
;	help,asect
	ai=[ai,asect]
endfor
ai=ai(1:*,*)   ; remove starter point
en_size=size(ai)
enrgy_int = edef(0,0) + findgen(en_size(1))*espan/en_size(1)
x_lbl = string(format='("E (eV)    dwell ",f6.3," ms")',dwell)
;s={t:'4d', d:a, di:ai, x:energy, xi: enrgy_int, y:line, xl:'E (eV)', yl:'r (um)', dl:'linescan '+fileshort}
s={t:'2d', d:ai, x:enrgy_int, y:line, xl:x_lbl, yl:'als ('+ext+')r (um)', $
    E: 0, dl:'LS '+fileshort}
if keyword_set(pl) then splot2d, s, _extra=e, charsize=1

endif else begin 	; go here if user pushes 'cancel'
    print, ' file name of zero length. Read aborted'
    s=0
endelse

return, s
end