; Copyright (c) 1998-2010 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	WRITE_CTS
;
;LAST CHANGED: ----------------------------------- 24-Jul-2010
;
; PURPOSE:
;	This function converts an aXis binary stack to set of ascii spectra (*.cts)
; suitable for input to PyMCA
;
; CATEGORY:
;	utiltiy
;
; CALLING SEQUENCE:
;	RESULT = WRITE_CTS(group=group, verbose=verbose, silent=silent, _extra=e)
;
; INPUTS:
;
; KEYWORDS:
;	GROUP		group leader (Axis_ID if called from aXis2000 and axis)
;	MULTI_DET 	multiple detector signals in one file
;	SILENT  	if set, no feedback
;	EXTRA 		passed on parameters
;
; OUTPUTS:
; set of 1-column ascii files suitable to read into PYMCA
; sum of all all XRF spectra
;
; COMMON BLOCKS:
;   @AXIS_COM - set of common blocks for AXIS
;	Volume_data - stack
;   @BSIF_COM - common block for netCDF
;
; SIDE EFFECTS:
;
; RESTRICTIONS:
; assumes all XGLabs file names involve cycling the last 3 spaces before '.' as the sequence number
;
; PROCEDURE:
;
; RESTRICTIONS:
;
; MODIFICATION HISTORY:
; (26-Feb-10 aph) first developed from ax_xrf_convert
; (24-Jul-10 aph) correct the filename generation algorithm
;-

FUNCTION WRITE_CTS, group=group, verbose=verbose, silent=silent, _extra=e
@axis_com
@stack_process_com
COMMON volume_data, image_stack
@bsif_com
on_error,2

print, 'WRITE_CTS - converting aXis2000 stack to 1-column count files'

s= 0

fileget=PICKFILE2(/Read, FILTER='*.ncb', /LPATH, DEFPATH=defpath)
if strlen(fileget) GT 0 THEN BEGIN
 	t = ax_name(fileget)
	file_name = t(1) + '.ncb'
	fileshort=t(1)
	file_path = t(0)
	stack_rb, (file_path + file_name)		;read stack in ncb format
	ts = size(image_stack)
	print, ts
	n_rows = ts(1)
	n_cols = ts(2)
	n_data = ts(3)
endif else return, s

out_dir = fileshort + '_cts'
if keyword_Set(group) then  $
     out_dir=get_text(prompt ='name of folder for output', val = out_dir, group=group) $
     else out_dir=get_text(prompt ='name of folder for output', val = out_dir)
file_path_out= file_path + out_dir
file_mkdir, file_path_out
if not keyword_set(silent)then begin
;	print, 'file_path= ', file_path
;	print, 'fileshort= ', fileshort
;	print, 'fileroot= ', fileroot
;	print, ''
	axis_log, 'Output files (*.cts) stored in ' + file_path_out
endif

out_name = fileshort + '_'
if keyword_Set(group) then  $
     out_name=get_text(prompt ='name of *.cts files', val = out_name, group=group) $
     else out_name=get_text(prompt ='name of *.cts files', val = out_name)

; ------ write each spectrum (image_stack(*,*.E)
spec = intarr(n_data)
spec_sum = fltarr(n_data)
Widget_control,/hourglass
zero = '000'
off_z = 3
if n_rows*n_cols GT 1000 then begin
  zero = '0000'
  off_z = 4
endif
if n_rows*n_cols GT 10000 then begin
  zero = '00000'
  off_z = 5
endif
if n_rows*n_cols GT 100000 then begin 
  zero = '000000'
  off_z = 6
endif
for i = 0, n_rows-1  do begin
	for j = 0, n_cols-1 do begin
		spec = image_stack(i,j,*)
		spec_sum = spec_sum + float(spec)
    num = fix(i*n_cols+j)
    t_num = strtrim(strcompress(string(num)),2)
    num_txt = zero
      strput, num_txt, t_num, off_z-strlen(t_num)
      tn = strcompress(num_txt,/remove_all)
  		fileout = out_name + tn
		fileout = file_path_out + ax_sep() + fileout + '.cts'
		openw, lun, fileout, /get_lun
		printf,lun, spec, format='(I7)'
		close, lun & free_lun, lun
	endfor
endfor

; ------- convert sum of all XRF spectra to axis 1d structure, with binning as needed
xl = 'Channel #'
yl = 'sum of ' +  string(n_rows*n_cols) + ' XRF spectra '
x = indgen(n_data)
d = spec_sum
dl = out_name
fileout = file_path_out + ax_sep() + out_name + 'sum.txt'
s = {t:'1d', x:x, d:d, dn:d, xl:xl, yl:yl, dl:dl}
q=spc_save(s, file=fileout, /noLbl)
t=ax_name(q)
if not keyword_set(silent)then  axis_log, 'sum of all spectra wrote to ' + t(0)
return, s

end
