; Copyright (c) 1998-2013 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		READ_H5_PTYCHO
;
;LAST CHANGED: ----------------------------------- 	15-Jan-2013
;
; PURPOSE:
;	This function reads content of *.pt  pytography results file
; using the IDL H5_browser (post IDL 6.0)
;
; CATEGORY:
;	data read-in; operates stand alone or in aXi2000
;
; CALLING SEQUENCE:
;	Result = READ_H5_PTYCHO(file=file,  energy = energy, dwell = dwell, pix_size = pix_size)
;
; INPUTS: keyword only
;
; KEYWORDS:
;	AXIS	indicates called from axis
;	DWELL	dwell time of measurement
;	ENERGY 	energy of image
;	FILE	name of file
;	PIX_SIZE	pixel size of ptycograph
;
; OUTPUTS:
;	Returns an aXis format image (for the extracted STXM file)
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks

; MODIFICATION HISTORY:
; (28-nov-12 aph) first version as a script for only STXM_image
; (04-Dec-12 aph) AXIS standard header added
; (08-Dec-12 aph) use ptycho_select to access all items
; (15-Jan-13 aph) neaten up
;-

function read_h5_ptycho, file = file, energy = energy, dwell = dwell, pix_size = pix_size, axis = axis
@axis_com
@ptycho_select_com
on_error,2

; determine if AXIS is running (therefore may have called this function)
; either when AXIS keyword is supplied or if any widget active
if  keyword_set(axis) then axis_on = 1 else axis_on = widget_info(/active)

IF NOT keyword_set(file) then begin
	fltr='*.pt'
	file = pickfile2(/READ, FILTER=fltr, /LPATH, DEFPATH=defpath)
ENDIF
s = ''
if file EQ '' THEN RETURN, s  ; bail-out if no filename
check = findfile(file)		 ; bail-out if non-existent file
if strlen(check(0)) EQ 0 then begin
	axis_log, 'Cannot find ' + file
	return, s
endif

; ---------  construct short name
t = ax_name(file) & fileshort = t(1)

; ----- use h5_browser to examine file and select item
a=h5_parse(file,/read_data)

; --------- get label from data file
t = ax_name(a._file) & ptycho_label = t(1)

; ------------ determine photon energy
; Energy = 708.2
if n_elements(ptycho_E) GT 0 AND NOT keyword_set(energy) then energy = ptycho_E
if axis_on then energy = get_num(Prompt = 'Energy (eV)', Val = energy, group = AXIS_ID) $
		else  energy = get_num(Prompt = 'Energy (eV)', Val = energy)
ptycho_E = energy	; save for later use
print, 'Energy (eV) = ', energy

; ------------ determine dwell
; dwell = 1000
if n_elements(ptycho_dwell) GT 0 AND NOT keyword_set(dwell) then dwell = ptycho_dwell
if axis_on then dwell = get_num(Prompt = 'dwell (ms)', Val = dwell, group = AXIS_ID) $
		else  dwell = get_num(Prompt = 'dwell (ms)', Val = dwell)
ptycho_dwell = dwell	; save for later use
print, 'dwell (ms) = ', dwell


; --------- launch widget to select ptychography data item -------
pt_item_list = strarr(6)
pt_item_list(0) = 'Diffraction_stack'
pt_item_list(1) = 'Illumination_map'
pt_item_list(2) = 'Probe_real'
pt_item_list(3) = 'Probe_imaginary'
pt_item_list(4) = 'STXM_image'
pt_item_list(5) = 'Pmask'

if axis_on then ptycho_select, GROUP_LEADER = AXIS_ID else  ptycho_select

CASE string(pt_item) of

'Diffraction_stack': BEGIN

axis_log, ' not yet written. 15-Jan-13 aph'
END


'Illumination_map': BEGIN

	d = a.Illumination_Map._data
	t = size(d)
	x=findgen(t(1))
	y=findgen(t(2))
	yl = 'Illumination map'
	xl =  string(dwell, energy, format=  '("dwell = ",f8.1," ms.  E = ",f8.2," eV")')
	s = {t:'2d', x:x, y:y, d:d, e: energy, xl:xl, yl: yl, dl: ptycho_label}
	t=axb_save(s)

	print, 'Illumination map read from ', fileshort
	return, s
END

'Probe_real': BEGIN

	t = a.Probe._dimensions
	nx = t(0)  & ny = t(1)
	d = fltarr(nx, ny)
	d= a.probe._data.r
	x=findgen(nx)
	y=findgen(ny)
	yl = 'ptychography: probe - REAL'
	xl =  string(dwell, energy, format=  '("dwell = ",f8.1," ms.  E = ",f8.2," eV")')
	s = {t:'2d', x:x, y:y, d:d, e: energy, xl:xl, yl: yl, dl: ptycho_label}
	t=axb_save(s)

	print, 'Probe (REAL values)  read from ', fileshort
	return, s

END

'Probe_imaginary': BEGIN

	t = a.Probe._dimensions
	nx = t(0)  & ny = t(1)
	d = fltarr(nx, ny)
	d= a.probe._data.i
	x=findgen(nx)
	y=findgen(ny)
	yl = 'ptychography: probe - IMAGINARY'
	xl =  string(dwell, energy, format=  '("dwell = ",f8.1," ms.  E = ",f8.2," eV")')
	s = {t:'2d', x:x, y:y, d:d, e: energy, xl:xl, yl: yl, dl: ptycho_label}
	t=axb_save(s)

	print, 'Probe (IMAGINARY values)  read from ', fileshort
	return, s

END
'Pmask': BEGIN

axis_log, ' not yet written. 15-Jan-13 aph'

END


'STXM_image': BEGIN
; ----------------------- read in STXM_image ----------------
	d=rotate(float(a.stxm_image._data),7)
	; help, d

	; ------------ determine pixel size
	; pix_size = 60
	if n_elements(ptycho_pix) GT 0 AND NOT keyword_set(pix_size) then pix_size = ptycho_pix
	if axis_on then pix_size = get_num(Prompt = 'pix_size (nm)', Val = pix_size, group = AXIS_ID) $
			else  pix_size = get_num(Prompt = 'pix_size (nm)', Val = pix_size)
	ptycho_pix = pix_size	; save for later use - still in nm
	pix_size = 0.001*float(pix_size)
	print, 'pixel size (um) = ', pix_size

	t = size(d)
	x=findgen(t(1))*pix_size
	y=findgen(t(2))*pix_size
	xl = string(dwell, energy, format=  '("dwell = ",f8.1," ms.  E = ",f8.2," eV")')
	yl = 'ptycography from ' + fileshort
	s = {t:'2d', x:x, y:y, d:d, e: energy, xl:xl, yl: yl, dl: ptycho_label}
	t=axb_save(s)

	return, s
  END

ELSE: print, pt_item, ' item not found in HDF5 file'

ENDCASE

end

