; Copyright (c) 1998-2015 C. Zimba & A.P. Hitchcock  All rights reserved
;+
; NAME:
;	ZSTACK_SPECTRA.PRO
;
; LAST CHANGED: ----------------------------------- 03-aug-15
;
; AUTHORS:
;	Carl G. Zimba (NIST), Chris Jacobsen (SUNY - Stonybrook)
; PURPOSE:
;	Analysis of images and spectra abtained from a series of images from an x-ray microscope.
; CATEGORY:
;	Data analysis.
; CALLING SEQUENCE:
;	zstack_spectra
; INPUTS:
;	NONE
; KEYWORD PARAMETERS:
;	NONE
; OUTPUTS:
;	NONE
; COMMON BLOCKS:
;	zstack_common	:
;		image_stack			:	3-D matrix of multiple x-ray microscope images
;		filename_list			:	list of filename corresponding to images in image_stack
;		ev_list				:	list of x-ray energies corresponding to images in image_stack
;		msec_list				:	list of dwell times corresponding to images in image_stack
;		filename_display_list		:	list of filename, x-ray energy, and dwell time corresponding to images in image_stack
;		displayed_file_index		:	index in image_stack, filename_list, msec_list, and filename_display_list currently being displayed or processed
;		data_source			:	type of x-ray microscopy data: STXM:1, SXM:2, SM:3, ALS:4, POLY:5
;		image_zoom				:	zoom factor for displaying images
;		init_zoom				:	initial zoom factor for displaying images (used in ZSTACK_SAVE and ZSTACK_TUNE)
;		disp_min				:	minimum percentage intensity for display of images
;		disp_max				:	maximum percentage intensity for display of images
;		disp_gamma				:	gamma factor for display of images
;		movie_delay			:	delay used to display movie images of data stacks
;		n_files				:	number of images
;		list_filename			:	filename of image list
;		shift_filename			:	filename of alignment shifts
;		x_shift				:	array of x-coordinate alignment shifts
;		y_shift				:	array of y-coordinate alignment shifts
;	zstack_analyze_common			:
;		zstack_spectra_par			:	variables controlling ZSTACK Spectra dialog window
;		zstack_save_par			:	variables controlling ZSTACK Save dialog window
;		roi_index				:	vector of pixels corresponding to regions of interest for all i spectra
;		i_roi					:	vector of pixels corresponding to region of interest for current i spectrum
;		n_roi					:	number of regions of interest defined for i spectra
;		i0_roi				:	vector of pixels defining region of interest for i0 spectrum
;		is_i0_roi				:	Type of region of interest: 0: i0, 1: i
;		image_scale			:	Display images normalized by: 0: inv_image_stack, 1: i0 spectrum
;		image_ratio			:	Display images normalized by: 0: inv_image_stack, 1: i0 spectrum
;		ratio_image_index		:	index of image to use when ratio images to a constant image
;		spectrum				:	2-D matrix (j,k) of spectra corresponding to defined regions of interest, j=0: i0 spectrum, j=1-14: i spectra, k corresponds to x-ray energy of image
;		save_spectra_type		:	Type of file to save spectra as:
;		spectra_filename_header	:	base filename of spectra and regions of interest to be saved on ZSTACK Spectra dialog window
;		save_filename_header		:	base filename of images to be saved on ZSTACK Save dialog window
;		i0_filename			:	base filename of i0 spectrum tp be retrieved
;		i0_filetype			:	Type of file of retrieved I0 spectrum:	RAW, XAS, SPC
;		i_roi_color_index		:	vector of indices of colors for display of regions of interest for i spectra
;		i0_roi_color_index		:	color index for display of region of interest for i0 spectrum
;		data_shifted			:	0: data was not shifted and should not be clipped, 1: data was shifted and should be clipped, -1: denotes ZSTACK_PROFILE.PRO is being run as a stand-alone procedure
;		n_clipped_cols			:	number of columns in clipped image
;		n_clipped_rows			:	number of rows in clipped image
;		clipbox				:	vector of four points defining dimensions of region unclipped by alignment: [xleft,xright,ybot,ytop]
;		do_edge_clip			:	Clip edges of null data created by image alignment:	0: NO, 1: YES
;		init_zoom				:	intial value of image zoom factor
;	zstack_color_common
;		bottom_color_index		:	index of lowermost color of gradient colorscale
;		top_color_index			:	index of uppermost color of gradient colorscale
;		black_color_index		:	index of black color
;		white_color_index		:	index of white color
;		blue_color_index		:	index of blue color
;		red_color_index			:	index of red color
;		green_color_index		:	index of green color
;		cyan_color_index		:	index of cyan color
;		magenta_color_index		:	index of magenta color
;		yellow_color_index		:	index of yellow color
;		lime_color_index		:	index of lime green color
;		gold_color_index		:	index of gold color
;		rose_color_index		:	index of rose color
;		yelgrn_color_index		:	index of yellow-green color
;		purple_color_index		:	index of purple color
;		orange_color_index		:	index of orange color
;		sky_color_index			:	index of sky blue color
;		beige_color_index		:	index of beige color
; SIDE EFFECTS:
;
; RESTRICTIONS:
;
; PROCEDURE:
;   Called from within ZSTACK_ANALYZE.PRO.
; EXAMPLE:
;
; MODIFICATION HISTORY:
;   25-Mar-1998 - now works with 24 bit graphics, CJJ
;   24-Jul-1998 - two changes (based on suggestions by Adam
;     Hitchcock, but the implementations here are my own) CJJ :
;     1. allow user to read in i0 files from an ascii file of
;        (line by line) values of ev, khz.  The file is used
;        only if it spans a larger range of eV values than the
;        images in the stack span.
;     2. Click on the spectrum to change the file which is displayed
;   14-Aug-1998 - now if there's an alignment file, read it in and
;     apply it but STILL give the user the chance to align the stack.
;     This allows one to (for example) first align a stack manually,
;     and then do an automatic alignment over a restricted shift
;     range to "refine" the alignment. CJJ
;   28-Aug-1998: with IDL 5.1 the way 24 bit graphics is handled is
;     now different - need to invoke device,decomposed=0, CJJ
;   31-Aug-1998: Allow saving and reading of stack files as XDR data,
;     don't read ROI file by default on startup, use zoom=1 unless
;     explicitly told otherwise, have alignment shifts saved by
;     this program rather than by alignment programs, allow clipping
;     of data to exclude shift boundaries, CJJ
;   27-sep-1998: but tvrd() works differently on different devices
;     so need to write images for GIF files to Z buffer, CJJ
;   13-oct-1998: Save many images at once, start to work on
;     ratio image stuff, CJJ
;
; Modified 20feb99, CGZ
; Extensively modified and rewritten STACK_ANALYZE.PRO to form ZSTACK_ANALYZE.PRO
; Changes and additions are numerous:
;
; All procedures and common variables starting with 'stk' were changed to 'zstack'
; so that the existing routine, STACK_ANALYZE.PRO, can be run in parallel
;
; Reorganized STACK_ANALYZE.PRO so that
;	spectral analysis is done with ZSTACK_SPECTRA and zstack_spectra_* procedures
;		spectra and regions of interest are saved with ZSTACK_SPECTRA
;	saving of image data is done with ZSTACK_SAVE and ZSTACK_SAVE_* procedures
;
; Procedures have been modified, added, eliminated as below:
;	zstack_spectra_prep			:	formerly stk_anl_graphics
;	zstack_spectra_imgdisp		:	formerly stk_anl_imgdisp
;	zstack_spectra_plotspectrum	:	formerly stk_anl_plotspectrum
;	zstack_spectra_makespectrum	:	formerly stk_anl_makespectrum
;	zstack_spectra_clip_data		:	formerly stk_anl_clipdata
;	zstack_spectra_desensitive		:	new, adapted from stk_anl_desensitive
;	zstack_spectra_sensitive		:	new, adapted from stk_anl_sensitive
;	zstack_spectra_save_spec		:	new, enhanced from stk_anl_savexas
;	zstack_spectra_save_roi		:	formerly stk_anl_writeroi
;	zstack_spectra_roi			:	formerly stk_anl_roi
;	zstack_spectra_get_i0_file		:	new, enhanced from stack_analyze_event subroutine
;	zstack_spectra_event			:	new, adapted from stk_anl_event
;	zstack_spectra				:	new, adapted from stack_analyze
;	stk_aln_savefile			:	eliminated, moved to ZSTACK_ALIGN.PRO (zstack_align_save_shift)
;	stk_dat_readfile			:	eliminated
;	stk_anl_readroi				:	eliminated
;	stk_anl_roi_event			:	eliminated, replace with CW_DEFROI.PRO
;	stk_anl_roi				:	eliminated, replace with CW_DEFROI.PRO
;
; Functions of procedures are now:
;	zstack_spectra_prep			:	sets color table and color indices and defines common variables
;	zstack_spectra_imgdisp		:	displays images and shift plot
;	zstack_spectra_plotspectrum	:	displays extracted spectra
;	zstack_spectra_makespectrum	:	computes spectra from image intensities within regions of interest
;	zstack_spectra_clip_data		:	defines area of images not clipped by alignment shift
;	zstack_spectra_desensitive		:	deactivate features on Stack Spectra dialog window
;	zstack_spectra_sensitive		:	activate features on Stack Spectra dialog window
;	zstack_spectra_save_spec		:	save extracted spectra in a variety of formats
;	zstack_spectra_save_roi		:	save regions of interest for spectral extraction
;	zstack_spectra_roi			:	define regions of interest for spectral extraction
;	zstack_spectra_get_i0_file		:	new, enhanced from stack_analyze_event subroutine
;	zstack_spectra_event			:	event routine for zstack_spectra
;	zstack_spectra				:	main procedure for extracting spectra from stack
;
; Modified user interface window to include
;	display and movie of image and clipped image
;		regions of interest are plotted on both image and clipped image
;		cursor bar added to plot of spectra highlighting energy and intensity position
;			during movie or static display
;		eliminated text embedded in image, replaced it with text fields in dialog window
;	displayed file can be selected either using file list or spectral plot
;		clicking on spectral plot, prints file info and spectral intensity in Output Log
;	text fields added to display image information during movie or static display
;		this replaces text as part of image implemented in STACK_ANALYZE.PRO
;	eliminated "Add I Pixel" and "Add I0 Pixel" since it is embedded part of CW_DEFROI.PRO
;	eliminated ratio image feature - may restore later
;	eliminated IDL Slicer feature - may restore later
;	"Load New Color Table" option added
;		can change the color table in either ZSTACK_SPECTRA, ZSTACK_SAVE, or ZSTACK_PROFILE
;	saving of spectra in multiple formats: *.xas, *.spc, *.gif
;	added ability to extract spectra from multiple regions simultaneously (up to 14 spectra)
;
; Replaced stk_anl_roi and stk_anl_roi_event with CW_DEFROI.PRO
;	New routine is shipped with IDL and is more robust on multiple platforms
;	Allow selection of regions of interest as polygon, single point, rectangle, or circle
;
; Changed the way the color table is specified:
;	- calls ZSTACK_COLOR.PRO from within zstack_spectra_prep
;	- much simpler now, only executed once, allows multiple colors easily
;
; Deleted unused variables
;	char_ypix,charsize,textregion_nlines,textregion_ypix,
;	i_images_color_index,i0_images_color_index,i0_images_list,i_images_list
;
; Renamed ev to ev_list, and msec to msec_list
;
; Added option for data from Polymer STXM
;	/poly keyword and stxm1_sxm2_sm3_als4_poly5 variable
;	Made stxm1_sxm2_sm3_als4_poly5 part of zstack_common variable
;
; Modified Stack Spectra dialog window for 3 columns, including a long list of image files
;
; Modified COMMON block structure so that common variables are shared with all zstack routines
;	i.e., zstack_common contains variables used by
;		ZSTACK_ANALYZE.PRO, ZSTACK_READLIST.PRO, ZSTACK_BUILDLIST.PRO, and ZSTACK_ALIGN.PRO
; Common variables are now defined only once in each file (in zstack_spectra_prep), not repetitively as before.
; Common variables for data are now determined within ZSTACK_BUILDLIST.PRO or ZSTACK_READLIST.PRO:
;	image_stack, filename_list, ev_list, msec_list, filename_display_list
;	This requires that files be read only once, eliminated time-intensive duplicate calls of
;		STXM_READ.PRO or SXM_READ.PRO
;
; Modified zstack_spectra_par, and zstack_save_par
;	so that they only contain only variables associated with dialog window.
;	All other variables have been moved to zstack_analyze_common
;
; Modified ZSTACK_SAVE_SPEC_SAVE to better utilize save_spectra_type variable
; Moved list_filename and shift_filename variables to zstack_common variable
; Added n_files variable to zstack_common variable
; Header with list_filename and shift_filename now wriiten as part of *.spc and *.roi files
;
; Ratio of image to I0 spectrum appears to be useless
;	Previously, some slight variation in image contrast was present
;	but was likely due to the way the percent_image was computed:
;	Former:	percent_image = 100. * image_stack(*,*,i_file) / max(image_stack(*,*,i_file))
;					(via obsolete invmax_image_stack variable)
;		which didn't scale image intensity accurately between 0 and 100
;	Now:		this_image = image_stack(*,*,i_file) / min(image_stack(*,*,i_file))
;			percent_image = 100. * this_image / max(this_image)
;		which forces image intensity to be accurately scaled between 0 and 100
;	So eliminate 'ratio image to i0 spectrum' option
;	Explore possibility of displaying image using log or exp scale, in addition to present linear scale
;
; Added x and y offsets to zstack_spectra dialog window
;		so that cw_defroi window will be apparent, 30sep99 CGZ
; Minor modifications of sensitive attribute of widget controls for better dialog window
;
; (06oct00 cgz)
;	Created zstack_save_common to include variables for saving various image and movie formats
;		Moved zstack_save_par and save_filename_header from zstack_spectra_common
;		Modified zstack_spectra.pro to conform
; (26oct00 cgz)
;	Added button to invoke IDL Slicer with full (unclipped) aligned image stack
; (31oct00 cgz)
;		Modified zstack_buildlist_savelist to include dialog_pickfile so directory can be selected
;		Replaced platform-dependent code with call to zstack_analyze_extract_filename
; (06nov00 cgz)
;	Moved button to invoke IDL Slicer to zstack_save widget
; (28nov00 cgz) migrated zstack_profile_colorbar routine to zstack_analyze.pro
; (24-feb-01 aph) add on_error,2 to each procedure
; (02-may-04 aph) commented out 'zstack_to_axis' ; added loadct, 0 to try to avoid color scale problems
; (06jul05 aph) add color-restore button
; (04-Sep-07 aph) track down why Zstack_spectra occasionally did not display spectra
;      - due to ev_list being a string [in some cases ??] and the min / max not working properly on strings
; (03-Mar-08 MJ) Keyword /SCROLL in widget_base() needs X_SCROLL_SIZE.
;                Use of /SCROLL and /MODAL at the same time not allowed.
; (13-apr-08 aph) turn on pre-set window size (X_Scroll_size, Y_Scroll_size) only for non-Windows OS
; (07-mar-09 aph) added binary_filenmane to zstack_common
; (27-Jul-09 aph) external common
; (03-aug-09 aph) change base-10 log to natural log (as noted byLarry Nittler, Carnegie)
; (02-aug-15 aph) ensure DefPath is defined correctly as exit Zstack_spectra; added @axis_com to all routines
; (03-Aug-15 aph) changed label to zlabel since label is an array in axis_com
; ;-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_spectra_prep
;print,'zstack_spectra_prep'

@axis_com
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

on_error,2
zstack_color
return
END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_spectra_desensitive
;print,'zstack_spectra_desensitive'

@axis_com
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

on_error,2
widget_control, zstack_spectra_par.add_i0_roi_label, sensitive = 0
widget_control, zstack_spectra_par.reset_i0_label, sensitive = 0
widget_control, zstack_spectra_par.reset_both_i0_i_label, sensitive = 0
widget_control, zstack_spectra_par.add_i_roi_label, sensitive = 0
widget_control, zstack_spectra_par.reset_last_i_label, sensitive = 0
widget_control, zstack_spectra_par.reset_all_i_label, sensitive = 0
widget_control, zstack_spectra_par.i0_filename_label, sensitive = 0
widget_control, zstack_spectra_par.i0_fileformat_list_label, sensitive = 0
widget_control, zstack_spectra_par.i0_retrieve_label, sensitive = 0
widget_control, zstack_spectra_par.i0_browse_label, sensitive = 0
widget_control, zstack_spectra_par.i0_scale_factor_label, sensitive = 0
widget_control, zstack_spectra_par.spectra_filename_label, sensitive = 0
widget_control, zstack_spectra_par.spectra_dataformat_list_label, sensitive = 0
widget_control, zstack_spectra_par.spectra_filetype_list_label, sensitive = 0
widget_control, zstack_spectra_par.spectra_fileformat_list_label, sensitive = 0
widget_control, zstack_spectra_par.spectra_save_label, sensitive = 0
widget_control, zstack_spectra_par.roi_filename_label, sensitive = 0
widget_control, zstack_spectra_par.roi_save_label, sensitive = 0
widget_control, zstack_spectra_par.roi_read_label, sensitive = 0
widget_control, zstack_spectra_par.roi_browse_label, sensitive = 0
IF (movie_active EQ 0) THEN BEGIN
	widget_control, zstack_spectra_par.filename_display_list_label, sensitive = 0
ENDIF ELSE BEGIN
	widget_control, zstack_spectra_par.filename_display_list_label, sensitive = 1
ENDELSE
widget_control, zstack_spectra_par.prev_image_label, sensitive = 0
widget_control, zstack_spectra_par.next_image_label, sensitive = 0
widget_control, zstack_spectra_par.play_movie_label, sensitive = 0
widget_control, zstack_spectra_par.display_parameters_label, sensitive = 0
widget_control, zstack_spectra_par.plot_parameters_label, sensitive = 0
widget_control, zstack_spectra_par.spectra_plot_label, sensitive = 0
widget_control, zstack_spectra_par.profile_label, sensitive = 0
widget_control, zstack_spectra_par.save_label, sensitive = 0
widget_control, zstack_spectra_par.exit_label, sensitive = 0
return
END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_spectra_sensitive
;print,'zstack_spectra_sensitive'

@axis_com
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

on_error,2
widget_control, zstack_spectra_par.add_i0_roi_label, sensitive = 1
IF ((size(i0_roi))(0) NE 0) THEN BEGIN
	widget_control, zstack_spectra_par.reset_i0_label, sensitive = 1
ENDIF ELSE BEGIN
	widget_control, zstack_spectra_par.reset_i0_label, sensitive = 0
ENDELSE
IF ( ((size(i0_roi))(0) NE 0) AND (n_roi NE 0) ) THEN BEGIN
	widget_control, zstack_spectra_par.reset_both_i0_i_label, sensitive = 1
ENDIF ELSE BEGIN
	widget_control, zstack_spectra_par.reset_both_i0_i_label, sensitive = 0
ENDELSE
widget_control, zstack_spectra_par.add_i_roi_label, sensitive = 1
IF (n_roi NE 0) THEN BEGIN
	widget_control, zstack_spectra_par.reset_last_i_label, sensitive = 1
	widget_control, zstack_spectra_par.reset_all_i_label, sensitive = 1
ENDIF ELSE BEGIN
	widget_control, zstack_spectra_par.reset_last_i_label, sensitive = 0
	widget_control, zstack_spectra_par.reset_all_i_label, sensitive = 0
ENDELSE
widget_control, zstack_spectra_par.i0_filename_label, sensitive = 1
widget_control, zstack_spectra_par.i0_fileformat_list_label, sensitive = 1
IF (strlen(i0_filename) NE 0) THEN BEGIN
	widget_control, zstack_spectra_par.i0_retrieve_label, sensitive = 1
ENDIF ELSE BEGIN
	widget_control, zstack_spectra_par.i0_retrieve_label, sensitive = 0
ENDELSE
widget_control, zstack_spectra_par.i0_browse_label, sensitive = 1
widget_control, zstack_spectra_par.i0_scale_factor_label, sensitive = 1
widget_control, zstack_spectra_par.spectra_filename_label, sensitive = 1
IF ( (strlen(spectra_filename_header) NE 0) AND $
		( (n_roi NE 0) OR ((size(i0_roi))(0) NE 0) ) ) THEN BEGIN
	IF ( (n_roi NE 0) AND ((size(i0_roi))(0) NE 0) ) THEN BEGIN
		widget_control, zstack_spectra_par.spectra_dataformat_list_label, $
			set_value = ['Select type of spectra :','Single Beam Data','Percent Transmittance Data','Absorbance Data']
	ENDIF ELSE BEGIN
		widget_control, zstack_spectra_par.spectra_dataformat_list_label, $
			set_value = ['Select type of spectra :','Single Beam Data']
	ENDELSE
	widget_control, zstack_spectra_par.spectra_dataformat_list_label, sensitive = 1
	widget_control, zstack_spectra_par.spectra_filetype_list_label, sensitive = 1
	widget_control, zstack_spectra_par.spectra_fileformat_list_label, sensitive = 1
	dummy = where(save_spectra_type EQ [0,0,0],count)
	IF (count EQ 0) THEN BEGIN
		widget_control, zstack_spectra_par.spectra_save_label, sensitive = 1
	ENDIF ELSE BEGIN
		widget_control, zstack_spectra_par.spectra_save_label, sensitive = 0
	ENDELSE
ENDIF ELSE BEGIN
	widget_control, zstack_spectra_par.spectra_dataformat_list_label, sensitive = 0
	widget_control, zstack_spectra_par.spectra_filetype_list_label, sensitive = 0
	widget_control, zstack_spectra_par.spectra_fileformat_list_label, sensitive = 0
	widget_control, zstack_spectra_par.spectra_save_label, sensitive = 0
ENDELSE
widget_control, zstack_spectra_par.roi_filename_label, sensitive = 1
IF ( ((size(i0_roi))(0) NE 0) OR (n_roi NE 0) ) THEN BEGIN
;	widget_control, zstack_spectra_par.roi_filename_label, sensitive = 1
	IF (strlen(roi_filename) NE 0) THEN BEGIN
		widget_control, zstack_spectra_par.roi_save_label, sensitive = 1
	ENDIF ELSE BEGIN
		widget_control, zstack_spectra_par.roi_save_label, sensitive = 0
	ENDELSE
ENDIF ELSE BEGIN
;	widget_control, zstack_spectra_par.roi_filename_label, sensitive = 0
	widget_control, zstack_spectra_par.roi_save_label, sensitive = 0
ENDELSE
IF (strlen(roi_filename) NE 0) THEN BEGIN
	widget_control, zstack_spectra_par.roi_read_label, sensitive = 1
	widget_control, zstack_spectra_par.roi_browse_label, sensitive = 1
ENDIF ELSE BEGIN
	widget_control, zstack_spectra_par.roi_read_label, sensitive = 0
	widget_control, zstack_spectra_par.roi_browse_label, sensitive = 0
ENDELSE
widget_control, zstack_spectra_par.roi_browse_label, sensitive = 1
widget_control, zstack_spectra_par.filename_display_list_label, sensitive = 1
widget_control, zstack_spectra_par.prev_image_label, sensitive = 1
widget_control, zstack_spectra_par.next_image_label, sensitive = 1
widget_control, zstack_spectra_par.play_movie_label, sensitive = 1
widget_control, zstack_spectra_par.display_parameters_label, sensitive = 1
dummy = where(spectrum NE 0, count)
IF (count NE 0) THEN BEGIN	; if some spectra exist
	widget_control, zstack_spectra_par.plot_parameters_label, sensitive = 1
	widget_control, zstack_spectra_par.spectra_plot_label, sensitive = 1
ENDIF ELSE BEGIN
	widget_control, zstack_spectra_par.plot_parameters_label, sensitive = 0
	widget_control, zstack_spectra_par.spectra_plot_label, sensitive = 0
ENDELSE
widget_control, zstack_spectra_par.profile_label, sensitive = 1
widget_control, zstack_spectra_par.save_label, sensitive = 1
widget_control, zstack_spectra_par.exit_label, sensitive = 1
return
END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_spectra_imgdisp,i_file
;print,'zstack_spectra_imgdisp'

@axis_com
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

on_error,2
; create image ready for display including regions of interest
zstack_analyze_imgprep,i_file,image
zstack_analyze_bytescale, image, byte_image
; Display image, applying zoom factor
wset,zstack_spectra_par.image_win
IF (image_zoom GE 1.0) THEN BEGIN
	IF ((image_zoom) EQ fix(image_zoom)) THEN BEGIN
		tv,rebin(byte_image,n_cols*image_zoom,n_rows*image_zoom,/sample),0,0
	ENDIF ELSE BEGIN
		tv,congrid(byte_image,n_cols*image_zoom,n_rows*image_zoom),0,0
	ENDELSE
ENDIF ELSE BEGIN
	IF ( ((1./image_zoom) EQ fix(1./image_zoom)) AND $
			((image_zoom*n_cols) EQ fix(image_zoom*n_cols)) AND $
			((image_zoom*n_rows) EQ fix(image_zoom*n_rows)) ) THEN BEGIN
		tv,rebin(byte_image,n_cols*image_zoom,n_rows*image_zoom,/sample),0,0
	ENDIF ELSE BEGIN
		tv,congrid(byte_image,n_cols*image_zoom,n_rows*image_zoom),0,0
	ENDELSE
ENDELSE
; Display clipped shifted image in clipped_image_win
; 	if there is a non-zero shift, i.e., alignment was kept
; 	if shift is discarded, i.e., all zero, don't display clipped_win_image
IF (data_shifted EQ 1) THEN BEGIN
	; Clip edges of image stack where no data is present due to alignment shifts
	byte_image = byte_image(clipbox[0]:clipbox[1],clipbox[2]:clipbox[3])
	wset,zstack_spectra_par.clipped_image_win
	IF (image_zoom GE 1.0) THEN BEGIN
		IF ((image_zoom) EQ fix(image_zoom)) THEN BEGIN
			tv,rebin(byte_image,n_clipped_cols*image_zoom,n_clipped_rows*image_zoom,/sample),0,0
		ENDIF ELSE BEGIN
			tv,congrid(byte_image,n_clipped_cols*image_zoom,n_clipped_rows*image_zoom),0,0
		ENDELSE
	ENDIF ELSE BEGIN
		IF ( ((1./image_zoom) EQ fix(1./image_zoom)) AND $
				((image_zoom*n_clipped_cols) EQ fix(image_zoom*n_clipped_cols)) AND $
				((image_zoom*n_clipped_rows) EQ fix(image_zoom*n_clipped_rows)) ) THEN BEGIN
			tv,rebin(byte_image,n_clipped_cols*image_zoom,n_clipped_rows*image_zoom,/sample),0,0
		ENDIF ELSE BEGIN
			tv,congrid(byte_image,n_clipped_cols*image_zoom,n_clipped_rows*image_zoom),0,0
		ENDELSE
	ENDELSE
ENDIF
; Display image description info
widget_control, zstack_spectra_par.filename_display_list_label,$
		set_droplist_select=i_file
return
END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_spectra_plotspectrum, i_file
;print,'zstack_spectra_plotspectrum'

@axis_com
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

on_error,2
IF (strupcase(!d.name) NE 'Z') THEN BEGIN
	wset,zstack_spectra_par.spectra_plot_win
ENDIF ELSE BEGIN
	print,'Z spectra'
ENDELSE
erase
dummy = where(spectrum NE 0,count)	; if no spectra exist, leave window erased
IF (count EQ 0) THEN BEGIN
	return
ENDIF
; Determine range of x-values in plot
ev_list_tmp = ev_list		; save temporarily  ; aph (4-sep-07) CORRECTION
ev_list = float(ev_list)	; CONVERT to real
IF (x_autoscale EQ 1)	 THEN BEGIN	; autoscale x-axis
	!x.range = [min(ev_list),max(ev_list)]
ENDIF ELSE BEGIN
	!x.range = [plot_x_min, plot_x_max]
ENDELSE
x_index = where(ev_list GE !x.range[0] AND ev_list LE !x.range[1], count)
ev_list = ev_list_tmp        ;  RESTORE in case needs to be a string
; Calculate spectra (Single Beam vs Percent Transmittance vs Absorbance)
; Add in offset
; Determine range of y-values in plot
CASE spectrum_display OF	; Single Beam vs Percent Transmittance vs Absorbance
	1 : BEGIN		; plot single beam spectra
		dummy = where(spectrum[0,*] NE 0,count)
		IF (count NE 0) THEN BEGIN	; if i0 spectrum exists
			dummy = where(spectrum[1:14,*] NE 0,count)
			IF (count GT 0) THEN BEGIN	; if both i0 and some i spectra exist
				spectra = spectrum[0:n_roi,*]				; [i0,i1,i2,i3,...]
				IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
					FOR i=1,n_roi DO BEGIN
						spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
					ENDFOR
				ENDIF
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
					!y.range = [min(spectra[0:n_roi,x_index]),max(spectra[0:n_roi,x_index])]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectra,/nodata, $
						xtitle='eV',ytitle='Single Beam Intensity', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				FOR i=0,n_roi DO BEGIN
					oplot,ev_list,spectra[i,*],color=spectra_color_index(i)
				ENDFOR
				IF (spectrum_offset NE 0) THEN BEGIN	; plot offset label
					zlabel = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
					xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
							alignment=0.0,label
				ENDIF
			ENDIF ELSE BEGIN	; if only i0 spectrum exists
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
					!y.range = [min(spectrum[0,x_index]),max(spectrum[0,x_index])]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectrum[0,*],/nodata,xtitle='eV',ytitle='I0', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				oplot,ev_list,spectrum[0,*],color=spectra_color_index(0)
			ENDELSE
		ENDIF ELSE BEGIN	; if i0 spectrum does not exist
			spectra = spectrum[1:n_roi,*]				; [i1,i2,i3,...]
			IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
				FOR i=1,n_roi-1 DO BEGIN
					spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
				ENDFOR
			ENDIF
			IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
				!y.range = [min(spectra[0:n_roi-1,x_index]),max(spectra[0:n_roi-1,x_index])]
				!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
				!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
			ENDIF ELSE BEGIN
				!y.range = [plot_y_min, plot_y_max]
			ENDELSE
			plot,ev_list,spectra,/nodata, $
					xtitle='eV',ytitle='Single Beam Intensity', $
					xtick_get = xticks, ytick_get = yticks, $
					color=plot_axes_color_index, background=plot_bkgd_color_index
			FOR i=0,n_roi-1 DO BEGIN
				oplot,ev_list,spectra[i,*],color=spectra_color_index(i+1)
			ENDFOR
			IF (n_roi GT 1) AND (spectrum_offset NE 0) THEN BEGIN	; plot offset label
					zlabel = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
					xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
							alignment=0.0,label
			ENDIF
		ENDELSE
	END
	2 : BEGIN		; plot percent transmittance spectra
		dummy = where(spectrum[0,*] NE 0,count)
		IF (count NE 0) THEN BEGIN	; if i0 spectrum exists
			dummy = where(spectrum[1:14,*] NE 0,count)
			IF (count GT 0) THEN BEGIN	; if both i0 and some i spectra exist
				spectra = fltarr(n_roi,n_files)
				FOR i=1,n_roi DO BEGIN	; calculate percent transmittance spectra
					spectra[i-1,*] = 100.*spectrum[i,*]/spectrum[0,*]	; [i1/i0,i2/i0,i3/i0,...]
				ENDFOR
				IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
					FOR i=1,n_roi-1 DO BEGIN
						spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
					ENDFOR
				ENDIF
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
					!y.range = [min(spectra[0:n_roi-1,x_index]),max(spectra[0:n_roi-1,x_index])]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectra,/nodata, $
						xtitle='eV',ytitle='Percent Tranmittance', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				FOR i=0,n_roi-1 DO BEGIN
					oplot,ev_list,spectra[i,*],color=spectra_color_index(i+1)
				ENDFOR
				IF (spectrum_offset NE 0) THEN BEGIN	; plot offset label
					zlabel = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
					xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
							alignment=0.0,label
				ENDIF
			ENDIF ELSE BEGIN	; if only i0 spectrum exists
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
					!y.range = [min(spectrum[0,x_index]),max(spectrum[0,x_index])]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectrum[0,*],/nodata,xtitle='eV',ytitle='I0', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				oplot,ev_list,spectrum[0,*],color=spectra_color_index(0)
			ENDELSE
		ENDIF ELSE BEGIN	; if i0 spectrum does not exist
			spectra = spectrum[1:n_roi,*]				; [i1,i2,i3,...]
			IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
				FOR i=1,n_roi-1 DO BEGIN
					spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
				ENDFOR
			ENDIF
			IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
				!y.range = [min(spectra[0:n_roi-1,x_index]),max(spectra[0:n_roi-1,x_index])]
				!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
				!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
			ENDIF ELSE BEGIN
				!y.range = [plot_y_min, plot_y_max]
			ENDELSE
			plot,ev_list,spectra,/nodata, $
					xtitle='eV',ytitle='Single Beam Intensity', $
					xtick_get = xticks, ytick_get = yticks, $
					color=plot_axes_color_index, background=plot_bkgd_color_index
			FOR i=0,n_roi-1 DO BEGIN
				oplot,ev_list,spectra[i,*],color=spectra_color_index(i+1)
			ENDFOR
			IF (n_roi GT 1) AND (spectrum_offset NE 0) THEN BEGIN	; plot offset label
					zlabel = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
					xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
							alignment=0.0,label
			ENDIF
		ENDELSE
	END
	3 : BEGIN		; plot absorbance spectra
		dummy = where(spectrum[0,*] NE 0,count)
		IF (count NE 0) THEN BEGIN	; if i0 spectrum exists
			dummy = where(spectrum[1:14,*] NE 0,count)
			IF (count GT 0) THEN BEGIN	; if both i0 and some i spectra exist
				spectra = fltarr(n_roi,n_files)
				FOR i=1,n_roi DO BEGIN	; calculate absorbance spectra ; (03-Aug-09 aph) changed from alog10 ==> alog
					spectra[i-1,*] = -alog(spectrum[i,*]/spectrum[0,*])	; [i1/i0,i2/i0,i3/i0,...]
				ENDFOR
				IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
					FOR i=1,n_roi-1 DO BEGIN
						spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
					ENDFOR
				ENDIF
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
					!y.range = [min(spectra[0:n_roi-1,x_index]),max(spectra[0:n_roi-1,x_index])]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectra,/nodata, $
						xtitle='eV',ytitle='Absorbance', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				FOR i=0,n_roi-1 DO BEGIN
					oplot,ev_list,spectra[i,*],color=spectra_color_index(i+1)
				ENDFOR
				IF (spectrum_offset NE 0) THEN BEGIN	; plot offset label
					zlabel = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
					xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
							alignment=0.0,label
				ENDIF
			ENDIF ELSE BEGIN	; if only i0 spectrum exists
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
					!y.range = [min(spectrum[0,*]),max(spectrum[0,*])]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectrum[0,*],/nodata,xtitle='eV',ytitle='I0', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				oplot,ev_list,spectrum[0,*],color=spectra_color_index(0)
			ENDELSE
		ENDIF ELSE BEGIN	; if i0 spectrum does not exist
			spectra = spectrum[1:n_roi,*]				; [i1,i2,i3,...]
			IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
				FOR i=1,n_roi-1 DO BEGIN
					spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
				ENDFOR
			ENDIF
			IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
				!y.range = [min(spectra[0:n_roi-1,x_index]),max(spectra[0:n_roi-1,x_index])]
				!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
				!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
			ENDIF ELSE BEGIN
				!y.range = [plot_y_min, plot_y_max]
			ENDELSE
			plot,ev_list,spectra,/nodata, $
					xtitle='eV',ytitle='Single Beam Intensity', $
					xtick_get = xticks, ytick_get = yticks, $
					color=plot_axes_color_index, background=plot_bkgd_color_index
			FOR i=0,n_roi-1 DO BEGIN
				oplot,ev_list,spectra[i,*],color=spectra_color_index(i+1)
			ENDFOR
			IF (n_roi GT 1) AND (spectrum_offset NE 0) THEN BEGIN	; plot offset label
					zlabel = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
					xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
							alignment=0.0,label
			ENDIF
		ENDELSE
	END
	ELSE : BEGIN
		print,'spectrum_display not specified'
	END
ENDCASE
!x.range = [min(xticks),max(xticks)]
!y.range = [min(yticks),max(yticks)]
IF (ev_list(i_file) GT !x.range[0]) AND (ev_list(i_file) LT !x.range[1]) THEN BEGIN
	plots,ev_list(i_file)+[0.,0.],!y.crange,color=spectra_cursor_color_index
ENDIF
return
END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_spectra_makespectrum
;print,'zstack_spectra_makespectrum'

@axis_com
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

on_error,2
;  spectrum has been previously set as a zero matrix of size [15,n_files]
;	with [0,*] reserved for i0 spectrum and [1:14,*] reserved for i spectra
i0_svec = size(i0_roi)
IF ((i0_svec(0) NE 0) AND (i0_roi[0] NE -1))THEN BEGIN
	i0_signal = fltarr(n_files)
	FOR i_ev=0,(n_files-1) DO BEGIN
		khz = image_stack(*,*,i_ev)
		i0_signal(i_ev) = total(khz(i0_roi))/float(n_elements(i0_roi))
	ENDFOR
	spectrum[0,*] = i0_signal
ENDIF
i_svec = size(i_roi)
IF (i_svec(0) NE 0) THEN BEGIN
	i_signal = fltarr(n_files)
	FOR i_ev=0,(n_files-1) DO BEGIN
		khz = image_stack(*,*,i_ev)
			; select most recent roi
		roi = i_roi[roi_index[n_roi-1]:roi_index[n_roi]-1]
		i_signal(i_ev) = total(khz(roi))/float(n_elements(roi))
	ENDFOR
	spectrum[n_roi,*] = i_signal
ENDIF
return
END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_spectra_save_spec ;zzzzz
;	,sb=sb,ab=ab,single=single,multiple=multiple,xas=xas,spc=spc,txt=txt,gif=gif
;print,'zstack_spectra_save_spec'

@axis_com
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

on_error,2
; default: save_spectra_type = [2,2,2]
	;	[1,*,*]	single beam
	;	[2,*,*]	percent tranmittance: 100. * (I/I0)
	;	[3,*,*]	absorbance: -log10(I/I0)
	;	[*,1,*]	files with one spectrum each
	;	[*,2,*]	files with multiple spectra
	;	[*,*,1]	*.xas
	;	[*,*,2]	*.spc
	;	[*,*,3]	*.spc
	;	[*,*,4]	*.gif
CASE save_spectra_type[0] OF	; Single Beam vs Percent Transmittance vs Absorbance
	1 : BEGIN		; create single beam spectra
		spectra = spectrum[0:n_roi,*]				; [i0,i1,i2,i3,...]
	END
	2 : BEGIN		; create percent transmittance spectra
		spectra = fltarr(n_roi,n_files)
		FOR i=1,n_roi DO BEGIN
			spectra[i-1,*] = 100.*spectrum[i,*]/spectrum[0,*]	; [i1/i0,i2/i0,i3/i0,...]
		ENDFOR
	END
	3 : BEGIN		; create absorbance spectra
		spectra = fltarr(n_roi,n_files)
		FOR i=1,n_roi DO BEGIN
			spectra[i-1,*] = -alog(spectrum[i,*]/spectrum[0,*])	; [i1/i0,i2/i0,i3/i0,...]
		ENDFOR
	END
	ELSE : BEGIN
		print,'save_spectra_type[0] not specified'
	END
ENDCASE
spc_header = ''
IF (strlen(strtrim(list_filename,2)) NE 0) THEN BEGIN
	spc_header = ' : '+list_filename
ENDIF
IF (strlen(strtrim(shift_filename,2)) NE 0) THEN BEGIN
	spc_header = spc_header+' : '+shift_filename
ENDIF
spc_header = spc_header+' !'
CASE save_spectra_type[2] OF	; *.xas vs. *.spc vs *.txt vs *.gif
	1 : BEGIN		; save spectra as *.XAS files
				; ALWAYS save data in *.xas format as one spectrum per file
				; If Multiple is chosen, write multiple files with incremental names
		CASE save_spectra_type[0] OF	; Single Beam vs Percent Transmittance vs Absorbance
			1 : BEGIN		; Single Beam - use all data in spectra variable
				dummy = where(spectra[0,*] NE 0,count)
				IF (count NE 0) THEN BEGIN	; if i0 spectrum exists
					num = '00'
					write_xas,spectra_filename_header+'_'+num+'.xas', ev_list,spectra[0,*],$
						comments='Single Beam Spectrum - Stack list ROI '+spectra_filename_header+'_'+num+'.roi'
					print,'Wrote spectrum file "'+data_directory+spectra_filename_header+'_'+num+'.xas"'
				ENDIF
				IF (n_roi NE 0) THEN BEGIN
					FOR i=1,n_roi DO BEGIN
						IF (strlen(strtrim(i,2)) EQ 1) THEN BEGIN
							num = '0'+strtrim(i,2)
						ENDIF ELSE BEGIN
							num = strtrim(i,2)
						ENDELSE
						write_xas,spectra_filename_header+'_'+num+'.xas', ev_list,spectra[i,*],$
							comments='Single Beam Spectrum - Stack list ROI '+spectra_filename_header+'_'+num+'.roi'
						print,'Wrote spectrum file "'+data_directory+spectra_filename_header+'_'+num+'.xas"'
					ENDFOR
				ENDIF
			END
			2 : BEGIN		; Transmittance - use all data in spectra variable except first column
				FOR i=0,n_roi-1 DO BEGIN
					IF (strlen(strtrim(i+1,2)) EQ 1) THEN BEGIN
						num = '0'+strtrim(i+1,2)
					ENDIF ELSE BEGIN
						num = strtrim(i+1,2)
					ENDELSE
					write_xas,spectra_filename_header+'_'+num+'.xas', ev_list,spectra[i,*],$
						comments='Transmission Spectrum - Stack list ROI '+spectra_filename_header+'_'+num+'.roi'
					print,'Wrote spectrum file "'+data_directory+spectra_filename_header+'_'+num+'.xas"'
				ENDFOR
			END
			3 : BEGIN		; Absorbance - use all data in spectra variable except first column
				FOR i=0,n_roi-1 DO BEGIN
					IF (strlen(strtrim(i+1,2)) EQ 1) THEN BEGIN
						num = '0'+strtrim(i+1,2)
					ENDIF ELSE BEGIN
						num = strtrim(i+1,2)
					ENDELSE
					write_xas,spectra_filename_header+'_'+num+'.xas', ev_list,spectra[i,*],$
						comments='Absorbance Spectrum - Stack list ROI '+spectra_filename_header+'_'+num+'.roi'
					print,'Wrote spectrum file "'+data_directory+spectra_filename_header+'_'+num+'.xas"'
				ENDFOR
			END
			ELSE : BEGIN
				print,'save_spectra_type[0] not specified'
			END
		ENDCASE
	END
	2 : BEGIN		; save spectra as *.SPC files
		CASE save_spectra_type[1] OF	; Files with Single or Multiple spectra
			1 : BEGIN		; save each spectrum as separate *.SPC file
				CASE save_spectra_type[0] OF	; Single Beam vs Percent Transmittance vs Absorbance
					1 : BEGIN		; Single Beam - use all data in spectra variable
						spc_header = '! Single Beam Spectra'+spc_header
;						dummy = fltarr(2,n_files)

						dummy = where(spectra[0,*] NE 0,count)
						IF (count NE 0) THEN BEGIN	; if i0 spectrum exists
							dummy = fltarr(2,n_files+1)
							dummy[0,1:n_files] = transpose(ev_list)
							dummy[1,1:n_files] = spectra[0,*]
							dummy[0,0] = -1
							num = '00'
							dummy[1,0] = num
							openw,lun,spectra_filename_header+'_'+num+'.spc',/get_lun
							printf,lun,spc_header
							printf,lun,dummy,format='(f12.5,",",f12.5)'
							close,lun
							free_lun,lun
							print,'Wrote spectrum file "'+data_directory+spectra_filename_header+'_'+num+'.spc"'
						ENDIF
						IF (n_roi NE 0) THEN BEGIN
							dummy = fltarr(2,n_files+1)
							FOR i = 1,n_roi DO BEGIN
								dummy[0,1:n_files] = transpose(ev_list)
								dummy[1,1:n_files] = spectra[i,*]
								dummy[0,0] = -1
								IF (strlen(strtrim(i,2)) EQ 1) THEN BEGIN
									num = '0'+strtrim(i,2)
								ENDIF ELSE BEGIN
									num = strtrim(i,2)
								ENDELSE
								dummy[1,0] = num
								openw,lun,spectra_filename_header+'_'+num+'.spc',/get_lun
								printf,lun,spc_header
								printf,lun,dummy,format='(f12.5,",",f12.5)'
								close,lun
								free_lun,lun
								print,'Wrote spectrum file "'+data_directory+spectra_filename_header+'_'+num+'.spc"'
							ENDFOR
						ENDIF
					END
					2 : BEGIN		; Transmittance - use all data in spectra variable except first column
						spc_header = '! Transmission Spectra'+spc_header
						dummy = fltarr(2,n_files+1)
						FOR i = 0,(n_roi-1) DO BEGIN
							dummy[0,1:n_files] = transpose(ev_list)
							dummy[1,1:n_files] = spectra[i,*]
							dummy[0,0] = -1
							IF (strlen(strtrim(i+1,2)) EQ 1) THEN BEGIN
								num = '0'+strtrim(i+1,2)
							ENDIF ELSE BEGIN
								num = strtrim(i+1,2)
							ENDELSE
							dummy[1,0] = num
							openw,lun,spectra_filename_header+'_'+num+'.spc',/get_lun
							printf,lun,spc_header
							printf,lun,dummy,format='(f12.5,",",f12.5)'
							close,lun
							free_lun,lun
							print,'Wrote spectrum file "'+data_directory+spectra_filename_header+'_'+num+'.spc"'
						ENDFOR
					END
					3 : BEGIN		; Absorbance - use all data in spectra variable except first column
						spc_header = '! Absorbance Spectra'+spc_header
						dummy = fltarr(2,n_files+1)
						FOR i = 0,(n_roi-1) DO BEGIN
							dummy[0,1:n_files] = transpose(ev_list)
							dummy[1,1:n_files] = spectra[i,*]
							dummy[0,0] = -1
							IF (strlen(strtrim(i+1,2)) EQ 1) THEN BEGIN
								num = '0'+strtrim(i+1,2)
							ENDIF ELSE BEGIN
								num = strtrim(i+1,2)
							ENDELSE
							dummy[1,0] = num
							openw,lun,spectra_filename_header+'_'+num+'.spc',/get_lun
							printf,lun,spc_header
							printf,lun,dummy,format='(f12.5,",",f12.5)'
							close,lun
							free_lun,lun
							print,'Wrote spectrum file "'+data_directory+spectra_filename_header+'_'+num+'.spc"'
						ENDFOR
					END
				ENDCASE
			END
			2 : BEGIN		; save multiple spectra in one *.SPC file
				CASE save_spectra_type[0] OF	; Single Beam vs Absorbance
					1 : BEGIN		; Single Beam - use all data in spectra variable
						spc_header = '! Multiple Single Beam Spectra'+spc_header
						dummy = fltarr(n_roi+2,n_files+1)
						dummy[1:n_roi+1,1:n_files] = spectra
						dummy[0,1:n_files] = transpose(ev_list)
						dummy[0,0] = -1
						FOR i=1,n_roi+1 DO BEGIN
							dummy[i,0] = i
						ENDFOR
					END
					2 : BEGIN		; Transmittance - use all data in spectra variable except first column
						spc_header = '! Multiple Transmission Spectra'+spc_header
						dummy = fltarr(n_roi+1,n_files+1)
						dummy[1:n_roi,1:n_files] = spectra
						dummy[0,1:n_files] = transpose(ev_list)
						dummy[0,0] = -1
						FOR i=1,n_roi DO BEGIN
							dummy[i,0] = i
						ENDFOR
					END
					3 : BEGIN		; Absorbance - use all data in spectra variable except first column
						spc_header = '! Multiple Absorbance Spectra'+spc_header
						dummy = fltarr(n_roi+1,n_files+1)
						dummy[1:n_roi,1:n_files] = spectra
						dummy[0,1:n_files] = transpose(ev_list)
						dummy[0,0] = -1
						FOR i=1,n_roi DO BEGIN
							dummy[i,0] = i
						ENDFOR
					END
				ENDCASE
				openw,lun,spectra_filename_header+'.spc',/get_lun
				printf,lun,spc_header
				printf,lun,dummy,format='('+string(n_roi+1)+'(f12.5,","),f12.5)'
				close,lun
				free_lun,lun
				print,'Wrote spectrum file "'+data_directory+spectra_filename_header+'.spc"'
			END
		ENDCASE
	END
	3 : BEGIN		; save spectra as *.TXT files
;dialog for file header
text = get_text(prompt='Label')
		CASE save_spectra_type[1] OF	; Files with Single or Multiple spectra
			1 : BEGIN		; save each spectrum as separate *.TXT file
				CASE save_spectra_type[0] OF	; Single Beam vs Percent Transmittance vs Absorbance
					1 : BEGIN		; Single Beam - use all data in spectra variable
						spc_header = '! Single Beam Spectra'+spc_header
;						dummy = fltarr(2,n_files)

						dummy = where(spectra[0,*] NE 0,count)
						IF (count NE 0) THEN BEGIN	; if i0 spectrum exists
							dummy = fltarr(2,n_files+1)
							dummy[0,1:n_files] = transpose(ev_list)
							dummy[1,1:n_files] = spectra[0,*]
							dummy[0,0] = -1
							num = '00'
							dummy[1,0] = num
							openw,lun,spectra_filename_header+'_'+num+'.txt',/get_lun
							printf,lun,'% '
							printf,lun,spc_header
							printf,lun,'% '+string(n_elements(ev_list))
							printf,lun,dummy,format='(f12.5,"  ",f12.5)'
							close,lun
							free_lun,lun
;							temp = spc_save(dummy, file=spectra_filename_header+'_'+num+'.txt')
							print,'Wrote spectrum file "'+data_directory+spectra_filename_header+'_'+num+'.txt"'
						ENDIF
						IF (n_roi NE 0) THEN BEGIN
							dummy = fltarr(2,n_files+1)
							FOR i = 1,n_roi DO BEGIN
								dummy[0,1:n_files] = transpose(ev_list)
								dummy[1,1:n_files] = spectra[i,*]
								dummy[0,0] = -1
								IF (strlen(strtrim(i,2)) EQ 1) THEN BEGIN
									num = '0'+strtrim(i,2)
								ENDIF ELSE BEGIN
									num = strtrim(i,2)
								ENDELSE
								dummy[1,0] = num
								openw,lun,spectra_filename_header+'_'+num+'.txt',/get_lun
							printf,lun,'% '
							printf,lun,spc_header
							printf,lun,'% '+string(n_elements(ev_list))
							printf,lun,dummy,format='(f12.5,"  ",f12.5)'
								close,lun
								free_lun,lun
;								temp = spc_save(dummy, file=spectra_filename_header+'_'+num+'.txt')
								print,'Wrote spectrum file "'+data_directory+spectra_filename_header+'_'+num+'.txt"'
							ENDFOR
						ENDIF
					END
					2 : BEGIN		; Transmittance - use all data in spectra variable except first column
						spc_header = '! Transmission Spectra'+spc_header
						dummy = fltarr(2,n_files+1)
						FOR i = 0,(n_roi-1) DO BEGIN
							dummy[0,1:n_files] = transpose(ev_list)
							dummy[1,1:n_files] = spectra[i,*]
							dummy[0,0] = -1
							IF (strlen(strtrim(i+1,2)) EQ 1) THEN BEGIN
								num = '0'+strtrim(i+1,2)
							ENDIF ELSE BEGIN
								num = strtrim(i+1,2)
							ENDELSE
							dummy[1,0] = num
							openw,lun,spectra_filename_header+'_'+num+'.txt',/get_lun
							printf,lun,'% '
							printf,lun,spc_header
							printf,lun,'% '+string(n_elements(ev_list))
							printf,lun,dummy,format='(f12.5,"  ",f12.5)'
							close,lun
							free_lun,lun
;							temp = spc_save(dummy, file=spectra_filename_header+'_'+num+'.txt')
							print,'Wrote spectrum file "'+data_directory+spectra_filename_header+'_'+num+'.txt"'
						ENDFOR
					END
					3 : BEGIN		; Absorbance - use all data in spectra variable except first column
						spc_header = '! Absorbance Spectra'+spc_header
						dummy = fltarr(2,n_files+1)
						FOR i = 0,(n_roi-1) DO BEGIN
							dummy[0,1:n_files] = transpose(ev_list)
							dummy[1,1:n_files] = spectra[i,*]
							dummy[0,0] = -1
							IF (strlen(strtrim(i+1,2)) EQ 1) THEN BEGIN
								num = '0'+strtrim(i+1,2)
							ENDIF ELSE BEGIN
								num = strtrim(i+1,2)
							ENDELSE
							dummy[1,0] = num
							openw,lun,spectra_filename_header+'_'+num+'.txt',/get_lun
							printf,lun,'% '
							printf,lun,spc_header
							printf,lun,'% '+string(n_elements(ev_list))
							printf,lun,dummy,format='(f12.5,"  ",f12.5)'
							close,lun
							free_lun,lun
;							temp = spc_save(dummy, file=spectra_filename_header+'_'+num+'.txt')
							print,'Wrote spectrum file "'+data_directory+spectra_filename_header+'_'+num+'.spc"'
						ENDFOR
					END
				ENDCASE
			END
			2 : BEGIN		; save multiple spectra in one *.TXT file
				CASE save_spectra_type[0] OF	; Single Beam vs Absorbance
					1 : BEGIN		; Single Beam - use all data in spectra variable
						spc_header = '! Multiple Single Beam Spectra'+spc_header
						dummy = fltarr(n_roi+2,n_files+1)
						dummy[1:n_roi+1,1:n_files] = spectra
						dummy[0,1:n_files] = transpose(ev_list)
						dummy[0,0] = -1
						FOR i=1,n_roi+1 DO BEGIN
							dummy[i,0] = i
						ENDFOR
					END
					2 : BEGIN		; Transmittance - use all data in spectra variable except first column
						spc_header = '! Multiple Transmission Spectra'+spc_header
						dummy = fltarr(n_roi+1,n_files+1)
						dummy[1:n_roi,1:n_files] = spectra
						dummy[0,1:n_files] = transpose(ev_list)
						dummy[0,0] = -1
						FOR i=1,n_roi DO BEGIN
							dummy[i,0] = i
						ENDFOR
					END
					3 : BEGIN		; Absorbance - use all data in spectra variable except first column
						spc_header = '! Multiple Absorbance Spectra'+spc_header
						dummy = fltarr(n_roi+1,n_files+1)
						dummy[1:n_roi,1:n_files] = spectra
						dummy[0,1:n_files] = transpose(ev_list)
						dummy[0,0] = -1
						FOR i=1,n_roi DO BEGIN
							dummy[i,0] = i
						ENDFOR
					END
				ENDCASE
				openw,lun,spectra_filename_header+'.txt',/get_lun
				printf,lun,spc_header
				printf,lun,dummy,format='('+string(n_roi+1)+'(f12.5,","),f12.5)'
				close,lun
				free_lun,lun
				print,'Wrote spectrum file "'+data_directory+spectra_filename_header+'.spc"'
			END
		ENDCASE
	END
	4 : BEGIN		; save spectra as a *.GIF file
		CASE save_spectra_type[1] OF
			1 : BEGIN		; save each spectra as separate *S.gif files
				print,'Saving each spectra as a separate *.gif file is not yet implemented'
			END
			2 : BEGIN		; save multiple spectra in one *S.gif file
				gif_filename = spectra_filename_header+'.gif'
				old_plot=!d.name
				set_plot,'z',/copy
				device,set_resolution=[400,300]

				CASE save_spectra_type[0] OF	; Single Beam vs Percent Transmittance vs Absorbance
					1 : BEGIN		; plot single beam spectra
						dummy = where(spectrum[0,*] NE 0,count)
						IF (count NE 0) THEN BEGIN	; if i0 spectrum exists
							dummy = where(spectrum[1:14,*] NE 0,count)
							IF (count EQ 0) THEN BEGIN	; if only i0 spectrum exists
								plot,ev_list,spectrum[0,*],/nodata,xtitle='eV',ytitle='I0', $
										color=plot_axes_color_index, background=plot_bkgd_color_index
								oplot,ev_list,spectrum[0,*],color=spectra_color_index(0)
							ENDIF ELSE BEGIN	; if both i0 and some i spectra exist
								spectra = spectrum[0:n_roi,*]				; [i0,i1,i2,i3,...]
								IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
									FOR i=1,n_roi DO BEGIN
										spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
									ENDFOR
								ENDIF
								y_min = min(spectra[0:n_roi,*],max=y_max)
								y_min = (1.05 * y_max) < (0.95 * y_min)
								y_max = (1.05 * y_max) > (0.95 * y_min)
								plot,ev_list,spectra,/nodata, $
										xtitle='eV',ytitle='Single Beam Intensity', $
										color=plot_axes_color_index, background=plot_bkgd_color_index, $
										yrange=[y_min,y_max]
								FOR i=0,n_roi DO BEGIN
									oplot,ev_list,spectra[i,*],color=spectra_color_index(i)
								ENDFOR
								IF (spectrum_offset NE 0) THEN BEGIN	; plot offset label
									zlabel = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
									xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
											alignment=0.0,label
								ENDIF
							ENDELSE
						ENDIF ELSE BEGIN	; if i0 spectrum does not exist
							spectra = spectrum[1:n_roi,*]				; [i1,i2,i3,...]
							IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
								FOR i=1,n_roi-1 DO BEGIN
									spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
								ENDFOR
							ENDIF
							y_min = min(spectra[0:n_roi-1,*],max=y_max)
							y_min = (1.05 * y_max) < (0.95 * y_min)
							y_max = (1.05 * y_max) > (0.95 * y_min)
							plot,ev_list,spectra,/nodata, $
									xtitle='eV',ytitle='Single Beam Intensity', $
									color=plot_axes_color_index, background=plot_bkgd_color_index, $
									yrange=[y_min,y_max]
							FOR i=0,n_roi-1 DO BEGIN
								oplot,ev_list,spectra[i,*],color=spectra_color_index(i+1)
							ENDFOR
							IF (n_roi GT 1) AND (spectrum_offset NE 0) THEN BEGIN	; plot offset label
									zlabel = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
									xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
											alignment=0.0,label
							ENDIF
						ENDELSE
					END

					2 : BEGIN		; plot percent transmittance spectra
						dummy = where(spectrum[0,*] NE 0,count)
						IF (count NE 0) THEN BEGIN	; if i0 spectrum exists
							dummy = where(spectrum[1:14,*] NE 0,count)
							IF (count EQ 0) THEN BEGIN	; if only i0 spectrum exists
								plot,ev_list,spectrum[0,*],/nodata,xtitle='eV',ytitle='I0', $
										color=plot_axes_color_index, background=plot_bkgd_color_index
								oplot,ev_list,spectrum[0,*],color=spectra_color_index(0)
							ENDIF ELSE BEGIN	; if both i0 and some i spectra exist
								spectra = fltarr(n_roi,n_files)
								FOR i=1,n_roi DO BEGIN	; calculate percent transmittance spectra
									spectra[i-1,*] = 100.*spectrum[i,*]/spectrum[0,*]	; [i1/i0,i2/i0,i3/i0,...]
								ENDFOR
								IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
									FOR i=1,n_roi-1 DO BEGIN
										spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
									ENDFOR
								ENDIF
								y_min = min(spectra[0:n_roi-1,*],max=y_max)
								y_min = (1.05 * y_max) < (0.95 * y_min)
								y_max = (1.05 * y_max) > (0.95 * y_min)
								plot,ev_list,spectra,/nodata, $
										xtitle='eV',ytitle='Percent Tranmittance', $
										color=plot_axes_color_index, background=plot_bkgd_color_index, $
										yrange=[y_min,y_max]
								FOR i=0,n_roi-1 DO BEGIN
									oplot,ev_list,spectra[i,*],color=spectra_color_index(i+1)
								ENDFOR
								IF (spectrum_offset NE 0) THEN BEGIN	; plot offset label
									zlabel = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
									xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
											alignment=0.0,label
								ENDIF
							ENDELSE
						ENDIF ELSE BEGIN	; if i0 spectrum does not exist
							spectra = spectrum[1:n_roi,*]				; [i1,i2,i3,...]
							IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
								FOR i=1,n_roi-1 DO BEGIN
									spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
								ENDFOR
							ENDIF
							y_min = min(spectra[0:n_roi-1,*],max=y_max)
							y_min = (1.05 * y_max) < (0.95 * y_min)
							y_max = (1.05 * y_max) > (0.95 * y_min)
							plot,ev_list,spectra,/nodata, $
									xtitle='eV',ytitle='Single Beam Intensity', $
									color=plot_axes_color_index, background=plot_bkgd_color_index, $
									yrange=[y_min,y_max]
							FOR i=0,n_roi-1 DO BEGIN
								oplot,ev_list,spectra[i,*],color=spectra_color_index(i+1)
							ENDFOR
							IF (n_roi GT 1) AND (spectrum_offset NE 0) THEN BEGIN	; plot offset label
									zlabel = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
									xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
											alignment=0.0,label
							ENDIF
						ENDELSE
					END

					3 : BEGIN		; plot absorbance spectra
						dummy = where(spectrum[0,*] NE 0,count)
						IF (count NE 0) THEN BEGIN	; if i0 spectrum exists
							dummy = where(spectrum[1:14,*] NE 0,count)
							IF (count EQ 0) THEN BEGIN	; if only i0 spectrum exists
								plot,ev_list,spectrum[0,*],/nodata,xtitle='eV',ytitle='I0', $
										color=plot_axes_color_index, background=plot_bkgd_color_index
								oplot,ev_list,spectrum[0,*],color=spectra_color_index(0)
							ENDIF ELSE BEGIN	; if both i0 and some i spectra exist
								spectra = fltarr(n_roi,n_files)
								FOR i=1,n_roi DO BEGIN	; calculate absorbance spectra
									spectra[i-1,*] = -alog(spectrum[i,*]/spectrum[0,*])	; [i1/i0,i2/i0,i3/i0,...]
								ENDFOR
								IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
									FOR i=1,n_roi-1 DO BEGIN
										spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
									ENDFOR
								ENDIF
								y_min = min(spectra[0:n_roi-1,*],max=y_max)
								y_min = (1.05 * y_max) < (0.95 * y_min)
								y_max = (1.05 * y_max) > (0.95 * y_min)
								plot,ev_list,spectra,/nodata, $
										xtitle='eV',ytitle='Absorbance', $
										color=plot_axes_color_index, background=plot_bkgd_color_index, $
										yrange=[y_min,y_max]
								FOR i=0,n_roi-1 DO BEGIN
									oplot,ev_list,spectra[i,*],color=spectra_color_index(i+1)
								ENDFOR
								IF (spectrum_offset NE 0) THEN BEGIN	; plot offset label
									zlabel = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
									xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
											alignment=0.0,label
								ENDIF
							ENDELSE
						ENDIF ELSE BEGIN
							spectra = spectrum[1:n_roi,*]				; [i1,i2,i3,...]
							IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
								FOR i=1,n_roi-1 DO BEGIN
									spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
								ENDFOR
							ENDIF
							y_min = min(spectra[0:n_roi-1,*],max=y_max)
							y_min = (1.05 * y_max) < (0.95 * y_min)
							y_max = (1.05 * y_max) > (0.95 * y_min)
							plot,ev_list,spectra,/nodata, $
									xtitle='eV',ytitle='Single Beam Intensity', $
									color=plot_axes_color_index, background=plot_bkgd_color_index, $
									yrange=[y_min,y_max]
							FOR i=0,n_roi-1 DO BEGIN
								oplot,ev_list,spectra[i,*],color=spectra_color_index(i+1)
							ENDFOR
							IF (n_roi GT 1) AND (spectrum_offset NE 0) THEN BEGIN	; plot offset label
									zlabel = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
									xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
											alignment=0.0,label
							ENDIF
						ENDELSE
					END
					ELSE : BEGIN
						print,'save_spectra_type[0] not specified'
					END
				ENDCASE

				image2D = tvrd()
				tvlct,r,g,b,/get
				write_gif,spectra_filename_header+'S.gif',image2D,r,g,b
				print,'Wrote spectrum file "'+data_directory+spectra_filename_header+'S.gif"'
				set_plot,old_plot
			END
		ENDCASE
	END
ENDCASE
return
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_spectra_save_roi
;print,'zstack_spectra_save_roi'

@axis_com
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

on_error,2
length_i0_roi = n_elements(i0_roi)
IF (n_elements(roi_index) GT 1) THEN BEGIN
	length_i_roi = max( roi_index(1:n_elements(roi_index)-1) - roi_index(0:n_elements(roi_index)-2))
ENDIF ELSE BEGIN
	length_i_roi = 0
ENDELSE
; create roi array with io_roi in first column, followed by successive i_roi
;	unused indices are set to -1
roi = intarr(1 + n_roi,1 + max([length_i_roi, length_i0_roi]))
roi[*,*] = -1
roi[0,0] = 0
IF ((size(i0_roi))(0) NE 0) THEN BEGIN
	roi[0,1:length_i0_roi] = i0_roi
ENDIF ELSE BEGIN
	IF (i0_roi NE 0) THEN roi[0,1] = i0_roi
ENDELSE
FOR i=1,n_roi DO BEGIN
	roi[i,0] = i
	dummy = i_roi[roi_index(i-1):(roi_index(i)-1)]		; extract ith set of roi indices
	roi[i,1:n_elements(dummy)] = dummy
ENDFOR
; append length numbers to front of roi array
dummy = intarr(1,1 + max([length_i_roi, length_i0_roi]))
dummy[0,0] = -1
FOR i=1,max([length_i_roi, length_i0_roi]) DO BEGIN
	dummy[0,i] = i
ENDFOR
roi = [dummy,temporary(roi)]
;	output roi indices in columns corresponding to i0, i1, i2, etc.
;	unused indices are set to -1
roi_header = '! Regions of Interest for Spectra'
IF (strlen(strtrim(list_filename,2)) NE 0) THEN BEGIN
	roi_header = roi_header+' : '+list_filename
ENDIF
IF (strlen(strtrim(shift_filename,2)) NE 0) THEN BEGIN
	roi_header = roi_header+' : '+shift_filename
ENDIF
roi_header = roi_header+' !'
openw,lun,roi_filename,/get_lun
printf,lun,roi_header
printf,lun,'PLOTIT('+strtrim(string(max([length_i_roi, length_i0_roi])),2)
printf,lun,roi,format='('+string(n_roi+1)+'(i7,","),i7)'
close,lun
free_lun,lun
print,'Wrote ROI to file "'+data_directory+roi_filename+'"'
return
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_spectra_read_roi,filename
;print,'zstack_spectra_read_roi'

@axis_com
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

on_error,2
filelist = findfile(filename)
svec = size(filelist)
IF (svec(0) EQ 0) THEN BEGIN
    print,'Could not open file "'+filename+'"'
    return
ENDIF
spectrum = fltarr(15,n_files)
zstack_read_mapper,roi_filename, dummy, saved_roi
svec = size(saved_roi)
IF (svec(0) EQ 1) THEN BEGIN
	saved_roi = transpose(saved_roi)
ENDIF
dummy = where(saved_roi[0,*] NE -1,count)
IF (count GT 1) THEN BEGIN
	i0_roi = reform(fix(saved_roi[0,1:count-1]))
	zstack_spectra_makespectrum
ENDIF ELSE BEGIN
	i0_roi = 0
ENDELSE
IF (svec(0) EQ 2) THEN BEGIN
	FOR i=1,(svec(1)-1) DO BEGIN
		IF ( i EQ 1) THEN BEGIN
			dummy = where(saved_roi[i,*] NE -1,count)
			i_roi = reform(fix(saved_roi[i,1:count-1]))
			roi_index = [0,n_elements(i_roi)]
			n_roi = 1
			zstack_spectra_makespectrum
		ENDIF ELSE BEGIN
			dummy = where(saved_roi[i,*] NE -1,count)
			i_roi = [i_roi,reform(fix(saved_roi[i,1:count-1]))]
			roi_index = [roi_index,n_elements(i_roi)]
			n_roi = n_roi + 1
			zstack_spectra_makespectrum
		ENDELSE
	ENDFOR
ENDIF ELSE BEGIN
	i_roi = 0
	n_roi = 0
	roi_index = 0
ENDELSE
return
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_spectra_roi
;print,'zstack_spectra_roi'

@axis_com
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

on_error,2
pixarea = abs(x_stop-x_start)*abs(y_stop-y_start) / $
  (float(n_cols-1)*float(n_rows-1))
IF (image_zoom EQ fix(image_zoom)) THEN BEGIN	; if zoom is an integer value
	roi = cw_defroi(zstack_spectra_par.image_label,zoom=[image_zoom,image_zoom])
		; need zoom parameter in case zoom NE 1, CGZ
ENDIF ELSE BEGIN	; if zoom is not an integer value
	; Must redisplay image so that CW_DEFROI thinks it is displayed with integral zoom factor
	; Starting with nonintegral_zoom_data = byte_image, it is scaled to fill zoomed window
	; and redisplayed using zoom=1.  Now CW_DEFROI can be used properly.
	; ROI points must then be converted back to scale of original data.
	; Image with ROI are redisplayed upon exit of this routine with ZSTACK_SPECTRA_IMGDISP
	; Spectra are redisplayed upon exit with zstack_spectra_makespectrum and zstack_spectra_plotspectrum
	zstack_analyze_imgprep,displayed_file_index,image
	zstack_analyze_bytescale, image, byte_image
;	nonintegral_zoom_data = byte_image
;	IF (image_zoom GE 1.0) THEN BEGIN
;		data = congrid(nonintegral_zoom_data,n_cols*image_zoom,n_rows*image_zoom)
;	ENDIF ELSE BEGIN
;		IF ( ((image_zoom*n_cols) EQ fix(image_zoom*n_cols)) AND $
;			((image_zoom*n_rows) EQ fix(image_zoom*n_rows)) ) THEN BEGIN
;			data = rebin(nonintegral_zoom_data,n_cols*image_zoom,n_rows*image_zoom,/sample)
;		ENDIF ELSE BEGIN
;			data = congrid(nonintegral_zoom_data,n_cols*image_zoom,n_rows*image_zoom)
;		ENDELSE
;	ENDELSE
	IF (image_zoom GE 1.0) THEN BEGIN
		IF ((image_zoom) EQ fix(image_zoom)) THEN BEGIN
			data = rebin(byte_image,n_cols*image_zoom,n_rows*image_zoom,/sample)
		ENDIF ELSE BEGIN
			data = congrid(byte_image,n_cols*image_zoom,n_rows*image_zoom)
		ENDELSE
	ENDIF ELSE BEGIN
		IF ( ((1./image_zoom) EQ fix(1./image_zoom)) AND $
				((image_zoom*n_cols) EQ fix(image_zoom*n_cols)) AND $
				((image_zoom*n_rows) EQ fix(image_zoom*n_rows)) ) THEN BEGIN
			data = rebin(byte_image,n_cols*image_zoom,n_rows*image_zoom,/sample)
		ENDIF ELSE BEGIN
			data = congrid(byte_image,n_cols*image_zoom,n_rows*image_zoom)
		ENDELSE
	ENDELSE
	wset,zstack_spectra_par.image_win
	tv,data,0,0
	roi = cw_defroi(zstack_spectra_par.image_label,zoom=[1,1])
	; convert roi data back to scale of original data
	data(*,*) = 0.
	data(roi) = 1.
	roi = where(congrid(data,n_cols,n_rows) GE 0.5, count)
ENDELSE
svec = size(roi)
IF (svec(0) NE 0) THEN BEGIN	; proceed only if roi is real region
	; trim roi to exclude regions that are clipped by image alignment
	; only if clipbox is non-zero, i.e., some of the image is clipped
	IF ((clipbox(1) NE 0) AND (clipbox(3) NE 0)) THEN BEGIN
		dummy = fltarr(n_cols,n_rows)
		dummy(clipbox[0]:clipbox[1],clipbox[2]:clipbox[3]) = -1
		roi_dummy = fltarr(n_cols,n_rows)
		roi_dummy(roi) = -1
		dummy = temporary(dummy) * temporary(roi_dummy)
		roi = where((dummy EQ 1),count)
	ENDIF
	IF (is_i0_roi NE 0L) THEN BEGIN
		svec = size(i0_roi)
		IF (svec(0) EQ 0) THEN BEGIN
			i0_roi = roi
		ENDIF ELSE BEGIN
			i0_roi = [i0_roi,roi]
			i0_roi = i0_roi(uniq(i0_roi,sort(i0_roi)))
		ENDELSE
		print,'Area of I0 ROI: '+$
			strtrim(string(n_elements(i0_roi)),2)+' pixels, '+$
			strtrim(string(n_elements(i0_roi)*pixarea,format='(f10.3)'),2)+$
			' square microns'
	ENDIF ELSE BEGIN
		n_roi = n_roi +1	; n_roi = 1 for first I spectrum, 2 for second i spectrum, etc.
		svec = size(i_roi)
		IF (svec(0) EQ 0) THEN BEGIN
			i_roi = roi
		ENDIF ELSE BEGIN
			i_roi = [i_roi,roi]
		ENDELSE
		print,'Area of I  ROI: '+$
			strtrim(string(n_elements(roi)),2)+' pixels, '+$
			strtrim(string(n_elements(roi)*pixarea,format='(f10.3)'),2)+$
			' square microns'
		; create roi_index
		IF (n_elements(roi_index) EQ 1) THEN BEGIN
			roi_index = fix([0,n_elements(i_roi)])
		ENDIF ELSE BEGIN
			roi_index = fix([roi_index,n_elements(i_roi)])
				; index of i_roi where each element denotes beginning each individual i_roi
				; i.e., i_roi = [0,253,685,1165] means that
				; the first i_roi region begins at index 0 and ends at index 252,
				; the second i_roi begins at index 253 and ends at index 684, etc.
		ENDELSE
	ENDELSE
ENDIF
return
END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; This routine strips out the first stuff before a comma
FUNCTION zstack_spectra_read_spc_firstword, input_string, firstword_string
;	This is copied from read_csv.pro written by Chris Jacobsen (SUNY - Stony Brook)
;	It is included here (renamed) to make ZSTACK_SPECTRA independent and self-contained
;print,'zstack_spectra_read_spc_firstword'
on_error,2
IF (strlen(input_string) EQ 0) THEN BEGIN
    firstword_string = ''
    return,0
ENDIF
comma_pos = strpos(input_string,',')
IF (comma_pos EQ (-1)) THEN BEGIN
    firstword_string = input_string
    input_string = ''
ENDIF ELSE BEGIN
    firstword_string = strmid(input_string,0,comma_pos)
    length = strlen(input_string)-comma_pos-1
    input_string = strmid(input_string,comma_pos+1,length)
ENDELSE
return,1
END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_spectra_read_spc, filename, items, help=help, quiet=quiet, string=string

@axis_com


;	This is copied from read_csv.pro written by Chris Jacobsen (SUNY - Stony Brook)
;	It is included here (renamed) to make ZSTACK_SPECTRA independent and self-contained
;print,'zstack_spectra_read_spc'
on_error,2
IF (keyword_set(help) OR (n_elements(filename) EQ 0) OR $
    (n_params() EQ 0)) THEN BEGIN
    print, 'zstack_spectra_read_spc,filename,items [/quiet]'
    print, '  Reads in data in the format of Excel .CSV files'
    print, '  /string for reading all data as character strings
    return
ENDIF
filelist = findfile(filename)
svec = size(filelist)
IF (svec(0) EQ 0) THEN BEGIN
    print,'Could not open file "'+filename+'"'
    return
ENDIF
on_ioerror,zstack_spectra_read_spc_oops
get_lun,lun
openr, lun, filename
input_string = ""
got_firstline = 0
n_lines = 0
items_per_line = 0
zero_byte = byte('0')
zero_byte = zero_byte(0)
nine_byte = byte('9')
nine_byte = nine_byte(0)
point_byte = byte('.')
point_byte = point_byte(0)
plus_byte = byte('+')
plus_byte = plus_byte(0)
minus_byte = byte('-')
minus_byte = minus_byte(0)
space_byte = byte(' ')
space_byte = space_byte(0)
tab_byte = byte(9)
WHILE (NOT EOF(lun)) DO BEGIN
    readf, lun, input_string
    firstchar = byte(strmid(input_string,0,1))
    firstchar = firstchar(0)
    IF ((firstchar EQ space_byte) OR (firstchar EQ tab_byte) OR $
        (firstchar EQ plus_byte) OR (firstchar EQ minus_byte) OR $
        (firstchar EQ point_byte) OR $
        ((firstchar GE zero_byte) AND (firstchar LE nine_byte)) ) THEN BEGIN
        IF (got_firstline EQ 0) THEN BEGIN
            IF (zstack_spectra_read_spc_firstword(input_string,word) EQ 0) THEN BEGIN
                goto, zstack_spectra_read_spc_oops
            ENDIF
            words = [word]
            WHILE (zstack_spectra_read_spc_firstword(input_string,word) NE 0) DO BEGIN
                words = [words,word]
            ENDWHILE
            items_per_line = n_elements(words)
            n_lines = 1
            IF keyword_set(string) THEN BEGIN
                items = strarr(items_per_line,n_lines)
                items(0:(items_per_line-1),0) = $
                  words(0:(items_per_line-1))
            ENDIF ELSE BEGIN
                this_items = fltarr(items_per_line)
                items = fltarr(items_per_line,n_lines)
                reads,words,this_items
                items(0:(items_per_line-1),0) = $
                  this_items(0:(items_per_line-1))
            ENDELSE
            got_firstline = 1
        ENDIF ELSE BEGIN
            IF (zstack_spectra_read_spc_firstword(input_string,word) EQ 0) THEN BEGIN
                goto, zstack_spectra_read_spc_oops
            ENDIF
            words = [word]
            WHILE (zstack_spectra_read_spc_firstword(input_string,word) NE 0) DO BEGIN
                words = [words,word]
            ENDWHILE
            items_per_this_line = n_elements(words)
            n_lines = n_lines+1
            old_items = items
            IF keyword_set(string) THEN BEGIN
                items = strarr(items_per_line,n_lines)
                items(0:(items_per_line-1),0:(n_lines-2)) = $
                  old_items(0:(items_per_line-1),0:(n_lines-2))
                items(0:(items_per_this_line-1),(n_lines-1)) = words
            ENDIF ELSE BEGIN
                reads,words,this_items
                items = fltarr(items_per_line,n_lines)
                items(0:(items_per_line-1),0:(n_lines-2)) = $
                  old_items(0:(items_per_line-1),0:(n_lines-2))
                items(0:(items_per_this_line-1),(n_lines-1)) = this_items
            ENDELSE
        ENDELSE
    ENDIF
ENDWHILE
zstack_spectra_read_spc_oops :
close,lun
free_lun,lun
return
END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_spectra_read_spec,spectrum_name,filetype,signal
;print,'zstack_spectra_read_spec'

@axis_com
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

on_error,2
; Check validity of spectrum_name as file
dummy = findfile(spectrum_name, count=count)
IF (count EQ 0) THEN BEGIN
	print,string(7B),'I0 spectrum file, '+spectrum_name+', not found',string(7B)
	return
ENDIF
CASE filetype OF
	'spc' : BEGIN
		items = 0
		zstack_spectra_read_spc,spectrum_name,items
		svec = size(items)
		IF (svec(0) EQ 2) AND (svec(1) EQ 2) THEN BEGIN	; only use *.spc files with one spectrum
			temp_ev = items(0,1:svec(2)-1)
			temp_signal = items(1,1:svec(2)-1)
			temp_ev_min = min(temp_ev,max=temp_ev_max)
			ev_min = min(ev_list,max=ev_max)
			IF ((temp_ev_min LE ev_min) AND (temp_ev_max GE ev_max)) THEN BEGIN
;				new_ev = interpol(temp_ev,temp_ev,ev_list)
				signal = interpol(temp_signal,temp_ev,ev_list)
			ENDIF ELSE BEGIN
    				dummy = where(ev_list GE temp_ev_min AND ev_list LE temp_ev_max, count)
				temp_ev_list = ev_list[dummy]
;				new_ev = interpol(temp_ev,temp_ev,temp_ev_list)
				signal = interpol(temp_signal,temp_ev,temp_ev_list)
				temp_signal = signal
				signal = ev_list
				signal[*] = !values.f_nan
				signal[dummy] = temp_signal
			ENDELSE
		ENDIF ELSE BEGIN	; trap for *.spc files with multiple spectra
			print,string(7B),'I0 spectrum file, '+spectrum_name+', had improper dimensions :',string(7B)
			print,svec
			signal = ev_list
			signal[*] = 0
		ENDELSE
	END
	'txt' : BEGIN
		items = 0
		zstack_spectra_read_spc,spectrum_name,items
		svec = size(items)
		IF (svec(0) EQ 2) AND (svec(1) EQ 2) THEN BEGIN	; only use *.spc files with one spectrum
			temp_ev = items(0,1:svec(2)-1)
			temp_signal = items(1,1:svec(2)-1)
			temp_ev_min = min(temp_ev,max=temp_ev_max)
			ev_min = min(ev_list,max=ev_max)
			IF ((temp_ev_min LE ev_min) AND (temp_ev_max GE ev_max)) THEN BEGIN
;				new_ev = interpol(temp_ev,temp_ev,ev_list)
				signal = interpol(temp_signal,temp_ev,ev_list)
			ENDIF ELSE BEGIN
    				dummy = where(ev_list GE temp_ev_min AND ev_list LE temp_ev_max, count)
				temp_ev_list = ev_list[dummy]
;				new_ev = interpol(temp_ev,temp_ev,temp_ev_list)
				signal = interpol(temp_signal,temp_ev,temp_ev_list)
				temp_signal = signal
				signal = ev_list
				signal[*] = !values.f_nan
				signal[dummy] = temp_signal
			ENDELSE
		ENDIF ELSE BEGIN	; trap for *.spc files with multiple spectra
			print,string(7B),'I0 spectrum file, '+spectrum_name+', had improper dimensions :',string(7B)
			print,svec
			signal = ev_list
			signal[*] = 0
		ENDELSE
	END

	'xas' : BEGIN
		read_xas,spectrum_name,temp_ev,temp_signal,header

		IF ( (n_elements(temp_ev) NE 0) AND (n_elements(temp_signal) NE 0) ) THEN BEGIN
			temp_ev_min = min(temp_ev,max=temp_ev_max)
			round_ev_list = round(1000.*ev_list)/1000.
;			print,'round_ev_list :',round_ev_list
			ev_min = min(round_ev_list,max=ev_max)
			IF ((temp_ev_min LE ev_min) AND (temp_ev_max GE ev_max)) THEN BEGIN
;				new_ev = interpol(temp_ev,temp_ev,round_ev_list)
				signal = interpol(temp_signal,temp_ev,round_ev_list)
			ENDIF ELSE BEGIN
    				dummy = where(round_ev_list GE temp_ev_min AND round_ev_list LE temp_ev_max, count)
				temp_ev_list = ev_list[dummy]
;				new_ev = interpol(temp_ev,temp_ev,temp_ev_list)
				signal = interpol(temp_signal,temp_ev,temp_ev_list)
				temp_signal = signal
				signal = ev_list
				signal[*] = !values.f_nan
				signal[dummy] = temp_signal
			ENDELSE
		ENDIF
	END
	'raw' : BEGIN
		CASE data_source OF
			1 : BEGIN	; STXM data from X1A/NSLS
				n_cols = 0
				this_ev = 0.
;				read_stxm,this_filename,sd,/header,/nocopybytes
				read_stxm,this_filename,sd,temp_signal
				IF (n_cols NE 0) THEN BEGIN
					temp_ev = 12398.52/sd.wavelength
				ENDIF
				temp_ev_min = min(temp_ev,max=temp_ev_max)
				ev_min = min(ev_list,max=ev_max)
				IF ((temp_ev_min LE ev_min) AND (temp_ev_max GE ev_max)) THEN BEGIN
;					new_ev = interpol(temp_ev,temp_ev,ev_list)
					signal = interpol(temp_signal,temp_ev,ev_list)
				ENDIF ELSE BEGIN
    					dummy = where(ev_list GE temp_ev_min AND ev_list LE temp_ev_max, count)
					temp_ev_list = ev_list[dummy]
;					new_ev = interpol(temp_ev,temp_ev,temp_ev_list)
					signal = interpol(temp_signal,temp_ev,temp_ev_list)
					temp_signal = signal
					signal = ev_list
					signal[*] = !values.f_nan
					signal[dummy] = temp_signal
				ENDELSE
			END
			2 : BEGIN	; cryo-STXM data from X1A/NSLS
				n_cols = 0
				this_ev = 0.
;				read_sxm,this_filename,/header
				read_sxm,this_filename
				IF (n_cols NE 0) THEN BEGIN
					temp_signal = float(image_data(*,*,0))
					temp_ev = 1239.852/sxm_par.now_nm
				ENDIF
				temp_ev_min = min(temp_ev,max=temp_ev_max)
				ev_min = min(ev_list,max=ev_max)
				IF ((temp_ev_min LE ev_min) AND (temp_ev_max GE ev_max)) THEN BEGIN
;					new_ev = interpol(temp_ev,temp_ev,ev_list)
					signal = interpol(temp_signal,temp_ev,ev_list)
				ENDIF ELSE BEGIN
    					dummy = where(ev_list GE temp_ev_min AND ev_list LE temp_ev_max, count)
					temp_ev_list = ev_list[dummy]
;					new_ev = interpol(temp_ev,temp_ev,temp_ev_list)
					signal = interpol(temp_signal,temp_ev,temp_ev_list)
					temp_signal = signal
					signal = ev_list
					signal[*] = !values.f_nan
					signal[dummy] = temp_signal
				ENDELSE
			END
			3 : BEGIN
			END
			4 : BEGIN	; BL7 data from ALS
			END
			5 : BEGIN	; Polymer STXM data rom ALS
			END
		ENDCASE
	END
ENDCASE
return
END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_spectra_event,event
;print,'zstack_spectra_event'

@axis_com
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

on_error,2
CASE event.id OF
    zstack_spectra_par.add_i0_roi_label : BEGIN
        zstack_spectra_desensitive
		widget_control, zstack_spectra_par.image_label, sensitive = 1
        is_i0_roi = 1L
        print,'Adding I0 pixels'
        zstack_spectra_roi
        zstack_spectra_makespectrum
        zstack_spectra_plotspectrum,displayed_file_index
        zstack_spectra_imgdisp,displayed_file_index
		widget_control, zstack_spectra_par.image_label, sensitive = 0
        zstack_spectra_sensitive
; image_ratio can now be allowed to be set to 1 via button on dialog window - to be implemented
    END
    zstack_spectra_par.reset_i0_label : BEGIN
		zstack_spectra_desensitive
		print,'resetting I0 pixels'
		i0_roi = 0
		spectrum[0,*] = 0
		zstack_spectra_makespectrum
		zstack_spectra_plotspectrum,displayed_file_index
		zstack_spectra_imgdisp,displayed_file_index
		zstack_spectra_sensitive
    END
    zstack_spectra_par.reset_both_i0_i_label : BEGIN
		zstack_spectra_desensitive
		print,'resetting both I0 and I pixels'
		i0_roi = 0
		i_roi = 0
		n_roi = 0
		roi_index = 0
		spectrum[0:14,*] = 0
		zstack_spectra_makespectrum
		zstack_spectra_plotspectrum,displayed_file_index
		zstack_spectra_imgdisp,displayed_file_index
		zstack_spectra_sensitive
   END
    zstack_spectra_par.add_i_roi_label : BEGIN
		IF (n_roi LT 14) THEN BEGIN
;			print,'n_roi =',n_roi
			zstack_spectra_desensitive
			widget_control, zstack_spectra_par.image_label, sensitive = 1
			is_i0_roi = 0L
			print,'Adding I pixels to ROI #'+strtrim(n_roi+1,2)
			zstack_spectra_roi
			zstack_spectra_makespectrum
			zstack_spectra_plotspectrum,displayed_file_index
			zstack_spectra_imgdisp,displayed_file_index
			widget_control, zstack_spectra_par.image_label, sensitive = 0
			zstack_spectra_sensitive
		ENDIF
		IF (n_roi EQ 14) THEN BEGIN	; disable 'Add I ROI' button
			widget_control,zstack_spectra_par.add_i_roi_label, sensitive=0
		ENDIF
    END
	zstack_spectra_par.reset_last_i_label : BEGIN
		zstack_spectra_desensitive
		print,'resetting last set of I pixels'
		IF (n_roi GT 1) THEN BEGIN
			n_roi = n_roi -1
			roi_index = roi_index[0:n_roi]
			i_roi = i_roi[0:roi_index[n_roi]-1]
			spectrum[n_roi,*] = 0
		ENDIF ELSE BEGIN
			i_roi = 0
			n_roi = 0
			roi_index = 0
			spectrum[1:14,*] = 0
		ENDELSE
;		IF (n_roi EQ 0) THEN i_roi = 0
		zstack_spectra_makespectrum
		zstack_spectra_plotspectrum,displayed_file_index
		zstack_spectra_imgdisp,displayed_file_index
		zstack_spectra_sensitive
	END
    zstack_spectra_par.reset_all_i_label : BEGIN
		zstack_spectra_desensitive
	print,'resetting all I pixels'
        i_roi = 0
        n_roi = 0
        roi_index = 0
        spectrum[1:14,*] = 0
        zstack_spectra_makespectrum
        zstack_spectra_plotspectrum,displayed_file_index
        zstack_spectra_imgdisp,displayed_file_index
        zstack_spectra_sensitive
   END
    zstack_spectra_par.i0_filename_label : BEGIN
        on_ioerror, i0_filename_label_oops
        temp_string = ''
        widget_control, zstack_spectra_par.i0_filename_label,$
          get_value = temp_string
        i0_filename = strtrim(temp_string(0),2)
        i0_filename_label_oops :
        widget_control, zstack_spectra_par.i0_filename_label, $
          set_value = ' '+i0_filename
        zstack_spectra_sensitive
    END
	zstack_spectra_par.i0_fileformat_list_label : BEGIN
		i0_filetype = (['unspecified','raw','xas','spc','txt'])[event.index]
	END
	zstack_spectra_par.i0_retrieve_label : BEGIN
		zstack_spectra_desensitive
		zstack_spectra_read_spec,i0_filename,i0_filetype,data
		i0_reference_spectrum = data
		spectrum[0,*] = i0_scale_factor * i0_reference_spectrum
;		spectrum[0,*] = data
		zstack_spectra_plotspectrum,displayed_file_index
		zstack_spectra_imgdisp,displayed_file_index
		zstack_spectra_sensitive
	END
    zstack_spectra_par.i0_browse_label : BEGIN
		on_ioerror, browse_i0_label_oops
		zstack_spectra_desensitive
		cd,current=current_directory

		CASE i0_filetype OF
			'raw' : BEGIN
				extension = (['.nc','.sxm','.sm','.dat','poly'])[data_source-1]
				dummy = dialog_pickfile(filter='*'+extension, /read, /must_exist, $
						path=data_directory, get_path=data_directory, $
						title='Select I0 spectrum file :')
			END
			'xas' : BEGIN
				dummy = dialog_pickfile(filter='*.xas', /read, /fix_filter, /must_exist, $
						path=data_directory, get_path=data_directory, $
						title='Select I0 spectrum file :')
			END
			'spc' : BEGIN
				dummy = dialog_pickfile(filter='*.spc', /read, /fix_filter, /must_exist, $
						path=data_directory, get_path=data_directory, $
						title='Select I0 spectrum file :')
			END
			'txt' : BEGIN
				dummy = dialog_pickfile(filter='*.txt', /read, /fix_filter, /must_exist, $
						path=data_directory, get_path=data_directory, $
						title='Select I0 spectrum file :')
			END
			ELSE:	BEGIN
				dummy = dialog_pickfile(filter='*.*', /read, /must_exist, $
						path=data_directory, get_path=data_directory, $
						title='Select I0 spectrum file :')
				i0_extension = strmid(dummy,rstrpos(dummy,'.'),strlen(dummy))
				CASE i0_extension OF
					'.nc' : BEGIN
						i0_filetype = 'raw'
						i0_data_source = 1
						widget_control, zstack_spectra_par.i0_fileformat_list_label,$
								set_droplist_select=1
					END
					'.sxm' : BEGIN
						i0_filetype = 'raw'
						i0_data_source = 2
						widget_control, zstack_spectra_par.i0_fileformat_list_label,$
								set_droplist_select=1
					END
					'.sm' : BEGIN
						i0_filetype = 'raw'
						i0_data_source = 3
						widget_control, zstack_spectra_par.i0_fileformat_list_label,$
								set_droplist_select=1
					END
					'.dat' : BEGIN
						i0_filetype = 'raw'
						i0_data_source = 4
						widget_control, zstack_spectra_par.i0_fileformat_list_label,$
								set_droplist_select=1
					END
					'.poly' : BEGIN
						i0_filetype = 'raw'
						widget_control, zstack_spectra_par.i0_fileformat_list_label,$
								set_droplist_select=1
						i0_data_source = 5
					END
					'.xas' : BEGIN
						i0_filetype = 'xas'
						widget_control, zstack_spectra_par.i0_fileformat_list_label,$
								set_droplist_select=2
					END
					'.spc' : BEGIN
						i0_filetype = 'spc'
						widget_control, zstack_spectra_par.i0_fileformat_list_label,$
								set_droplist_select=3
					END
					'.txt' : BEGIN
						i0_filetype = 'txt'
						widget_control, zstack_spectra_par.i0_fileformat_list_label,$
								set_droplist_select=3
					END
					ELSE : BEGIN
						IF (strlen(dummy) EQ 0) THEN BEGIN
							GOTO, browse_i0_label_oops	; trap if CANCEL is chosen in dialog
						ENDIF ELSE BEGIN
							print,string(7B),'Unrecognized I0 file type'
							i0_filetype = 'unknown'
						ENDELSE
					END
				ENDCASE
			END
		ENDCASE
		IF (dummy EQ '') THEN BEGIN
			data_directory = current_directory
				; sets data_directory to initial directory if CANCEL is chosen during dialog
				; without this, data_directory becomes null string
			GOTO, browse_i0_label_oops
				; trap if CANCEL is chosen in dialog
		ENDIF
		zstack_analyze_extract_filename, dummy, i0_filename
		; strip the directory from dummy to get i0_filename
		widget_control, zstack_spectra_par.i0_filename_label, $
				set_value = ' '+i0_filename
		IF (i0_filetype NE 'unknown') THEN BEGIN
;			zstack_spectra_get_i0_file,i0_filename
		ENDIF

		zstack_spectra_read_spec,i0_filename,i0_filetype,data
		i0_reference_spectrum = data
		spectrum[0,*] = i0_scale_factor * i0_reference_spectrum
;		spectrum[0,*] = data
		i0_roi = [-1]
		zstack_spectra_imgdisp,displayed_file_index
		zstack_spectra_plotspectrum,displayed_file_index
		browse_i0_label_oops :
		zstack_spectra_sensitive
    END
	zstack_spectra_par.i0_scale_factor_label : BEGIN
		on_ioerror, i0_scale_factor_label_oops
		zstack_spectra_desensitive
		init_value = i0_scale_factor
		temp_string = ''
		widget_control, zstack_spectra_par.i0_scale_factor_label, $
				get_value = temp_string
		IF (strlen(temp_string(0)) EQ 0) THEN GOTO, i0_scale_factor_label_oops
		reads,temp_string(0),i0_scale_factor
		i0_scale_factor_label_oops :
		IF (i0_scale_factor NE init_value) THEN BEGIN
			widget_control, zstack_spectra_par.i0_scale_factor_label, $
					set_value = ' '+strtrim(string(i0_scale_factor,format='(f10.3)'),2)
			IF (n_elements(reference_spectrum) NE 0) THEN BEGIN
				spectrum[0,*] = i0_scale_factor * i0_reference_spectrum
			ENDIF
		ENDIF
		zstack_plot_imgdisp,displayed_file_index
		zstack_spectra_plotspectrum,displayed_file_index
		zstack_spectra_sensitive
	END

    zstack_spectra_par.spectra_filename_label : BEGIN
        temp_string = ''
        widget_control, zstack_spectra_par.spectra_filename_label,get_value = temp_string
        on_ioerror, spectra_filename_label_oops
        spectra_filename_header = strtrim(temp_string(0),2)
        dotpos = strpos(spectra_filename_header,'.')
        IF (dotpos NE (-1)) THEN BEGIN
            spectra_filename_header = strmid(spectra_filename_header,0,dotpos)
        ENDIF
        spectra_filename_label_oops :
        widget_control, zstack_spectra_par.spectra_filename_label, $
				set_value = ' '+spectra_filename_header
        zstack_spectra_sensitive
    END
	zstack_spectra_par.spectra_dataformat_list_label : BEGIN	; Single Beam vs Percent Tranmittance vs Absorbance
		save_spectra_type(0) = event.index
		zstack_spectra_sensitive
	END
	zstack_spectra_par.spectra_fileformat_list_label : BEGIN	; *.xas vs *.spc vs *.txt vs *S.gif
		save_spectra_type(2) = event.index
		print,'save_spectra_type :',save_spectra_type
		IF (save_spectra_type(2) EQ 1) THEN BEGIN
			save_spectra_type(1) = 1
			widget_control, zstack_spectra_par.spectra_filetype_list_label, $
				set_value = ['Select number of data files :','File for each spectrum']
			widget_control,zstack_spectra_par.spectra_filetype_list_label, set_droplist=1
		ENDIF ELSE BEGIN
			widget_control, zstack_spectra_par.spectra_filetype_list_label, $
				set_value = ['Select number of data files :','File for each spectrum','All spectra in one file']
		ENDELSE
		zstack_spectra_sensitive
	END
	zstack_spectra_par.spectra_filetype_list_label : BEGIN		; File for each spectrum vs All spectra in one file
		save_spectra_type(1) = event.index
		zstack_spectra_sensitive
	END
	zstack_spectra_par.spectra_save_label : BEGIN
		IF (strlen(spectra_filename_header) NE 0) THEN BEGIN	;added, CGZ, save only if spectra_filename_header is not empty
			zstack_spectra_desensitive
			widget_control, hourglass=1
			zstack_spectra_save_spec
			widget_control, hourglass=0
			zstack_spectra_sensitive
		ENDIF
	END
	zstack_spectra_par.roi_filename_label : BEGIN
		temp_string = ''
		widget_control, zstack_spectra_par.roi_filename_label,get_value = temp_string
		on_ioerror, roi_filename_label_oops
		roi_filename = strtrim(temp_string(0),2)
		IF (strlen(roi_filename) NE 0) THEN BEGIN
			dotpos = strpos(roi_filename,'.')
			IF (dotpos EQ -1) THEN BEGIN
				roi_filename = roi_filename+'.roi'
			ENDIF ELSE BEGIN
				IF (strpos(strlowcase(roi_filename),'.roi') EQ -1) THEN BEGIN
					roi_filename = roi_filename+'roi'
				ENDIF
			ENDELSE
		ENDIF
		roi_filename_label_oops :
		widget_control, zstack_spectra_par.roi_filename_label, $
				set_value = ' '+roi_filename
		zstack_spectra_sensitive
	END

	zstack_spectra_par.roi_save_label : BEGIN
		on_ioerror, stack_save_align_file_oops
		zstack_spectra_desensitive
		cd,current=current_directory
		dummy = dialog_pickfile(filter='*.roi', /write, /fix_filter, $
				file=roi_filename, path=data_directory, get_path=data_directory, $
				title='Save file with regions of interest as :')
		IF (dummy EQ '') THEN BEGIN
			data_directory = current_directory
				; sets data_directory to initial directory if CANCEL is chosen during dialog
				; without this, data_directory becomes null string
			GOTO, stack_save_align_file_oops
				; trap if CANCEL is chosen in dialog
		ENDIF

		zstack_analyze_extract_filename, dummy, roi_filename
		IF (strlen(roi_filename) NE 0) THEN BEGIN	;added, CGZ, save only if spectra_filename_header is not empty
			widget_control, hourglass=1
			zstack_spectra_save_roi
			widget_control, hourglass=0
		ENDIF
		stack_save_align_file_oops :
		widget_control, hourglass=0
		widget_control, zstack_spectra_par.roi_filename_label, $
				set_value = ' '+roi_filename
		zstack_spectra_sensitive
	END
    zstack_spectra_par.roi_read_label : BEGIN
		widget_control, hourglass=1
		zstack_spectra_read_roi,roi_filename
		widget_control, hourglass=0

;		zstack_spectra_makespectrum
		zstack_spectra_imgdisp,displayed_file_index
		zstack_spectra_plotspectrum, displayed_file_index
		zstack_spectra_sensitive
    END

    zstack_spectra_par.roi_browse_label : BEGIN
		on_ioerror, browse_roi_label_oops
		zstack_spectra_desensitive
		cd,current=current_directory
		dummy = dialog_pickfile(filter='*.roi', /read, /fix_filter, /must_exist, $
				path=data_directory, get_path=data_directory, $
				title='Select ROI file :')
		IF (dummy EQ '') THEN BEGIN
			data_directory = current_directory
				; sets data_directory to initial directory if CANCEL is chosen during dialog
				; without this, data_directory becomes null string
			GOTO, browse_roi_label_oops
				; trap if CANCEL is chosen in dialog
		ENDIF
		zstack_analyze_extract_filename, dummy, roi_filename
		; strip the directory from dummy to get roi_filename
		widget_control, zstack_spectra_par.roi_filename_label, $
				set_value = ' '+roi_filename
		widget_control, hourglass=1
		zstack_spectra_read_roi,roi_filename
		widget_control, hourglass=0

;		zstack_spectra_makespectrum
		zstack_spectra_imgdisp,displayed_file_index
		zstack_spectra_plotspectrum, displayed_file_index
		browse_roi_label_oops :
		widget_control, hourglass=0
		zstack_spectra_sensitive
    END

	zstack_spectra_par.image_label : BEGIN
	END
	zstack_spectra_par.clipped_image_label : BEGIN
	END
    zstack_spectra_par.filename_display_list_label : BEGIN
		zstack_spectra_desensitive
        displayed_file_index = event.index
        zstack_spectra_imgdisp, displayed_file_index
        zstack_spectra_plotspectrum, displayed_file_index
		zstack_spectra_sensitive
    END
	zstack_spectra_par.prev_image_label : BEGIN
		zstack_spectra_desensitive
		IF (displayed_file_index GT 0) THEN BEGIN
			displayed_file_index = displayed_file_index-1
		ENDIF ELSE BEGIN
			displayed_file_index = n_files-1
		ENDELSE
		zstack_spectra_imgdisp,displayed_file_index
		zstack_spectra_plotspectrum, displayed_file_index
		zstack_spectra_sensitive
	END

	zstack_spectra_par.next_image_label : BEGIN
		zstack_spectra_desensitive
		IF (displayed_file_index LT (n_files-1)) THEN BEGIN
			displayed_file_index = displayed_file_index+1
		ENDIF ELSE BEGIN
			displayed_file_index = 0
		ENDELSE
		zstack_spectra_imgdisp,displayed_file_index
		zstack_spectra_plotspectrum, displayed_file_index
		zstack_spectra_sensitive
	END

    zstack_spectra_par.play_movie_label : BEGIN
		movie_active = 1
        zstack_spectra_desensitive
        FOR i=0,(n_files-1) DO BEGIN
		IF (displayed_file_index LT (n_files-1)) THEN BEGIN
			displayed_file_index = displayed_file_index+1
		ENDIF ELSE BEGIN
			displayed_file_index = 0
		ENDELSE
            zstack_spectra_imgdisp, displayed_file_index
            zstack_spectra_plotspectrum, displayed_file_index
            wait,movie_delay
        ENDFOR
		movie_active = 0
        zstack_spectra_sensitive
    END
	zstack_spectra_par.display_parameters_label : BEGIN
		zstack_spectra_desensitive
		wshow,zstack_spectra_par.main_base_win,0
		zstack_display,'spectra'
	END
	zstack_spectra_par.plot_parameters_label : BEGIN
		zstack_spectra_desensitive
		wshow,zstack_spectra_par.main_base_win,0
		zstack_plot,'spectra'
	END
    zstack_spectra_par.spectra_plot_label : BEGIN
        ;; only act on a down click of the mouse
        if (event.type eq 0) then begin
		zstack_spectra_desensitive
            wset,zstack_spectra_par.spectra_plot_win
            this_ev_abs = convert_coord(event.x,event.y,/device,/to_data)
            this_ev = this_ev_abs(0)
            dummy = min(abs(this_ev-ev_list),this_i_ev)
            print,'You clicked at '+$
              strtrim(string(ev_list(this_i_ev),format='(f10.2)'),2)+' eV, '+$
              strtrim(string(12398.52/ev_list(this_i_ev),format='(f10.3)'),2)+' A'
   ;         n_files = n_elements(filename_list)
            displayed_file_index = (this_i_ev>0)<(n_files-1)
            widget_control,$
              zstack_spectra_par.filename_display_list_label,$
              set_list_select=displayed_file_index
            zstack_spectra_imgdisp, displayed_file_index
            zstack_spectra_plotspectrum, displayed_file_index
		zstack_spectra_sensitive
        endif
    END
    zstack_spectra_par.profile_label : BEGIN
	  zstack_spectra_desensitive
;       wshow,zstack_spectra_par.main_base_win,0
       init_zoom = image_zoom
;       init_offset = spectrum_offset
	  zstack_profile,/passed_data
       wshow,zstack_spectra_par.main_base_win,0
    END
    zstack_spectra_par.save_label : BEGIN
	  zstack_spectra_desensitive
;       wshow,zstack_spectra_par.main_base_win,0
       init_zoom = image_zoom
;       init_offset = spectrum_offset
	  zstack_save
       wshow,zstack_spectra_par.main_base_win,0
    END

	zstack_spectra_par.reset_color : BEGIN
		; restore color table after IDL Slicer is finished
		wshow,zstack_spectra_par.main_base_win,1
		zstack_color
		; redisplay images and spectra
		zstack_spectra_imgdisp, displayed_file_index
		zstack_spectra_plotspectrum, displayed_file_index
	END

	zstack_spectra_par.IDL_Slicer_label : BEGIN
		wshow,zstack_spectra_par.main_base_win,0
;   reduced_data = image_stack[0:n_cols/2,0:n_rows/2,*]
;		slicer3,reduced_data,group=zstack_spectra_par.main_base,/modal
		slicer3,group=zstack_spectra_par.main_base,/modal
		; restore color table after IDL Slicer is finished
		wshow,zstack_spectra_par.main_base_win,1
		zstack_color
		; redisplay images and spectra
		zstack_spectra_imgdisp, displayed_file_index
		zstack_spectra_plotspectrum, displayed_file_index
	END
	zstack_spectra_par.exit_label : BEGIN
		image_stack = 0
		image_zoom = 1
		n_cols = 0
		n_rows = 0
		i_roi = 0
		i0_roi = 0
		n_roi = 0
		roi_index = 0
		spectrum = 0
; -------------------------------------------------------------------------------------
; Zimba destroyed his COMMON blocks at exit of spectra = NO WONDER there were crashes !!
; --------------- REMOVED  26-Jul-09----------------------------------------------------
;		zstack_common = 0
;		zstack_analyze_common = 0
;		zstack_align_common = 0
;		zstack_spectra_common = 0
;		zstack_save_common = 0
;		zstack_display_common = 0
;		zstack_color_common = 0

		print,'Exiting ZSTACK package from ZSTACK_SPECTRA'
		print, 'DefPath = ', DefPath
		t=ax_name(DefPath)
		lpath = t(0)
		widget_control,zstack_spectra_par.main_base,/destroy
		IF (axis_call EQ 1) THEN BEGIN
			; Back to AXIS, do the things APH coded
			loadct,0,/silent
			; zstack_to_axis  ; has been in for long time but does not exist !!
		ENDIF ELSE BEGIN
			retall
		ENDELSE
	END
ENDCASE
return
END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_spectra_dialog, Group_Leader=groupLeader
;print,'zstack_spectra_dialog'

@axis_com
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

on_error,2
; Initialize variables if they have no value
;	needed when running zstack_align_dialog as stand-alone prior to running zstack_align
;	i.e., when programming dialog window
IF (n_elements(n_cols) EQ 0) THEN n_cols = 100
IF (n_elements(n_rows) EQ 0) THEN n_rows = 100
IF (n_elements(image_zoom) EQ 0) THEN image_zoom = 1
IF (n_elements(data_shifted) EQ 0) then data_shifted = 1
IF (n_elements(n_clipped_cols) EQ 0) THEN n_clipped_cols = 75
IF (n_elements(n_clipped_rows) EQ 0) THEN n_clipped_rows = 75
IF (n_tags(zstack_spectra_par) EQ 0) THEN BEGIN
	zstack_spectra_par = $
	   {	zstack_spectra_par			    , $
		main_base					: 0L, $
		add_i0_roi_label			: 0L, $
		reset_i0_label				: 0L, $
		reset_both_i0_i_label			: 0L, $
		add_i_roi_label				: 0L, $
		reset_last_i_label			: 0L, $
		reset_all_i_label			: 0L, $
		path_label					: 0L, $
		path_browse_label			: 0L, $
		i0_filename_label			: 0L, $
		i0_fileformat_list_label		: 0L, $
		i0_retrieve_label			: 0L, $
		i0_browse_label				: 0L, $
		i0_scale_factor_label			: 0L, $
		spectra_filename_label		: 0L, $
		spectra_dataformat_list_label	: 0L, $
		spectra_filetype_list_label	: 0L, $
		spectra_fileformat_list_label	: 0L, $
		spectra_save_label			: 0L, $
		spectra_browse_label			: 0L, $
		roi_filename_label			: 0L, $
		roi_save_label				: 0L, $
		roi_read_label				: 0L, $
		roi_browse_label			: 0L, $
		image_scale_label			: 0L, $
		image_ratio_label			: 0L, $
		image_i0_ratio_label			: 0L, $
		display_single_beam_label		: 0L, $	; not currently implemented
		display_absorbance_label		: 0L, $	; not currently implemented
		image_label				: 0L, $
		clipped_image_label			: 0L, $
		colorbar_label				: 0L, $
		filename_display_list_label	: 0L, $
		next_image_label			: 0L, $
		prev_image_label			: 0L, $
		play_movie_label			: 0L, $
		display_parameters_label		: 0L, $
		plot_parameters_label			: 0L, $
		spectra_plot_label			: 0L, $
		profile_label				: 0L, $
		save_label					: 0L, $
		IDL_Slicer_label			: 0L, $
		exit_label					: 0L, $
		main_base_win				: 0L, $
		image_win					: 0L, $
		clipped_image_win			: 0L, $
		colorbar_win				: 0L, $
		spectra_plot_win			: 0L, $
		reset_color					: 0L  $	  ; (06jul05 aph) added
      }
     ENDIF
comp_screen_res=GET_SCREEN_SIZE()

; Start of dialog window for spectral analysis
IF N_Elements(groupLeader) EQ 0 THEN BEGIN
	IF !version.os_family EQ 'Windows' then begin
		zstack_spectra_par.main_base = widget_base(title='ZSTACK Spectra', /col, /SCROLL)
	ENDIF else begin
		zstack_spectra_par.main_base = widget_base(title='ZSTACK Spectra', /col, /SCROLL, $
	                X_SCROLL_SIZE=comp_screen_res(0)*.70,Y_SCROLL_SIZE=comp_screen_res(1)*.90 )
	ENDELSE
ENDIF ELSE BEGIN
	zstack_spectra_par.main_base = widget_base(title='ZSTACK Spectra', /col, $
			/Modal, Group_Leader=groupLeader )
ENDELSE
;zstack_spectra_par.main_base = widget_base(title='ZSTACK Spectra', /column, /scroll, $
;			xoffset=100, yoffset=50 )
			; use ofsets so that cw_defroi window will be apparent
row1 = widget_base(zstack_spectra_par.main_base,/row)
col1 = widget_base(row1,/column)
col2 = widget_base(row1,/column)
base = widget_base(col1,/column,/frame,space=0)
zlabel = widget_label( base, value = 'Select Regions for Spectra :' )
zlabel = widget_label( base, value = '(use Region of Interest dialog window)' )
row = widget_base(base,/row)
col1a = widget_base(row,/col)
zstack_spectra_par.add_i0_roi_label = widget_button(col1a,		value='  Add I0 region  ')
zstack_spectra_par.reset_i0_label = widget_button(col1a,		value='     Reset I0    ')
zstack_spectra_par.reset_both_i0_i_label = widget_button(col1a,	value='Reset both I0 & I')
col1b = widget_base(row,/col)
zstack_spectra_par.add_i_roi_label = widget_button(col1b,		value='   Add I region  ')
zstack_spectra_par.reset_last_i_label = widget_button(col1b,	value='   Reset last I  ')
zstack_spectra_par.reset_all_i_label = widget_button(col1b,	value='   Reset all I   ')
;	base = widget_base(col1,/column,/frame)
;	row = widget_base(base,/row)
;	zlabel = widget_label(row, value='Directory Path :',xsize=90,/align_left)
;	zlabel = widget_label(row, value='     ')
;	zstack_spectra_par.path_browse_label = widget_button( row, $
;			value = 'Browse', /align_right)
;	row = widget_base(base,/row)
;	zstack_spectra_par.path_label  = widget_text(row,/editable, xsize=40, ysize=2, /wrap)
base = widget_base(col1,/column,/frame)
row = widget_base(base,/row)
zlabel = widget_label(row, value='I0 filename :',xsize=90,/align_left)
zstack_spectra_par.i0_filename_label = widget_text(row,/editable, value='', xsize=25)
zstack_spectra_par.i0_fileformat_list_label = widget_droplist( base, xsize=200, $
		value = ['Select type of data file :','Raw','*.xas (tab-separated)','*.spc (comma-separated)','*.txt (tab-separated)'], /align_center)
zstack_spectra_par.i0_retrieve_label = widget_button(base, value='Retrieve I0 file')
zstack_spectra_par.i0_browse_label = widget_button(base, value='Browse for I0 file')
row = widget_base(base,/row)
zlabel = widget_label(row, value='I0 scale factor :',xsize=90,/align_left)
zstack_spectra_par.i0_scale_factor_label = widget_text(row,/editable, value='', xsize=25)
base = widget_base(col1,/column,/frame)
row = widget_base(base,/row)
zlabel = widget_label( row, value = 'Base filename: ',xsize=90,/align_left)
zstack_spectra_par.spectra_filename_label = widget_text(row,/editable,xsize=25)
zstack_spectra_par.spectra_dataformat_list_label = widget_droplist( base, xsize=200, $
		value = ['Select type of spectra :','Single Beam Data','Percent Transmittance Data','Absorbance Data'], /align_center)
zstack_spectra_par.spectra_fileformat_list_label = widget_droplist( base, xsize=200, $
		value = ['Select type of data file :','*.xas (tab-separated)','*.spc (comma-separated)','*.txt (tab-separated)','*S.gif (GIF graphic)'], /align_center)
zstack_spectra_par.spectra_filetype_list_label = widget_droplist( base, xsize=200, $
		value = ['Select number of data files :','File for each spectrum','All spectra in one file'], /align_center)
zstack_spectra_par.spectra_save_label = widget_button(base,value='Save Spectra')
base = widget_base(col1,/column,/frame)
row = widget_base(base,/row)
zlabel = widget_label( row, value = 'ROI filename: ',xsize=90,/align_left)
zstack_spectra_par.roi_filename_label = widget_text(row,/editable,xsize=25)
zstack_spectra_par.roi_save_label = widget_button(base,value='Save ROI as *.roi')
zstack_spectra_par.roi_read_label = widget_button(base,value='Retrieve *.roi')
zstack_spectra_par.roi_browse_label = widget_button(base,value='Browse for *.roi')
; Right-hand column of dialog window
base = widget_base(col2,/row,/frame,/align_center)
col2a = widget_base(base,/col)
IF (data_shifted EQ 1) THEN BEGIN
	zlabel = widget_label(col2a, value='Shifted Image',/align_center)
ENDIF ELSE BEGIN
	zlabel = widget_label(col2a, value='Unshifted Image',/align_center)
ENDELSE
zstack_spectra_par.image_label = widget_draw(col2a,$
              xsize=n_cols*image_zoom, ysize=n_rows*image_zoom,$
              retain=2,/button_events,/motion,/align_center)
; if there is a non-zero shift, i.e., alignment was kept, display clipped image
; if shift is discarded, i.e., all zero, don't display clipped image
IF (data_shifted EQ 1) THEN BEGIN
	col2b = widget_base(base,/col)
	zlabel = widget_label(col2b, value='Clipped Image',/align_center)
	zstack_spectra_par.clipped_image_label = widget_draw(col2b,$
			xsize=n_clipped_cols*image_zoom, ysize=n_clipped_rows*image_zoom,$
			retain=2,/button_events,/align_center)
ENDIF
col2c = widget_base(base,/col)
zlabel = widget_label(col2c, value='I',/align_center)
zstack_spectra_par.colorbar_label = widget_draw(col2c,$
              xsize=10*(image_zoom>1), ysize=n_rows*image_zoom,$
              retain=2, sensitive=0, /align_center)
zstack_spectra_par.filename_display_list_label = widget_droplist( col2, $
		value = 'Select File', /align_center, scr_xsize=250)
zstack_spectra_par.prev_image_label = widget_button(col2, value='Display Previous Image')
zstack_spectra_par.next_image_label = widget_button(col2, value='Display Next Image')
zstack_spectra_par.play_movie_label = widget_button(col2,value='Play movie')
row = widget_base( col2, /row ,/align_center)
zstack_spectra_par.reset_color = widget_button(row, value='   Reset color   ')
zstack_spectra_par.display_parameters_label = widget_button(row,value='Display Parameters')
zstack_spectra_par.plot_parameters_label = widget_button(row,value=' Plot  Parameters ')
row = widget_base(col2,/row,/align_center)
zstack_spectra_par.spectra_plot_label = widget_draw(row, $
			xsize=((3*n_cols*image_zoom+25)>250)<500, $
			ysize=((n_rows*image_zoom)>150)<250,retain=2,/button_events)
zstack_spectra_par.profile_label = widget_button(col2,value='Extract Intensity Profiles')
zstack_spectra_par.save_label = widget_button(col2,value='Save Images Menu')
;zstack_spectra_par.IDL_Slicer_label = widget_button(col2,value='IDL Slicer')
zstack_spectra_par.exit_label = widget_button(col2,value='Exit ZSTACK')
;;;;;;;;;;
Device, get_screen_size=screen_size
screen_center = [ screen_size(0) / 2 , screen_size(1) / 2 ]
geom = Widget_Info(zstack_spectra_par.main_base, /Geometry)
base_size = [geom.scr_xsize,geom.scr_ysize]
x_base_size = (fix(0.90*screen_size[0]) < base_size[0])
y_base_size = (fix(0.95*screen_size[1]) < base_size[1])
halfsize = [ x_base_size / 2 , y_base_size / 2 ]
widget_control, zstack_spectra_par.main_base, $
	XOffset = screen_center[0] - halfsize[0], $
	YOffset = screen_center[1] - halfsize[1], $
	scr_xsize = x_base_size, $
	scr_ysize = y_base_size
widget_control, zstack_spectra_par.main_base, /realize
; initialize image and spectra windows
zstack_spectra_par.main_base_win = !d.window
widget_control,zstack_spectra_par.image_label,get_value=window
zstack_spectra_par.image_win = window
IF (data_shifted EQ 1) THEN BEGIN
	widget_control,zstack_spectra_par.clipped_image_label,get_value=window
	zstack_spectra_par.clipped_image_win = window
ENDIF
widget_control,zstack_spectra_par.colorbar_label, get_value = window
zstack_spectra_par.colorbar_win = window
widget_control,zstack_spectra_par.spectra_plot_label,get_value=window
zstack_spectra_par.spectra_plot_win = window
; make image and plot widgets inactive
widget_control, zstack_spectra_par.image_label, sensitive = 0
IF (data_shifted EQ 1) THEN BEGIN
	widget_control, zstack_spectra_par.clipped_image_label, sensitive = 0
ENDIF
widget_control, zstack_spectra_par.plot_parameters_label, sensitive = 0
widget_control, zstack_spectra_par.spectra_plot_label, sensitive = 0
; initialize droplists
widget_control, zstack_spectra_par.i0_fileformat_list_label,$
		set_droplist_select=0
widget_control, zstack_spectra_par.spectra_dataformat_list_label,$
		set_droplist_select=0
widget_control, zstack_spectra_par.spectra_filetype_list_label,$
		set_droplist_select=0
widget_control, zstack_spectra_par.spectra_fileformat_list_label,$
		set_droplist_select=0
return
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_spectra, new_zoom=new_zoom, Group_Leader=groupLeader
;print,'zstack_spectra'

@axis_com
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

on_error,2
;IF NOT(keyword_set(new_zoom)) AND NOT(keyword_set(new_offset)) THEN BEGIN
IF NOT(keyword_set(new_zoom)) THEN BEGIN
	; Initialize variables
	spectrum = fltarr(15,n_files)
;	image_scale = 0
;	image_ratio = 0
;	spectrum_offset = 0.0
	spectra_filename_header = ''
	roi_filename = ''
	i_roi = 0
	i0_roi = 0
	n_roi = 0
	roi_index = 0
;	spectrum = 0
	i0_filename = ''
	i0_filetype = ''
	i0_scale_factor = 1.0
;	i0_reference_spectrum = ''
;	spec_filename_header = ''
;	save_filename_header = ''
	plot_x_min = 0.
	plot_x_max = 0.
	plot_y_min = 0
	plot_y_max = 0
	x_autoscale = 1
	y_autoscale = 1
;	display_spectra_type = [0,0]
;	; default: display_spectra_type = [0,0]
;		;	[0,*]		single beam
;		;	[1,*]		absorbance
;		;	[*,0]		SB/ABS set by program
;		;	[*,1]		SB/ABS set manually, do not override
	save_spectra_type = [0,0,0]
	; default: save_spectra_type = [2,2,2]
		;	[1,*,*]	single beam
		;	[2,*,*]	transmittance
		;	[3,*,*]	absorbance
		;	[*,1,*]	files with one spectrum each
		;	[*,2,*]	files with multiple spectra
		;	[*,*,1]	*.xas
		;	[*,*,2]	*.spc
		;	[*,*,3]	*.txt
		;	[*,*,4]	*.gif

;	IF (image_zoom LT 1) THEN BEGIN
;		image_zoom = 1	; zoom must be integer GE 1 for cw_defroi.pro
;		print,'Image zoom must be integer, reset to 1'
;	ENDIF
ENDIF
; Initialize zstack_spectra_common variables
;	i_roi = 0
;	i0_roi = 0
;	n_roi = 0
;	roi_index = 0
;	spectrum = 0
;	i0_filename = ''
;	i0_filetype = ''
;	spec_filename_header = ''
;	save_filename_header = ''
zstack_spectra_dialog, Group_Leader=groupLeader
widget_control, zstack_spectra_par.filename_display_list_label, $
		set_value = filename_display_list
widget_control, zstack_spectra_par.i0_scale_factor_label, $
		set_value = ' '+strtrim(string(i0_scale_factor,format='(f10.3)'),2)
; set initial value of spectra_filename_header if list_filename exists and spectra_filename_header is not already defined
IF ( (strlen(spectra_filename_header) EQ 0) AND (strlen(list_filename) NE 0) )THEN BEGIN
	spectra_filename_header = str_sep(list_filename,'.')
	spectra_filename_header = spectra_filename_header(0)
ENDIF
widget_control, zstack_spectra_par.spectra_filename_label, $
		set_value = ' '+spectra_filename_header
; set initial value of roi_filename if list_filename exists and roi_filename is not already defined
IF ( (strlen(roi_filename) EQ 0) AND (strlen(list_filename) NE 0) )THEN BEGIN
	roi_filename = str_sep(list_filename,'.')
	roi_filename = roi_filename(0)+'.roi'
ENDIF
widget_control, zstack_spectra_par.roi_filename_label, $
		set_value = ' '+roi_filename
zstack_spectra_sensitive
;dummy = where(spectrum NE 0,count)
;IF (count NE 0) THEN BEGIN
	zstack_spectra_makespectrum
	zstack_spectra_plotspectrum, displayed_file_index
;ENDIF
zstack_spectra_imgdisp, displayed_file_index
zstack_analyze_colorbar, zstack_spectra_par.colorbar_win
xmanager, 'zstack_spectra', zstack_spectra_par.main_base, $
  group_leader = zstack_spectra_par.main_base
return
end
