; Copyright (c)1998-2010 Jacob Stewart-Ornstein & A.P. Hitchcock  All rights reserved
;+
;NAME:
;		XSPECSIM.PRO
;
;LAST CHANGED: -----------------------------------  23-Jul-10
;
; PURPOSE:
;	This widget generates a simulated NEXAFS spectrum of a peptide or ptorein
;  from stored amino acid spectra (Yan Zubavichus,Andrey Shaporenko,
;  Michael Grunze and Michael Zharnikov, J. Phys. Chem A (Letters) 109, 2005, 6998-7000)
;  with provision for incorporation of a 'peptide bond correction'
;
; CATEGORY:
;	Data manipulation. Stand alone or runs from aXis2000
;
; XSpecSim routines
;	CANCEL			quit XSpecSim
;	CHKCLICK
;	SIMPLESUM
;	COMPLEX SUM		include peptide bond correction
;	LOADSEQUENCE
;	LOADENERGY
;	LOADEN
;	LOADAA
;	LOADDAT
;	DropListEncb
;	DropListDatcb
;
; ASSOOCIATED ROUTINES
; AA_WriteConfig  write a config file
; ExtrapolateAA   extrapolate to predefined E--scale
; GenExtrap       generaized ibterpolation
; ParseComposition determine numbers of H,C,N,O,S from AA list
; Read_AA_header  reader header of AA spectral files
; Read_AAdata     reads AA reference spectra (C, N,O)
; Read_Energies   reads energies of spectral reference
; ReadData        reads AA Data files
; XSpecSim_init   initialize environment for XSpecSim
; XSpecSim_run    run XSpec_Sim 
; 
; CALLING SEQUENCE:  XSpecSim
;
; INPUTS: via the gui: energy scale; amino acid sequence
;
; KEYWORDS: none
;
; OUTPUTS:
;	data file with simulated spectrum
;
; COMMON BLOCKS:
; axis_com     standard set of aXis2000 common blocks
;	XSPECSIM_COM	common for XSpecSim
;
; MODIFICATION HISTORY:
; (03-jan-07 jso) last version supplied
; (09-mar-07 aph) AXIS standard header added, name changed
; (20-Jul-10 aph) set up as a compile-when-run, in aXis2000
; (23-Jul-10 aph) modified SF_path to use that defined in XSpecSim_init
; 
;-

;-----------------------------------------------------------------
pro cancel, event
on_error,2
WIDGET_CONTROL, Event.Top, /DESTROY
end

;-----------------------------------------------------------------
pro chkClick, event
  @XSpecSim_Com

  if SaveSF eq 0 then begin
    SaveSF = 1
  endif else begin
    SaveSF = 0
  endelse

  aa_writeconfig
end
;-----------------------------------------------------------------
pro LoadEnergy, Event
@axis_com
@XSpecSim_Com
    filename = Dialog_Pickfile(/READ, /MUST_EXIST, Filter='*.enf', TITLE='Select Energy File To Load')
	IF filename NE "" THEN BEGIN
  		ENData = read_energies( File=filename, filter='*.enf')
  		if n_elements(endata) gt 0 and n_elements(aadata) gt 0 then begin
			ab = extrapolateaa(endata)
		endif
	ENDIF else begin
		Result = DIALOG_MESSAGE('a Valid file was not selected')
	endelse
end

;-----------------------------------------------------------------
pro Loaden, File=File
@axis_com
@XSpecSim_Com
	if not keyword_set(file) then begin
		file = Dialog_Pickfile(/READ, /MUST_EXIST, Filter='*.aah', TITLE='Select Amino Acid Data Files to Load')
	endif
	IF file NE "" THEN BEGIN
  		ENData = read_energies( File=file, filter='*.enf')
  		if n_elements(endata) gt 0 and n_elements(aadata) gt 0 then begin
			ab = extrapolateaa(endata)
		endif
	ENDIF else begin
		Result = DIALOG_MESSAGE('a Valid file was not selected')
	endelse
end

;-----------------------------------------------------------------
Pro LoadAA, Event
@axis_com
@XSpecSim_Com

    filename = Dialog_Pickfile(/READ, /MUST_EXIST, Filter='*.aah', TITLE='Select Amino Acid Data Files to Load')
	Loaddat, file=filename
end

pro LoadSequence, Event
@axis_com
 @XSpecSim_Com
  fname = pickfile2(Title='Select Sequence file')
  print, fname
  openr, iunit, fname, /get_lun
  text = ''
  line = ''
    while not  eof(iunit) do begin
      readf,iunit, line
      text = text + line
    endwhile
  close, iunit & free_lun, iunit

  text = strtrim(text,2)
  text = strupcase(text)
  text = strcompress(text)

  WIDGET_CONTROL, WIDtext, SET_VALUE=text
end

;-----------------------------------------------------------------
Pro LoadDat, file=file
@axis_com
@XSpecSim_Com
	if not keyword_set(file) then begin
		file = Dialog_Pickfile(/READ, /MUST_EXIST, Filter='*.aah', TITLE='Select Amino Acid Data Files to Load')
	endif
	AAList = read_AA_header(file=file)
    path = get_path(file)
    print, aalist
    AAdata = 	REPLICATE({Name:'alanine', Letter:'A', Formula:'', cSize:0, Energies:FLTARR(1000), Data:FLTARR(1000), carbon:0, nitrogen:0, oxygen:0, hydrogen:0, sulphur:0, phosphorus:0},N_ELEMENTS(aalist))
  	for i = 0, N_ELEMENTS(AAList)-1 do begin
  		AAen = read_AAData(file=path + '\' + AAList[i], filter='*.aas', letter=letterB, engdat=endata, Comp=Comp)
    	AAdata[i].letter = letterB
    	leng = n_elements(aaen[0,*])
    	AAdata[i].Energies = AAen[0,0:leng-1]
    	AAdata[i].Data = AAen[1,0:leng-1]
    	AAdata[i].cSize = leng
		aadata[i].carbon = comp.carbon
		aadata[i].nitrogen = comp.nitrogen
		aadata[i].oxygen = comp.oxygen
		aadata[i].hydrogen = comp.hydrogen
		aadata[i].sulphur = comp.sulphur
		aadata[i].phosphorus = comp.phosphorus
  	endfor

  	if n_elements(endata) GT 1 then begin
    	ab = extrapolateaa(endata)
  	endif

  print, aadata.letter
end

;-----------------------------------------------------------------
pro SimpleSum, Event
@axis_com
@XSpecSim_Com

  if (n_elements(aadata) lt 1) or (n_elements(endata) lt 1) then begin
    loaddat, file=datapaths[curdat]
    loaden, file=enpaths[curen]
	aa = aamoddata
	en = endata
  endif else begin
	aa = aamoddata
	en = endata
  endelse

  WIDGET_CONTROL, WIDtext, GET_VALUE=text
  print, text
  text = strtrim(text,2)
  text = strupcase(text)
  FinData = fltarr(1, n_elements(en))
  comp = {carbon:0, nitrogen:0, oxygen:0, hydrogen:0, sulphur:0, phosphorus:0}
  for i = 0, strlen(text[0])-1 do begin
	for j = 0, n_elements(AA)-1 do begin
	  	if AA[j].Letter eq strmid(text[0],i,1) then begin
			FinData = AA[j].data + FinData
			comp.carbon = comp.carbon + AAdata[j].carbon
			comp.nitrogen = comp.nitrogen + AAdata[j].nitrogen
			comp.oxygen = comp.oxygen+AAdata[j].oxygen
			comp.hydrogen = comp.hydrogen+AAdata[j].hydrogen
			comp.sulphur = comp.sulphur+AAdata[j].sulphur
			comp.phosphorus = comp.phosphorus+AAdata[j].phosphorus
		endif
	endfor
  endfor

findata = findata/strlen(text[0])

  comp.oxygen = comp.oxygen - strlen(text[0]) + 1 ;amide bond correction, lose 1 oxygen
  comp.hydrogen = comp.hydrogen - 2*strlen(text[0]) + 2;amide bond correction, lose 2 hydrogen
  sfcmpd = 'C' + string(comp.carbon) + ' N' + string(comp.nitrogen) + ' O' + string(comp.oxygen) + ' H' + string(comp.hydrogen) + ' S' + string(comp.sulphur) + ' P' + string(comp.phosphorus)
  sfcmpd = STRCOMPRESS(sfcmpd, /REMOVE_ALL)
  if n_elements(sf_path) EQ 0 then sf_path = 'c:\axis2000\sfdata\'
  SF_INIT, sf_path
  print, sfcmpd
  sfdat = sf(sfcmpd,en,density = 1, thickness = 0.001, result_type='trans', sf_dir=sf_path)
  sfdat = alog(1/sfdat)

  cedge = where((en gt 278) and (en lt 340))
  clow = where(en lt 283)
  chigh = where((en gt 315) and (en lt 340))
  nedge = where((en gt 380) and (en lt 480))
  nlow = where((en lt 398) and (en gt 380))
  nhigh = where((en gt 430) and (en lt 480))
  oedge = where((en gt 510) and (en lt 600))
  olow = where((en lt 525) and (en gt 480))
  ohigh = where((en gt 560) and (en lt 600))

	;cEdgeStuff
	if ((n_elements(clow) gt 1) and (n_elements(chigh) gt 1)) then begin
	totallength = n_elements(clow) + n_elements(chigh)

	;Float arrays for regression are assembled
	QarrX = fltarr(3,totalLength)
	QarrY = fltarr(totalLength)
	W = fltarr(totalLength)
	W[*]=1
	QarrX[0,0:n_elements(clow)-1] = en(clow)
	Qarrx[0,n_elements(clow):n_elements(clow)+n_elements(chigh)-1] = en(chigh)
	QarrX[1,0:n_elements(clow)-1] = findata(clow)
	Qarrx[1,n_elements(clow):n_elements(clow)+n_elements(chigh)-1] = findata(chigh)
	QarrX[2,*] = 1
	QarrY[0:n_elements(clow)-1] = sfdat(clow)
	QarrY[n_elements(clow):n_elements(clow)+n_elements(chigh)-1] = sfdat(chigh)

	result = regress2(QarrX, QarrY, W)
	SlopeE = result[0]
	Slope = result[1]
	Const = result[2]

	findata[cedge] = findata[cedge] * slope + en[cedge]*SlopeE + const
	endif

	;N Edge
	if ((n_elements(nlow) gt 1) and (n_elements(nhigh) gt 1)) then begin
	totallength = n_elements(nlow) + n_elements(nhigh)

	;Float arrays for regression are assembled
	QarrX = fltarr(3,totalLength)
	QarrY = fltarr(totalLength)
	W = fltarr(totalLength)
	W[*]=1
	QarrX[0,0:n_elements(nlow)-1] = en(nlow)
	Qarrx[0,n_elements(nlow):n_elements(nlow)+n_elements(nhigh)-1] = en(nhigh)
	QarrX[1,0:n_elements(nlow)-1] = findata(nlow)
	Qarrx[1,n_elements(nlow):n_elements(nlow)+n_elements(nhigh)-1] = findata(nhigh)
	QarrX[2,*] = 1
	QarrY[0:n_elements(nlow)-1] = sfdat(nlow)
	QarrY[n_elements(nlow):n_elements(nlow)+n_elements(nhigh)-1] = sfdat(nhigh)

	result = regress2(QarrX, QarrY, W)
	SlopeE = result[0]
	Slope = result[1]
	Const = result[2]

	findata[nedge] = findata[nedge] * slope + en[nedge]*SlopeE + const
	endif

	;O edge
	if ((n_elements(olow) gt 1) and (n_elements(ohigh) gt 1)) then begin
  	totallength = n_elements(olow) + n_elements(ohigh)
	;Float arrays for regression are assembled
	QarrX = fltarr(3,totalLength)
	QarrY = fltarr(totalLength)
	W = fltarr(totalLength)
	W[*]=1
	QarrX[0,0:n_elements(olow)-1] = en(olow)
	Qarrx[0,n_elements(olow):n_elements(olow)+n_elements(ohigh)-1] = en(ohigh)
	QarrX[1,0:n_elements(olow)-1] = findata(olow)
	Qarrx[1,n_elements(olow):n_elements(olow)+n_elements(ohigh)-1] = findata(ohigh)
	QarrX[2,*] = 1
	QarrY[0:n_elements(olow)-1] = sfdat(olow)
	QarrY[n_elements(olow):n_elements(olow)+n_elements(ohigh)-1] = sfdat(ohigh)

	result = regress2(QarrX, QarrY, W)
	SlopeE = result[0]
	Slope = result[1]
	Const = result[2]

	findata[oedge] = findata[oedge] * slope + en[oedge]*SlopeE + const
  	endif

  WIDGET_CONTROL, proteinname, GET_VALUE=pName
  plot, en, findata, TITLE=pname[0] + ' Spectra', XTITLE='eV', XRANGE = [en[0], en[n_elements(en)-1]], /XSTYLE
  oplot, en, sfdat

  spec = {t:'1d', x: en, d: findata, yl: 'absorbance (OD)', xl: 'Energy (eV)', dn: findata, dl: pname[0]}
  ;finalfile = spc_save(spec) ;needs axis running to use
  cfilepath = sav_AA(en=en, SpecData=findata,  sequence=otext, name = pname[0], group = XSpecSim_ID )

  ;WIDGET_CONTROL, chkSAVESF, GET_VALUE=SF_SAVE

  if savesf eq 1 then begin
  	energy = findgen(700 - 250) + 250
  	cfilepath = sav_AA(en=energy, data=alog(1/sf(sfcmpd,energy,density = 1, $
  	     thickness = 0.001, result_type='trans', sf_dir=sf_path))$
  	,  sequence=otext, name = 'SF_'+ pname[0], group = XSpecSim_ID)
  endif
end
;
;-----------------------------------------------------------------
pro ComplexSum, Event
@axis_com
@XSpecSim_Com

  if n_elements(aadata) lt 1 then begin
    loaddat, file=datapaths[curdat]
    loaden, file=enpaths[curen]
	aa = aamoddata
	en = endata
  endif else begin
	aa = aamoddata
	en = endata
  endelse

  WIDGET_CONTROL, WIDtext, GET_VALUE=text
  text = strtrim(text,2)
  text = strupcase(text)
  TextAdd = ''
  for i = 0, strlen(text[0])-1 do begin
	TextAdd=TextAdd + 'Z'
  endfor
  oText =text
  text = text + TextAdd

  comp = {carbon:0, nitrogen:0, oxygen:0, hydrogen:0, sulphur:0, phosphorus:0}
  FinData = fltarr(1, n_elements(en))
  for i = 0, strlen(text[0])-1 do begin
	for j = 0, n_elements(aamoddata)-1 do begin
	  	if AA[j].Letter eq strmid(text[0],i,1) then begin
			FinData = AA[j].data + FinData
			comp.carbon = comp.carbon + AAdata[j].carbon
			comp.nitrogen = comp.nitrogen + AAdata[j].nitrogen
			comp.oxygen = comp.oxygen + AAdata[j].oxygen
			comp.hydrogen = comp.hydrogen + AAdata[j].hydrogen
			comp.sulphur = comp.sulphur + AAdata[j].sulphur
			comp.phosphorus = comp.phosphorus + AAdata[j].phosphorus
		endif
	endfor
  endfor

  findata = findata/strlen(otext[0])

  comp.oxygen = comp.oxygen - strlen(otext[0]) + 1 ;amide bond correction, lose 1 oxygen for each amide bond
  comp.hydrogen = comp.hydrogen - 2*strlen(otext[0]) + 2;amide bond correction, lose 2 hydrogen for each amide bond
  sfcmpd = 'C' + string(comp.carbon) + ' N' + string(comp.nitrogen) + ' O' + string(comp.oxygen) + ' H' + string(comp.hydrogen) + ' S' + string(comp.sulphur) + ' P' + string(comp.sulphur)
  sfcmpd = STRCOMPRESS(sfcmpd, /REMOVE_ALL)
  SF_INIT, 'sfdata\'
  sfdat = sf(sfcmpd,en,density = 1, thickness = 0.001, result_type='trans', sf_dir='SFdata\') ;fix this fixed string
  sfdat = alog(1/sfdat)

  cedge = where((en gt 278) and (en lt 340))
  clow = where(en lt 283)
  chigh = where((en gt 315) and (en lt 340))
  nedge = where((en gt 380) and (en lt 480))
  nlow = where((en lt 398) and (en gt 380))
  nhigh = where((en gt 430) and (en lt 480))
  oedge = where((en gt 510) and (en lt 600))
  olow = where((en lt 525) and (en gt 480))
  ohigh = where((en gt 560) and (en lt 600))

	;cEdgeStuff
	if ((n_elements(clow) gt 1) and (n_elements(chigh) gt 1)) then begin
	totallength = n_elements(clow) + n_elements(chigh)

	;Float arrays for regression are assembled
	QarrX = fltarr(3,totalLength)
	QarrY = fltarr(totalLength)
	W = fltarr(totalLength)
	W[*]=1
	QarrX[0,0:n_elements(clow)-1] = en(clow)
	Qarrx[0,n_elements(clow):n_elements(clow)+n_elements(chigh)-1] = en(chigh)
	QarrX[1,0:n_elements(clow)-1] = findata(clow)
	Qarrx[1,n_elements(clow):n_elements(clow)+n_elements(chigh)-1] = findata(chigh)
	QarrX[2,*] = 1
	QarrY[0:n_elements(clow)-1] = sfdat(clow)
	QarrY[n_elements(clow):n_elements(clow)+n_elements(chigh)-1] = sfdat(chigh)

	result = regress2(QarrX, QarrY, W)
	SlopeE = result[0]
	Slope = result[1]
	Const = result[2]

	findata[cedge] = findata[cedge] * slope + en[cedge]*SlopeE + const
	endif

	;N Edge
	if ((n_elements(nlow) gt 1) and (n_elements(nhigh) gt 1)) then begin
	totallength = n_elements(nlow) + n_elements(nhigh)

	;Float arrays for regression are assembled
	QarrX = fltarr(3,totalLength)
	QarrY = fltarr(totalLength)
	W = fltarr(totalLength)
	W[*]=1
	QarrX[0,0:n_elements(nlow)-1] = en(nlow)
	Qarrx[0,n_elements(nlow):n_elements(nlow)+n_elements(nhigh)-1] = en(nhigh)
	QarrX[1,0:n_elements(nlow)-1] = findata(nlow)
	Qarrx[1,n_elements(nlow):n_elements(nlow)+n_elements(nhigh)-1] = findata(nhigh)
	QarrX[2,*] = 1
	QarrY[0:n_elements(nlow)-1] = sfdat(nlow)
	QarrY[n_elements(nlow):n_elements(nlow)+n_elements(nhigh)-1] = sfdat(nhigh)

	result = regress2(QarrX, QarrY, W)
	SlopeE = result[0]
	Slope = result[1]
	Const = result[2]

	findata[nedge] = findata[nedge] * slope + en[nedge]*SlopeE + const
	endif

	;O edge
	if ((n_elements(olow) gt 1) and (n_elements(ohigh) gt 1)) then begin
  	totallength = n_elements(olow) + n_elements(ohigh)
	;Float arrays for regression are assembled
	QarrX = fltarr(3,totalLength)
	QarrY = fltarr(totalLength)
	W = fltarr(totalLength)
	W[*]=1
	QarrX[0,0:n_elements(olow)-1] = en(olow)
	Qarrx[0,n_elements(olow):n_elements(olow)+n_elements(ohigh)-1] = en(ohigh)
	QarrX[1,0:n_elements(olow)-1] = findata(olow)
	Qarrx[1,n_elements(olow):n_elements(olow)+n_elements(ohigh)-1] = findata(ohigh)
	QarrX[2,*] = 1
	QarrY[0:n_elements(olow)-1] = sfdat(olow)
	QarrY[n_elements(olow):n_elements(olow)+n_elements(ohigh)-1] = sfdat(ohigh)

	result = regress2(QarrX, QarrY, W)
	SlopeE = result[0]
	Slope = result[1]
	Const = result[2]

	findata[oedge] = findata[oedge] * slope + en[oedge]*SlopeE + const
  	endif

  WIDGET_CONTROL, proteinname, GET_VALUE=pName
  plot, en, findata, TITLE=pname[0] + ' Spectra', XTITLE='eV', XRANGE = [en[0], en[n_elements(en)-1]], /XSTYLE
  oplot, en, sfdat
  cfilepath = sav_AA(en=en, SpecData=findata,  sequence=otext, name = pname[0], $
       modified='Peptide Bond Corrected', group = XSpecSim_ID)

 savesf=0
  if savesf eq 1 then begin
  	energy = findgen(700 - 250) + 250
  	cfilepath = sav_AA(en=energy, data=alog(1/sf(sfcmpd,energy,density = 1, $
  	      thickness = 0.001, result_type='trans', sf_dir=sf_path))$
  	,  sequence=otext, name = 'SF_'+ pname[0])
  endif
end

;-----------------------------------------------------------------
pro DropListEncb, event
@axis_com
@XSpecSim_Com
  curen = event.INDEX
  loaden, file=enpaths[curen]
  aa_writeconfig
end

;-----------------------------------------------------------------
pro DropListDatcb, event
@axis_com
@XSpecSim_Com
  curdat = event.index
  Loaddat, file=datapaths[curdat]
  aa_writeconfig
end

; -------------------------------------
; ************************************
; -------------------------------------

pro BASE_SPECSIM_event, Event
@axis_com
@XSpecSim_Com
  wWidget =  Event.top

  case Event.id of

    Widget_Info(wWidget, FIND_BY_UNAME='cmdSimpleSum'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        SimpleSum, Event
      end
    Widget_Info(wWidget, FIND_BY_UNAME='cmdComplexSum'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        ComplexSum, Event
      end
    Widget_Info(wWidget, FIND_BY_UNAME='cmdLoadEnergy'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        LoadEnergy, Event
      end
    Widget_Info(wWidget, FIND_BY_UNAME='cmdLoadAAData'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        LoadAA, Event
      end
    Widget_Info(wWidget, FIND_BY_UNAME='cmdLoadSequence'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        LoadSequence, Event
      end
    Widget_Info(wWidget, FIND_BY_UNAME='cmdDismiss'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        Cancel, Event
      end
    Widget_Info(wWidget, FIND_BY_UNAME='chkSaveSF'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        chkClick, Event
      end
    Widget_Info(wWidget, FIND_BY_UNAME='DropListData'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_DROPLIST' )then $
        DropListdatcb, Event
      end
    Widget_Info(wWidget, FIND_BY_UNAME='DropListEnergy'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_DROPLIST' )then $
        DropListencb, Event
      end
    else:
  endcase

end


;-----------------------------------------------------------------
pro BASE_SPECSIM, GROUP_LEADER=wGroup, _EXTRA=_VWBExtra_
@axis_com
@XSpecSim_Com

  BASE_SPECSIM = Widget_Base( GROUP_LEADER=wGroup, UNAME='BASE_SPECSIM'  $
      ,TITLE='X-SpecSim: peptides & proteins', XPAD=3, YPAD=3)
;      ,SCR_XSIZE=420 ,SCR_YSIZE=240 ,SPACE=3 ,XPAD=3,YPAD=3)

cmdLoadSequence = Widget_Button(BASE_SPECSIM, UNAME='cmdLoadSequence'  $
      ,XOFFSET=240 ,YOFFSET=5 ,SCR_XSIZE=120 ,SCR_YSIZE=20  $
      ,/ALIGN_CENTER ,VALUE='Load Sequence File')

  WID_LABEL_0 = Widget_Label(BASE_SPECSIM, UNAME='WID_LABEL_0'  $
      ,XOFFSET=10 ,YOFFSET=5  $ ; ,SCR_XSIZE=112 , SCR_YSIZE=25 $
      ,/ALIGN_LEFT ,VALUE='Amino Acid Sequence')

  WIDTEXT = Widget_Text(BASE_SPECSIM, UNAME='WID_TEXT_0' ,XOFFSET=1  $
      ,YOFFSET=30 ,SCR_XSIZE=370 ,SCR_YSIZE=75 ,/EDITABLE $
      ,XSIZE=20 ,YSIZE=1)

  WidDropDat = Widget_droplist(BASE_SPECSIM, UNAME='DropListData'  $
      ,XOFFSET=5 ,YOFFSET=120  $ ;,SCR_XSIZE=90 ,SCR_YSIZE=20  $
      ,/ALIGN_LEFT , VALUE=datafiles)

  WidDropEn = Widget_droplist(BASE_SPECSIM, UNAME='DropListEnergy'  $
      ,XOFFSET=120 ,YOFFSET=120 $; ,SCR_XSIZE=110 ,SCR_YSIZE=20  $
      ,/ALIGN_LEFT , VALUE=enfiles)

cmdLoadAAData = Widget_Button(BASE_SPECSIM, UNAME='cmdLoadAAData'  $
      ,XOFFSET=5 ,YOFFSET=150 $;  ,SCR_XSIZE=90 ,SCR_YSIZE=20  $
      ,/ALIGN_CENTER ,VALUE='Load AA Data')

  cmdLoadEnergy = Widget_Button(BASE_SPECSIM, UNAME='cmdLoadEnergy'  $
      ,XOFFSET=120 ,YOFFSET=150 $ ;,SCR_XSIZE=110 ,SCR_YSIZE=20  $
      ,/ALIGN_CENTER ,VALUE='Load Energy Range')

  WID_LABEL_1 = Widget_Label(BASE_SPECSIM, UNAME='WID_LABEL_1'  $
      ,XOFFSET=5 ,YOFFSET=180 $ ;,SCR_XSIZE=73 ,SCR_YSIZE=18  $
      ,/ALIGN_LEFT ,VALUE='Peptide Name:')

  ProteinName = Widget_Text(BASE_SPECSIM, UNAME='ProteinName_0'  $
      ,XOFFSET=77 ,YOFFSET=180 $ ; ,SCR_XSIZE=288 ,SCR_YSIZE=21  $
      ,/EDITABLE ,XSIZE=42 ,YSIZE=1)

  cmdSimpleSum = Widget_Button(BASE_SPECSIM, UNAME='cmdSimpleSum'  $
      ,XOFFSET=1 ,YOFFSET=210 $ ; ,SCR_XSIZE=73 ,SCR_YSIZE=20  $
      ,/ALIGN_LEFT ,VALUE='Uncorrected spectrum')

  cmdComplexSum = Widget_Button(BASE_SPECSIM, UNAME='cmdComplexSum'  $
      ,XOFFSET=140 ,YOFFSET=210 $ ;,SCR_XSIZE=80 ,SCR_YSIZE=20  $
      ,/ALIGN_Left ,VALUE='Peptide bond corrected spectrum')

cmdDismiss = Widget_Button(BASE_SPECSIM, UNAME='cmdDismiss'  $
      ,XOFFSET=320 ,YOFFSET=240 $ ; ,SCR_XSIZE=90, SCR_YSIZE=30   $
      ,/ALIGN_CENTER ,VALUE='Dismiss')

  WidBase1 = Widget_Base(BASE_SPECSIM, UNAME='WIDBASE1' ,XOFFSET=0  $
     ,YOFFSET=240 ,SCR_XSIZE=60 ,SCR_YSIZE=20 ,TITLE='IDL' ,COLUMN=1  $
    ,/NONEXCLUSIVE)

  chkSaveSF = Widget_Button(WidBase1, UNAME='chkSaveSF'  $
      ,XOFFSET=0 ,YOFFSET=0 $ ; ,SCR_XSIZE=100 ,SCR_YSIZE=20  $
      ,/ALIGN_LEFT ,VALUE='Save SF ')

  Widget_Control, /REALIZE, BASE_SPECSIM
  	Widget_Control, WidDropDat, SET_DROPLIST_SELECT=curdat
	widget_control, WidDropEn, SET_DROPLIST_SELECT=curen
	widget_control, chkSaveSF, SEt_Button=SaveSF

  XSpecSim_ID = Base_SpecSim

  XManager, 'BASE_SPECSIM', BASE_SPECSIM
end
;
; --------------------------------------------------------
;
pro XSpecSim, GROUP_LEADER=wGroup, _EXTRA=_VWBExtra_
;  BASE_SPECSIM, GROUP_LEADER=wGroup, _EXTRA=_VWBExtra_
@axis_com
@XSpecSim_Com
 XSpecSim_ID = long(0)

;XMANAGER, 'XSpecSim', XSpecSim_ID

BASE_SPECSIM, GROUP_LEADER=wGroup, _EXTRA=_VWBExtra_

end
