; Copyright (c) 1998-2021 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	READ_SGM_SPECTRA
;
;LAST CHANGED: ----------------------------------- 02 Mar-2021 (aph)
;
;PURPOSE:
;	This function reads  data  from an hdf5 file written by CLS SGM
; TEY, Io, clock, photodiode, and partial (or total FY can be read
; the SDD spectrum is displayed during read-in so the user can identify the limits for the peak of interest
;
;CATEGORY:
;	STAND ALONE or called in aXis2000:  spectral processing
;
;CALLING SEQUENCE:
; 	s = read_sgm(file=file,nof=nof, struct_only=struct_only, silent=silent)
;
;CALLED FROM AXIS:
;	Read~spectrum~SGM
;
;INPUTS: none
;
;KEYWORDS:
;   struct_only	= read & return structure
;	file		= name of file (if not in ncall, will launch dialog)
;	nof	= no name filter
;	silent		= do not print progress

;OUTPUTS:
;    returns spectrum
;
;COMMON BLOCKS:
;	axis_com
;
;MODIFICATION HISTORY:
; (26-Feb-21 aph) first version
; (02-Mar-21 aph) add read-in of user-selected FY spectra
;-

FUNCTION read_sgm,file=file,nof=nof, struct_only=struct_only, silent=silent

@axis_com
@sgm_com

on_error,2

; get filename if not supplied in call
fltr='*.hdf5'
if keyword_set(nof) then fltr='*'
title = 'Select SGM hdf5 data file'
if not keyword_set(file) then file = pickfile2(/READ, FILTER=fltr, /LPATH, DEFPATH=defpath, TITLE = TITLE)
; print, 'File to open is ', file
if strlen(file) EQ 0 THEN RETURN, 0

; use h5_parse to get data as an IDL structure
t = ax_name(file)  & fileshort = t(1)
if NOT keyword_set(silent) then print, 'Reading from file ', fileshort
a=h5_parse(file,/read_data)

; --- return the structure
IF keyword_set(struct_only) THEN RETURN, a


; ------- CHOOSE TYPE OF DATA to read
if n_elements(sgm_type) EQ 0 then sgm_type = 'T'

if widget_info(/active) EQ 0 then $
	sgm_type = get_text(prompt = '<T>ey, <I>o, <C>lock,<P>hotodiode, <F>luorescece yield', val=sgm_type) $
	else $
		sgm_type = get_text(prompt = '<T>ey, <I>o, <C>lock,<P>hotodiode, <F>luorescece yield', val=sgm_type, group = aXis_ID)
sgm_type=strmid(sgm_type,0,1)
sgm_type =strupcase(sgm_type)

; ---------- get energy  NB first point reads as NaN so remove it
	ev = reform(a.entry1.data.en._data)
	ev = ev(1:n_elements(ev)-1)

CASE sgm_type OF

'T': BEGIN
	;  --------- get the TEY data
	tey = reform(a.entry1.data.tey._data)
	tey = tey(1:n_elements(tey)-1)
	; -------- construct spectral file
	dl = fileshort + ' tey'
	tey={t:'1d', d:tey, x: ev, xl: 'Energy (eV)', dn: tey, dl: dl }
	axis_log, 'SGM TEY spectrum read from ' + fileshort

	return, tey
	END

'I': BEGIN
	;  --------- get the I0  data
	tey = reform(a.entry1.data.I0._data)
	tey = tey(1:n_elements(tey)-1)
	; -------- construct spectral file
	dl = fileshort + ' I0'
	io={t:'1d', d:tey, x: ev, xl: 'Energy (eV)', dn: tey, dl: dl }
	axis_log, 'SGM I0 spectrum read from ' + fileshort

	return, io
	END

'C': BEGIN
	;  --------- get the CLOCK data
	tey = reform(a.entry1.data.clock._data)
	tey = tey(1:n_elements(tey)-1)
	; -------- construct spectral file
	dl = fileshort + ' clock (s)'
	clock={t:'1d', d:tey, x: ev, xl: 'Energy (eV)', dn: tey, dl: dl }
	axis_log, 'SGM CLOCK signal read from ' + fileshort

	return, clock
	END

'P': BEGIN
	;  --------- get the Photodiode data
	tey = reform(a.entry1.data.PD._data)
	tey = tey(1:n_elements(tey)-1)
	; -------- construct spectral file
	dl = fileshort + ' photodiode '
	pd={t:'1d', d:tey, x: ev, xl: 'Energy (eV)', dn: tey, dl: dl }
	axis_log, 'SGM photodiode signal read from ' + fileshort

	return, pd
	END


'F': BEGIN

	if n_elements(ssd_ch) EQ 0 then ssd_ch = '4'
	if widget_info(/active) EQ 0 then $
	ssd_ch = get_text(prompt = 'SDD<1>,SDD<2>,SDD<3>,SDD<4> or <A>ll', val=ssd_ch) $
	else $
		ssd_ch = get_text(prompt = 'SDD<1>,SDD<2>,SDD<3>,SDD<4> or <S>um', val=ssd_ch, group = aXis_ID)
	ssd_ch=strmid(ssd_ch,0,1)

;	----------- read average SDD spectrum for requested SDD
	CASE ssd_ch OF

	'1':	sdd = a.entry1.data.sdd1._data
	'2':	sdd = a.entry1.data.sdd2._data
	'3':	sdd = a.entry1.data.sdd3._data
	'4':	sdd = a.entry1.data.sdd4._data
	ELSE: BEGIN
			sdd = a.entry1.data.sdd1._data
			sdd = sdd + a.entry1.data.sdd2._data
			sdd = sdd + a.entry1.data.sdd3._data
			sdd = sdd + a.entry1.data.sdd4._data
		  END
	ENDCASE

	t = size(sdd)
	n_x = t(1)
	n_e = t(2)
	x = findgen(n_x)
	counts = reform(sdd(*,n_e-1))
;	plot,x, counts

	; -------- construct SDD fluorescence file
	dl = fileshort + ' SDD ' + ssd_ch
	f={t:'1d', d:counts, x: x, xl: 'channel', dn: counts, dl: dl }
	axis_log, 'SDD '+ ssd_ch + ' average fluorescence  read from ' + fileshort

;	return,f
	if widget_info(/active) EQ 0 then begin
		splot, f
	endif else begin
		CurBuf_tmp = CurBuf
		HANDLE_VALUE, Data(0), f, /SET
		Label(0) = f.dl
		PlotBuf, 0
		CurBuf = CurBuf_tmp
	endelse

; ---------  plot and ask user to select integration limits
	if n_elements(ch_lo) EQ 0 then ch_lo  = 1
	if widget_info(/active) EQ 0 then $
		ch_lo = fix(get_num(prompt='lower limit', val=ch_lo)) $
		else  ch_lo = fix(get_num(prompt='lower limit', val=ch_lo, group = axis_ID))
	if n_elements(ch_hi) EQ 0 then ch_hi= n_x - 1
	if widget_info(/active) EQ 0 then $
		ch_hi = fix(get_num(prompt='lower limit', val=ch_hi)) $
			else ch_hi = fix(get_num(prompt='lower limit', val=ch_hi, group = axis_ID))

; --------- integrate over user selected limits
	fy = fltarr(n_e-1)
	for i = 0, n_e-2 do begin
		fy(i) = reform(total(sdd(ch_lo:ch_hi,i)))
	endfor

; -------- construct spectral file
	dl = fileshort + ' PFY from SDD ' + ssd_ch + '  I: ' + string(ch_lo, format='(I3)') + ' - ' + string(ch_hi, format='(I3)')
	t= reform(a.entry1.data.en._data)
	ev = t(1:n_e-2)
	fy = fy(1:n_elements(fy)-1)
	tmp = {t:'1d', d: fy, x: ev, xl: 'Photon energy (eV)', dn: fy, dl: dl }
	if ssd_ch EQ 'S' then t = 'Sum' else t = ssd_ch
	axis_log, 'SDD '+ t + ' partial fluorescence yield read from ' + fileshort

	return,tmp

	END


ELSE: print, 'invalid data identifier'


ENDCASE

END
