; Copyright (c) 1996-2021 ER, JDD, A.P. Hitchcock  All rights reserved
;+
;NAME:
;		SPLOT2D.PRO
;
;LAST CHANGED: ------------------------------ 	05 Jul 2021
;
; PURPOSE:
;	This procedure plots a 2D data structure (image) in an IDL window
;
; CATEGORY:
;	Image display. stand alone or inside aXis2000
;
; CALLING SEQUENCE:
;	SPLOT2D, s, [ASPECT=asp,ZOOM=zoom, _extra=e]
;
; INPUTS:
;  	 s	2D structure
;    basic input data structure: s={d:, x:, y:}
;     optional axis and title label fields include {xl:, yl:, dl:}
;
; KEYWORDS:
;	ASPECT	rescale window vertical size to
;            aspect ratio similar to the image dimensions
;	ZOOM	allows user to define x,y ranges
;   extra	pass along PLOT options
;
; REQUIREMENTS:  graph.pro MUST be compiled prior
;
; OUTPUTS:
;	No explicit outputs.   A new window is created if necessary
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;
; MODIFICATION HISTORY:
;  (19-Feb-96 jdd) initial creation
;  (15-Jun-96 jdd) added generalized interpretation of structures
;        in SPLOT, SPLOT2D, SPLOT3D
;        separated splot stuff from graph.pro
; (29-Jun-9 aph) allow numerical limts on ranges
; (06-Dec-97 aph) remove print of ytitle
; (03-jan-99 aph) add back in YTitle (!!)
; (24-jun-99 aph) consistent image/spectra color tables
; (13-aug-99 aph) no change - just using debugs to track reform problem
; (14-nov-00 cgz) migrated from SPLOTA.PRO
; (15-nov-00 cgz) Replaced call of NCOLS function by explicit (SIZE(var))(1)
; (30-nov-00 cgz) Replaced call of WSIZE function by explicit [!D.X_SIZE,!D.Y_SIZE]
; (30-nov-00 cgz) Replaced call of nwin with explicit window statement
; (21-Jul-14 aph) AXIS standard header added
; (05-Jul-21 aph) shorten displayed part of header so all to be visible
;-

pro splot2d, s, ASPECT=asp,ZOOM=zoom, _extra=e
@axis_com
on_error,2

dl='' & xl='' & yl=''
fexist=stag(s, ['dl', 'xl', 'yl'])
if fexist(0) then dl=s.dl
if fexist(1) then xl=s.xl
if fexist(2) then yl=s.yl

; determine indices of s.d within xrange, yrange
nxb=0
xmin=s.x(0)
;nxe=ncols(s.x)-1
nxe=(size(s.x))(1)-1
xmax=s.x(nxe)
nyb=0
ymin=s.y(0)
;nye=ncols(s.y)-1
nye=(size(s.y))(1)-1
ymax=s.y(nye)
if keyword_set(zoom) then begin
	prom=string(s.x(0)) + ' Xmin:'
	READ, xmin, PROMPT= prom
	prom=string(s.x(nxe)) + ' Xmax:'
	READ, xmax, PROMPT= prom
	prom=string(s.y(0)) + ' Ymin:'
	READ, ymin, PROMPT= prom
	prom=string(s.y(nye)) + ' Ymax:'
	READ, ymax, PROMPT= prom
;	for i=0, ncols(s.x)-1 do begin
	for i=0, (size(s.x))(1)-1 do begin
	   if s.x(i) le xmin then nxb=i
	   if s.x(i) lt xmax then nxe=i
	   if s.y(i) le ymin then nyb=i
	   if s.y(i) lt ymax then nye=i
	endfor
endif

if keyword_set(asp) then begin
   sz=size(s.d)
   dx=abs(xmax-xmin)
   dy=abs(ymax-ymin)
   aspect=dy/dx*1.2
   print, dx, dy, aspect/1.2
;   ws=wsize()
	ws = [ !D.X_SIZE, !D.Y_SIZE ]
;   nwin, !d.window+(!d.window eq -1), ws(0), round(ws(0)*aspect)
   window, !d.window>0, xsize=ws(0), ysize=round(ws(0)*aspect)
endif

; ------ only print that part of the header (s.dl) that fits in space available
;     scalefactor 1.4 can fit 60 characters; 1.0 can fit 40 char;  2.0 can fit 70 char (cjeck 5-Jul-21)
maxchar = fix(37*gr_scale)
dls = strmid(dl, 0, maxchar)
print, maxchar, '   ',  dls
plot2d, s.d(nxb:nxe,nyb:nye), s.x(nxb:nxe), s.y(nyb:nye), [xl, yl], title=dls, _extra=e

end


