; Copyright (c) 1998-2017 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		READ_DUV_STACK
;
;LAST CHANGED: ----------------------------------- 	19 Jun 2017
;
; PURPOSE:
;	This function reads a stack of Raman or fluorescence spectra
; measured at an array of (x,y) positions into an axis standard stack
; It also reads the sub-image for the stack and writes it as a axis image file
;
; CATEGORY:
;	Read-in. Runs stand alone or inside aXis2000.
;
; CALLING SEQUENCE:
;	tmp = READ_DUV_STACK(file=file)
;
; INPUTS: none	.
;
; KEYWORDS:
;	GROUP	group ID
;   SHOW    show spectrum
;
; OUTPUTS:
;	*.(dat,ncb) written
;
; COMMON BLOCKS:
; AXIS.COM    standard set of common blocks
; DUV_COM     DUV data specific common block
;  volume_Data
; BSIF_COM
; stack_process_com
;
; PROCEDURE:
;	Reads (x,y) positions from position file (NAME_positions.txt
; (FIRST convert from Excel to tab delimited text file)
;   Reads wavenumber values then set of Raman spectra from NAME.txt)
;   sets up (x,y,E) stack  (nx, ny, 1024)
;
; NB if want to use only a sub-set of full Raman spectrum need to
;    identify # of wavenmbers measured

; MODIFICATION HISTORY:
; (19-Jul-17 aph) first version
;-

FUNCTION  READ_DUV_STACK, show = show, group = group
@duv_com
@axis_com
@stack_process_com
COMMON volume_data, image_stack
@bsif_com

on_error,2

null = 0

fname = pickfile2(title='Select name of DUV_Raman stack',FILTER='*.txt')
if strlen(fname) EQ 0 then RETURN, null

t = ax_name(fname)
pos_file = t(0) + t(1)+ '_positions' + '.txt'

; print, 'Reading position file ', pos_file

tmp = read_duv_positions(file=pos_file)
if tmp EQ 0 then return, null

axis_log, ' stack size = ' + strtrim(string(duv_nx),2) +' * ' + strtrim(string(duv_ny),2)
; print, 'X-values', duv_x
; print, 'Y-values', duv_y

 openr, unit, fname, /get_lun
 n_ev = 1024						;Change or read-in if < 1024 wavenumbers
 ev = fltarr(n_ev)
 image_stack=fltarr(duv_nx, duv_ny, n_ev)

; ------ read wavenumber axis
 readf, unit, ev
; read Raman spectra
 spectrum=intarr(n_ev)
 t = 0.
; ----- read set of spectra
 for j = 0, duv_ny-1 do begin
 	for i = 0, duv_nx-1 do begin
 		readf, unit, t, spectrum
		if keyword_set(show) then plot,ev, spectrum
 		image_stack(i,j,*) = spectrum
; 		print, i, j, '     ', laser_eff(i,j), t
 	endfor
 endfor
 close, unit &  free_lun,unit

; ---- construct and write stack file

x_start= duv_x(0) & x_stop = duv_x(n_elements(duv_x)-1)
y_start= duv_y(0) & y_stop = duv_y(n_elements(duv_y)-1)
n_img = n_elements(ev)
duv_e = 10.
duv_dwell = 10.
FILENAME_EV_MSEC_LIST = strarr(n_img)
for i = 0, n_img-1 do begin
	FILENAME_EV_MSEC_LIST(i) = string(i+1) + ' ' + string(ev(i)) + ' 10 sec'
endfor

t = ax_name(fname)
sname = t(0)+t(1)+'.ncb'
sname=pickfile2(title='Choose name of stack file', file = sname, filter='*.ncb')

if sname NE '' then stack_wb, sname

; -------- write sub image -------
sub_image = pickfile2(title='Select sub-image (*.tif) file', filter='*.tif')
if strlen(sub_image) GT 1 then begin		; bail if used cancels
	tmp = img_load(file = sub_image, DEFPATH=defpath, /TIF, /VALUES)
	if n_tags(tmp) EQ 0 then return, null 		; bail if used cancels
	dl = t(1) + ' sub image'
	duv_e = 10.
	duv_dwell = 10.
	xl = ' x (um)   dwell (s) = ' + strtrim(string(duv_dwell),2)
	yl = ' y (um) PIXEL NUMBERS'
	dl = 'SUB_IMAGE for ' + t(1)
	tmp2 = {t:'2d', x:tmp.x, xl: xl, y:tmp.y, yl: yl, e:duv_e, d:tmp.d, dl:dl}
	lname = t(0)+t(1)+'.axb'
	t = axb_save(tmp2, file=lname)
	return, tmp2
endif

END
