; Copyright (c) 1998-2026 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	AX_CHEMI_MOVIE.PRO
;
;
;LAST CHANGED: ----SECOND TRY ------------------------------- 25-Dec-25  (04 Mar 2023 (aph)

; PURPOSE:
;	This procedure processes a list of stacks in a macro fashion
;   to generate time or potential series of component maps and color composites
;
; CATEGORY:
;	    Runs stand alone or inside aXis2000.
;
; CALLING SEQUENCE:
;	 AX_CHEMI_MOVIE, parameters
;
; INPUTS:
;	FILE (*.lst)	a text file containing (A)parameters (last keyword in parameter list must be 'end')
;                              and (B) names of stacks
;
; EXAMPLE:
;; ###### PARAMETERS
; input_source D:\aXis-Pass-Test-data\09-UTILITIES\05-EXECUTE MACRO\Cu_CO2R_pot_series\input\
; roi_io Y:\data\XRM\stxm_Bessy\Maxymus\2023\23-02\02-05\MPI_230205129\129j-io.roi
; ecal 0.85
; coeff Y:\data\XRM\stxm_Bessy\Maxymus\2023\23-02\Cu2p-Jiang-ref\Cu-Cu2O-CuO_cal2-Jiang-OD1.par
; map_clip N
; map_name 3svd_odj2		used to define type of maping (3svd or 3sf) and the reference spectra par file used
; end
;
; KEYWORDS:
;	OVERWRITE:  if set, do not check for existence of (*.ncb) but instead, overwrite it
;	VERBOSE:	if set, print the parameters and filenames of the stacks
;
; OUTPUTS:
;  analysed stacks in the sequence of stacks
;
; COMMON BLOCKS:
; AXIS_COM    standard set of common blocks
; STACK_PROCESS_COM
; BSIF_COM
; SDF.COM
; volume_data, image_stack
;
; MODIFICATION HISTORY:
; (24-Feb-23 aph) first version
; (04-Mar-23 aph) return defaults at end
; (17-May-23 aph) format NAME for html Help file
; (21-Dec-25 aph) revive ax_chemi_movie (try #2)
; (25-Dec-25 aph) improve user feedback (with, without verbose); add input_source parameter to make *.lst easier to read
;-

PRO AX_CHEMI_MOVIE, file = file, map_copy=map_copy, ncomp = ncomp, overwrite = overwrite, verbose = verbose, axis = axis

on_error,2
@AXIS_COM
@STACK_PROCESS_COM
@BSIF_COM
@sdf_com
COMMON volume_data, image_stack


; ------- if file not provided, allow user to navigate & select
if not keyword_set(file) then begin
	fltr='*.lst'
	file = pickfile2(/read, FILTER=fltr, /LPATH, DefPath=DefPath)
endif

; --- check for file existence
check = findfile(file)
if strlen(check(0)) EQ 0 then begin
	axis_log, file + ' not found. AX_CHEMI_MOVIE terminated'
	return
endif

; ----- read parameters
openr, iunit, file, /get_lun
text = ' '
test = ';'
while test EQ ';' DO BEGIN	; skip comment lines which begin with ;
	readf, iunit, text
	test = strtrim(strmid(text, 0,  1),2)
endwhile

roi_io = ' '
if keyword_set(verbose) then print, '######## PARAMETERS #########'
while text NE 'end' do begin
	textn = strmid(text, 0,  strpos(text,' '))
	textval = strmid(text, strpos(text,' ')+1)
	if keyword_set(verbose) then print, textn, ' = ', textval
	if textn EQ 'input_source' then input_source = textval
	if textn EQ 'roi_io' then roi_io = textval
	if textn EQ 'ecal' then ecal = float(textval)
	if textn EQ 'coeff' then coeff = textval
	if textn EQ 'map_clip' then map_clip = textval
	if textn EQ 'map_name' then map_name = textval
	if textn EQ 'map_copy' then map_copy = textval
;	if textn EQ 'output_folder' then output_folder = textval
	readf, iunit, text
endwhile

; ------- read names of stacks
text = ''
stack_files=strarr(1)
i = 0
while ~EOF(iunit) EQ 1 do begin
	readf, iunit, text
	stack_files(i) = input_source + text
	stack_files=[stack_files,' ']
	i = i+1
endwhile
stack_files = stack_files(0:n_elements(stack_files)-2)		; remove last line
n_stacks = i
if keyword_set(verbose) then print, 'In   ', file, '      there are ', n_stacks , '  stacks'

if keyword_set(verbose) then begin
	print, ' '
	print, '######## STACK FILES #########'
	for i = 0, n_elements(stack_files)-1 do begin
		print, i+1, '  ', stack_files(i)
	endfor
endif
close, iunit & free_lun, iunit

; -------- turn off i-ring normalization
norm_flag = 0

; ------------ identify the number of component maps being generated
pars = ax_par_load(coeff, /silent)
n_comps = pars.n
comp_names = pars.names

if keyword_set(verbose) then begin
	print, ' '
	print, '######## COMPONENT SPECTRA #########'
	for i = 0, n_elements(comp_names)-1 do begin
		print, i+1, '  ', comp_names(i)
	endfor
endif
; ***************************************************************************
; ----- generate  stack_list arrays for the sequence of component maps
; ***************************************************************************
if keyword_set(map_copy) then begin
	sl_files = strarr(n_comps)
	if keyword_set(verbose) then print, 'map_copy (output folder) is: ', map_copy
	for j = 0, n_comps-1 do begin
		sl_files(j) = map_copy + map_name + '_' + comp_names(j) + '.sl'
		if keyword_set(verbose) then print, 'Name of ', j+1, '   stack list file  is  ',  sl_files(j)
	endfor
endif

; ***************************************************************************
; ----- process *.hdr (sdf data format) stacks to component maps
; ***************************************************************************
; if keyword_set(map_copy) then begin


stack_ODC_name = strarr(n_stacks)

; 	***************   KLUGE !!!!  to get around 'disappearance of map_copy (Dec 22, 2025)	***************
;	24-Dec-2025: map_copy is the folder to contain *.sl and component maps
print_file = map_copy	; store the string  in an unused text variable in axis.com

; --- loop to process each file ------------------------------------------------------------
	print, '#####################################'
	print, '******* ax_chemi_movie **************'

for i = 0, n_elements(stack_files)-1 do begin
	map_copy =print_file
	if keyword_set(verbose) then print, i, 'map_copy ' , map_copy
	stackn = stack_files(i)
	t = ax_name(stackn)
	file_short = t(1)
	a=read_sdf(stackn, channel=1, /auto_save,  /silent)
	t = ax_name(a)
	print, file_short,'   converted to ', t(1) + '.' + t(2)
	file_short = t(1)
	print, '_____________________'

b  = ax_convert_to_odc (stack=a, roi_io = roi_io, ecal=0.85)
	t = ax_name(b)
	stack_ODC_name(i) = t(1)
	file_short = t(1)
	print, file_short,'   converted to OD; energy calibrated ; wrote to ', t(1) + '.' + t(2)
	print, '_____________________'

	map_name_t = file_short + '_' + map_name	; (Dec25) call ax_svd without map_copy ************************
	ax_svd, stack=b, coeff=coeff, map_clip = 'N', map_name = map_name_t, map_copy = map_copy, $ ***** Dec25
	         overwrite=1, /no_res_stack,  /silent, axis = axis ; /verbose ; ,
	print, file_short, '   converted to ', map_name_t ,' component maps ', comp_names

	print, '_____________________'
	if keyword_Set(axis) then begin
		t = ax_name(stackn)
		axis_log, t(1) + '  converted to maps'
	endif
endfor

; *************** write component file names to the stack list file ***************
; print, 'stack_ODC_name is ',  stack_ODC_name
; if keyword_set(map_copy) then begin
	map_copy = print_file		; restore map_copy
	if keyword_set(verbose) then print, 'writing stack list files to ', map_copy
 	for i = 0, n_comps-1 do begin
		openw,unit, sl_files(i), /get_lun
	 	printf, unit, map_copy 		;  ------- write path at top
		for j = 0, n_stacks-1 do begin
			mapj = stack_ODC_name(j) + '_' + map_name + '_' + strtrim(strcompress(comp_names(i)),2) + '.axb'
	 		printf, unit, mapj	;  write stack_list file for component i
	 		print, mapj
	 	endfor
 		close, unit & free_lun, unit     ;close & free unit for  component i
 	endfor
; endif

; goto, finish

; map_copy_bypass:

; ***************************************************************************
; ----- transfer component maps to map_copy (output folder)
; ************************************************************* **************
;
;
;	print, '_____________________'
;	print,  ' stack_ODC_name(s) ',  stack_ODC_name
;	print, '  '
;;
;
;for i = 0, n_elements(stack_files)-1 do begin
;	for i = 0, ncomp-1 do begin
;		print, stack_ODC_name(i)
;;		file_copy, file
;	endfor
;endfor

; ************************************************************* **************

; finish:

; --------- return variables to default so use of ax_sdf works properly in aXis2000
axis = 1
overwrite = 0

END