; Copyright (c) 1998-2022 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	AX_FIX_ROLLOVER
;
;LAST CHANGED: ----------------------------------- 09 Dec 2022 (aph)
;
;PURPOSE:
;	This function modifies one image or a sequence of images
; by identifying all adjacent points where  I(n) - I(n-1) exceeds a CHANGE THRESHOLD
; and then replaces the I(n) value with  (In value + OFFSET)
; This can be used to repair roll-over issues such as phase wrapping (phase reported as modulo(2pi)
; or 2**15 rollovers that arise in conversion of some image files
; typically, those which are read as unsigned integers rather than signed ones
;
;CATEGORY:
;	AXIS: image analysis  (does not run stand alone)
;
;CALLING SEQUENCE:
;	Result = AX_FIX_ROLLOVER (ONE=ONE, SIGN=SIGN, PHASE=PHASE, USER=USER, STACK=STACK)
;
;CALLED FROM AXIS:
;	->Images->Fix rollover or stacks~Fix rollover

;REQUIRED INPUTS: none
;
;KEYWORDS:
;	ONE		flag - process current image in aXis2000
;	PHASE	flag - use threshold and offset parameters for a 2pi phase rollover
;	SIGN	flag- if set, use threshold and offset parameters for 2**15 roll over
;	STACK	process a stack  (default for NOT one is to process files listed in a *.sl stack list file)
;	USER	flag - user defines threshold and offset
;
;
;OUTPUTS:
;	modified files are written to buffer 0 if /ONE or as a stack to disk if .NOT. /ONE
;

;COMMON BLOCKS:
;	@AXIS_COM	standard set of common blocks
;	@BSIF_COM	for *.nc file read/write (no longer used)
;	@stack_process_com
;	COMMON volume_data, image_stack
;
; ADDITIONAL PRO in file:
; ^^^^^^^^^^^^^^^^^^^^^^^
; FUNCTION  FIX_ROLLOVER, TMP, THRESHOLD=THRESHOLD, OFFSET=OFFSET
; executes the search anf change, first along columns, then along rows, then again along columns
;
;MODIFICATION HISTORY:
; (04-Jul-00 aph) first version
; (04-Jan-15 aph) add keyword, one, an option to process 1 file
; (24-Jun-21 aph) change to use *.axb files; allow user to select multiple files; adapt for roll over by 2pi (phase) - NOT COMPLETED
; (08-Jan-22 aph) allow user to define threshold and offset
; (29-May-22 aph) get working for 2pi phase unwrap; change from *.sl to stack;  put change in seperate function
; (09-Dec-22 aph) try again using CLS phase image of Ni-N-C-high
;-


FUNCTION  fix_rollover, tmp, threshold=threshold, offset=offset
@axis_com
@bsif_com

on_error,2

; scan each column to find changes of adjacent rows which are larger in magnitude than the threshold
; e.g.
;  for 2-pi phase roll-over a change of +3.14 means subtract 2pi (6.28318530718)
;         from following points in the current row  until next time threshold is exceded
;  for 2-pi phase roll-over a change of -3.14 means need to add 2pi (6.28318530718)
;         from following points in the current row  until next time threshold is exceded

 	for i = 0, n_elements(tmp.x)-1 do begin
 		change = 0
 		for j = 1, n_elements(tmp.y)-2  do begin
 			if abs(tmp.d(i,j) - tmp.d(i,j-1)) GT threshold then begin
 ;				print, i, '   ',j, '     ', tmp.d(i,j+1) - tmp.d(i,j)
 			      if tmp.d(i,j) - tmp.d(i,j-1) GE 0 then change = -1.*offset else  change = offset
 			endif
 			tmp.d(i,j) = tmp.d(i,j) + change
 			change = 0
 		endfor
	endfor

; NOW do the same from the x-direction
 	for j = 0, n_elements(tmp.y)-1 do begin
 		change = 0
 		for i = 1, n_elements(tmp.x)-2  do begin
 			if abs(tmp.d(i,j) - tmp.d(i-1,j)) GT threshold then begin
 ;				print, i, '   ',j, '     ', tmp.d(i,j+1) - tmp.d(i,j)
 			    if tmp.d(i,j) - tmp.d(i-1,j) GE 0 then change = -1.*offset else  change = offset
 			endif
 			tmp.d(i,j) = tmp.d(i,j) + change
 			change = 0
 		endfor
	endfor

; NOW do the same from the y-direction again with smaller threshold
		threshold = 0.8*threshold
	 	for i = 0, n_elements(tmp.x)-1 do begin
	 		change = 0
	 		for j = 1, n_elements(tmp.y)-2  do begin
	 			if abs(tmp.d(i,j) - tmp.d(i,j-1)) GT threshold then begin
 ;				print, i, '   ',j, '     ', tmp.d(i,j+1) - tmp.d(i,j)
	 			      if tmp.d(i,j) - tmp.d(i,j-1) GE 0 then change = -1.*offset else  change = offset
	 			endif
	 			tmp.d(i,j) = tmp.d(i,j) + change
	 			change = 0
	 		endfor
		endfor

; NOW do the same from the y-direction again  with smaller threshold
 	for j = 0, n_elements(tmp.y)-1 do begin
 		change = 0
 		for i = 1, n_elements(tmp.x)-2  do begin
 			if abs(tmp.d(i,j) - tmp.d(i-1,j)) GT threshold then begin
; 				print, i, '   ',j, '     ', tmp.d(i,j+1) - tmp.d(i,j)
 			    if tmp.d(i,j) - tmp.d(i-1,j) GE 0 then change = -1.*offset else  change = offset
 			endif
 			tmp.d(i,j) = tmp.d(i,j) + change
 			change = 0
 		endfor
	endfor

; -------------- annotate and return result
	tmp.dl = tmp.dl + ' FixR'
	return, tmp
END


; ---------------- framework

FUNCTION ax_fix_rollover, one=one, sign=sign, phase=phase, user=user, stack=stack

@axis_com
@bsif_com
@stack_process_com
COMMON volume_data, image_stack

on_error,2


s = 0	; default return ==>    did not execute
; --- default to phase rollower
	low_limit = 3
	roll_over = 6.28318530718 		; 2pi = 6.28318530718

; ------- define threshold and offset
IF keyword_set(phase) then begin			; user defined low_limit and roll_over increment
	low_limit = 2.2
	roll_over = 3.14159   ;		; 2pi = 6.28318530718
endif
IF 	keyword_set(sign) then begin		; default for 2**16 - 1 rollover
	low_limit = 0
	roll_over = 65535.
ENDIF
threshold = low_limit
offset = roll_over

; ------- use TOF parameters to (re)-store threshold & offset
if n_elements(stop1) GT 0 then low_limit = stop1  ; else threshold = low_limit
if n_elements(stop2) GT 0 then roll_over = stop2  ; else offset = roll_over


IF keyword_set(user) then begin
	 threshold = get_num(Prompt = 'Threshold for CHANGE', Val=low_limit, Title='Fix rollover - CHANGE threshold', group = Axis_ID)
	 offset = get_num(Prompt = 'Offset', Val=roll_over, Title='Fix rollover - offset', group = Axis_ID)
ENDIF
stop1 = threshold
stop2 = offset


; -------- check if have 2D image in the buffer
IF keyword_set(one) then BEGIN
	HANDLE_VALUE, Data(CurBuf), tmp
    IF n_tags(tmp) EQ  0 THEN BEGIN
        axis_log, 'ax_fix_rollover: no image to process'
    	RETURN, s
    ENDIF
    IF tmp.t NE'2d' THEN BEGIN
        axis_log, 'ax_fix_rollover: only for images'
       	RETURN, s
    ENDIF

; -------- process file in the current buffer in aXis2000
	IF keyword_set(sign) then BEGIN
	    neg_ind = where(tmp.d LT threshold, count)
	    IF count GT 0 then begin
	      	tmp.d(neg_ind) = tmp.d(neg_ind)+ offset
	      	tmp.d(1,1) = tmp.d(1,1) + 1e-6* tmp.d(1,1)   ; kluge to force REAL data
		   	tmp.dl = tmp.dl + ' FixR'
			RETURN, tmp
		 endif
	ENDIF ELSE BEGIN
		tmp = fix_rollover(tmp, threshold=threshold, offset=offset)
		RETURN, tmp
	ENDELSE
ENDIF ELSE BEGIN

; ###############################
; fix rollowver for a sequence of files in a stack (*.ncb), or listed in a stack list (*.sl) file

; code to switch *.sl, versus *.(dat.ncb)
	IF keyword_set(stack) THEN BEGIN

;  --------- read stack
		t = size(stack) 	; use type identifier to see if stack is a string
		if t(1) NE 7 then begin
			stack = pickfile2(/read, title = ' Select stack file', filter='*.ncb', /LPATH, DEFPATH=defpath)
			if strlen(stack(0)) EQ 0 then return, s
		endif else begin
			check = findfile(stack)
			if strlen(check(0)) EQ 0 then return,s
		endelse
		stack_rb,stack
		test = size(image_stack)
		nimg = test(3) & nx = test(1) & ny = test(2)
		text = string(format='("Stack: ",i4," images. ",/,"       ",i4," x ",i4)', nimg, nx, ny)
		x_step = (x_stop - x_start)/n_cols
		y_step = (y_stop - y_start)/n_rows
		xd = x_start + findgen(n_cols)*x_step
		yd = y_start + findgen(n_rows)*x_step

; -------- loop through to fix_rollover in each image
		FOR loop = 0, nimg-1 do begin
			WIDGET_CONTROL, /hourglass
			img = image_stack(*,*, loop)
			tmp = {t:'2d', x:xd, y:yd, e: 0.0, d:img, xl:'X', yl:'Y', dl: 'stack image '+ string(loop)}

			tmp = fix_rollover(tmp, threshold=threshold, offset=offset)

	; --------------- replace the modified image
			image_stack(*,*, loop) = tmp.d
			 axis_log, tmp.dl
		ENDFOR
; ------- save modified stack
		t = ax_name(stack)
		stack_file = t(0)  + t(1) +'_F'+ '.' + t(2)
		stack_wb, stack_file
		axis_log, stack_file + 'fixed rollover'
	ENDIF ELSE BEGIN

; ---------------- apply fix rollover to a set of *.axb files, defined by a stack list (*.sl) file

; ----- read stack list
		WIDGET_CONTROL, Uprompt, SET_VALUE= 'Select file-list file'
		list_file = pickfile2(/READ, FILTER='*.sl',/LPATH, DEFPATH=defpath)
		Write_path = Lpath
		IF strlen(list_file) EQ 0 THEN return, s
	; ------ read in the *.sl file
		stack_readlist,list_file, filename_list
		if n_elements(filename_list) LE 1 then begin
			axis_log, 'Could not read ' + list_file
			return, s 	; BAIL if no files
		endif
		filename_list_out = strarr(n_elements(filename_list))

		FOR loop = 0, n_elements(filename_list)-1 do begin
			WIDGET_CONTROL, /hourglass
			tmp = axb_load(file=filename_list(loop), /silent)
			IF n_tags(tmp) EQ 0 THEN RETURN, s
		    IF tmp.t NE '2d' THEN RETURN, s

			tmp = fix_rollover(tmp, threshold=threshold, offset=offset)

	; --------------- write out the result
			tmp.dl = tmp.dl + ' FixR'
			CurBuf = 0
			HANDLE_VALUE, Data(CurBuf), tmp, /set
			PlotBuf,CurBuf
			 t = ax_name(filename_list(loop))
			 filefix= t(0) + 'm_' + t(1) + '.axb'
			 tf=axb_save(tmp, file=filefix, /silent)
			 t = ax_name(filefix)
			 filename_list_out(loop)=t(1) + '.axb'
			 axis_log, filename_list_out(loop)
		ENDFOR


		; ------ write out stack list file of modified images ----------
		t = ax_name(list_file )
		out_list_file = t(0) + t(1) + '_F.' + t(2)
		openw,unit, out_list_file,/get_lun
		printf,unit,t(0)
		for i = 0, n_elements(filename_list_out)-1 do begin
			printf,unit, filename_list_out(i)
		endfor
		close,unit & free_lun, unit
		t = ax_name(out_list_file)
		axis_log, t(1) + '.sl = fixed rollover'
		RETURN, out_list_file
	ENDELSE
ENDELSE
END



;  ---------- subsequent is code from Jan - May 2022 ----------
;IF keyword_set(one) then begin
;	HANDLE_VALUE, Data(CurBuf), tmp
;    IF n_tags(tmp) EQ 0 THEN RETURN, s
;    IF tmp.t NE '2d' THEN RETURN, s
;
;     neg_ind = where(tmp.d LT threshold, count)
;     IF count GT 0 then begin
;      	tmp.d(neg_ind) = tmp.d(neg_ind)+ offset
;      	tmp.d(1,1) = tmp.d(1,1) + 1e-6* tmp.d(1,1)   ; kluge to force REAL data
;        CurBuf = 0
;	   	tmp.dl = tmp.dl + ' FR_U'
;	   	Label(CurBuf) = tmp.dl
;		HANDLE_VALUE, Data(CurBuf), tmp, /set
;		PlotBuf,CurBuf
;		s = 1
;	 endif
;	 if s eq 0 then print, 'Images~Fix rollover: nothing modified'
;	 return, s
;ENDIF ELSE BEGIN


; ---------- original code  July 2000 :   works for some roll-over cases
;	    neg_ind = where(tmp.d LT threshold, count)
;	    IF count GT 0 then begin
;	      	tmp.d(neg_ind) = tmp.d(neg_ind)+ offset
;;	      	tmp.d(1,1) = tmp.d(1,1) + 1e-6* tmp.d(1,1)   ; kluge to force REAL data
;	        CurBuf = 0
;		   	tmp.dl = tmp.dl + ' FixR'
;		   	Label(CurBuf) = tmp.dl
;			s = 1
;		 endif

; ------- developed June 2021 for 2pi roll over  but never completed; modified to user-defined  Jan 2022
;	IF keyword_set(user) then begin			; user defined low_limit and roll_over increment
;		low_limit = 4
;		roll_over = 6.283 		; 2pi = 6.28318530718
;	endif else begin
;		low_limit = 0
;		roll_over = 65535.
;	endelse
;	for i = 0, n_elements(filelist)-1 do begin
;;		WIDGET_CONTROL, /hourglass
;;		num = strmid(strtrim(string(1000+i),2),1,3)
;;		file = fileroot+num+'.'+ext
;;			tmp = read_bnl(file)		; changed to *.axb files
;		tmp = axb_load(file=filelist(i), /silent)
;		count = 1
;		countall=0
; 		WHILE count GT 0  DO begin				; multiple roll-overs can occur !
; 			neg_ind = where(tmp.d LT low_limit, count)
; 			if count GT 0 then begin
; 				countall = countall + count
;		       	tmp.d(neg_ind) = float(tmp.d(neg_ind)) + roll_over
;;	 	     	tmp.d(1,1) = tmp.d(1,1) +1e-6*tmp.d(1,1)   ; kluge to force REAL *,nc file !
;	 	    endif
;  		ENDWHILE
;  		print, filelist(i), ' fixed ', countall, ' roll overs'
;
;		tf=axb_save(tmp, file=filelist(i), /silent, /overwrite)	; OVERWRITE input file
;		t=ax_name(tf)
;;		WIDGET_CONTROL, Uprompt, BAD_ID = BadID, $
;;		    SET_VALUE='Rollover fixed image ' + string(t(1) + '.' + t(2))
;	endfor
;	return, s


; ========== unsuccessful attempts  - May 2022 ==============
; scan each column to find changes of adjacent rows which are larger than the threshold
;  for 2-pi phase roll-over a change of +3.14 means subtract 2pi (6.2832) from esubsequent rows by  until next time threshold is succeeded
;  for 2-pi phase roll-over a change of -3.14 means need to add 2pi (6.2832) to subsequent rows by 2pi (6.28318530718) until next time threshold is succeeded
;	change = 0
; 	for i = 0, n_elements(tmp.x)-2 do begin
; 		change = 0
; 		for j = 0, n_elements(tmp.y)-2  do begin
; 			if abs(tmp.d(i,j+1) - tmp.d(i,j)) GT threshold then begin
;; 				print, i, '   ',j, '     ', tmp.d(i,j+1) - tmp.d(i,j)
; 			      if tmp.d(i,j+1) - tmp.d(i,j) GE 0 then change = -1.*offset else  change = offset
; 			endif
; 			tmp.d(i,j+1) = tmp.d(i,j+1) + change
; 		endfor
;	endfor

; ------ 2nd try: when find boundary in a column, change rest of the column
; 	for i = 0, n_elements(tmp.x)-2 do begin
; 		change = 0
; 		for j = 0, n_elements(tmp.y)-2  do begin
; 			if abs(tmp.d(i,j+1) - tmp.d(i,j)) GT threshold then begin
; 			; change rest of this column
; 				if tmp.d(i,j+1) - tmp.d(i,j) GE 0 then change = -1.*offset else  change = offset
;				for k = j+1, n_elements(tmp.y)-1  do tmp.d(i,k) = tmp.d(i,k) + change
;				goto, next
;			endif
;		endfor
;		next:
;	endfor
;
;	tmp.dl = tmp.dl + ' fix_rollover'
;	return, tmp

; ---------------- 3rd try - use WHERE (Jan 22)
; 	print, '============  ONE IMAGE USER DEFINED'
;     neg_ind = where(tmp.d LT threshold, count)
;     IF count GT 0 then begin
;      	tmp.d(neg_ind) = tmp.d(neg_ind) + offset
;;      	tmp.d(1,1) = tmp.d(1,1) + 1e-6* tmp.d(1,1)   ; kluge to force REAL data
;        CurBuf = 0
;	   	tmp.dl = tmp.dl + ' FixR'
;	   	Label(CurBuf) = tmp.dl
;		HANDLE_VALUE, Data(CurBuf), tmp, /set
;		PlotBuf,CurBuf
;	ENDIF
