; Copyright (c) 1998-2000 A.P. Hitchcock  All rights reserved 
;+ 
;NAME: 
;	CLEAN 
; 
;LAST CHANGED: ----------------------------------- 23-feb-00 
; 
;PURPOSE: 
;	This procedure performs a 2d fourier transform of an image. 
;  The user then selects a sub-region of the 2dFT which is zeroed. 
; The rest of the array is thenn reverse transformed back into image space. 
; Apropriate selection of sub-region provides a powerful means 
; to remove periodic noise (e.g. from beam motion) from image data. 
; 
;CATEGORY: 
;	STAND ALONE: image processing 
; 
;CALLING SEQUENCE: 
;	CLEAN, array 
; 
;CALLED FROM AXIS: 
;	->Filter->clean(FT) 
; 
;INPUTS: 
;	ARRAY - 2d array 
; 
;KEYWORDS: 
;	HELP - if set, gives instructions 
; 
;OUTPUTS: 
;	The result is placed in the input array 
; 
;COMMON BLOCKS: 
;	@AXIS_COM	standard set of common blocks 
;   @BSIF_COM   common for BSIF data structures 
;   @IMG_COM    common for NSLS images 
; 
;RESTRICTIONS: 
;	Warning : 
;	Not assuming responsibility for misuse, or abuse of the image data. 
;	Picking too many peaks may damage the data. 
; 
;PROCEDURE: 
; ORIGINAL (X1A code) 
; 	Cleans up beam noise by picking noise peaks in the FFT with the mouse. 
; 	When done picking points, click the mouse off the edge of the FFT.  The 
; 	routine then puts up the cleaned FFT and writes the STXM file. 
; NOW 
;	user can only click twice, which defines a box. The contents of this box are removed. 
; 
;MODIFICATION HISTORY: 
; (09-mar-92 spw) first version 
; (xx-feb-93 cjj) improved 
; (12-jan-99 aph) incorporated into AXIS 
; (23-feb-00 aph) AXIS standard documentation 
;- 
pro clean, array, help=help 
 
@axis_com
@bsif_com
@img_com
 
if (keyword_set(help) or (n_elements(array) eq 0)) then begin 
	print,'clean,array' 
		print, ' Fourier filter artifact reduction. Dispalys 2d FT of array' 
		print,'Pick beam noise peaks with the mouse by clicking once for' 
		print,'  one corner of the region to be zeroed, and once for the' 
		print,'  other corner.  2d reverse FT regenerates image without signal in box' 
	return 
endif 
 
svec=size(array) 
if (svec(0) ne 2) then begin 
	print,'Nothing done - array is not 2D!' 
	return 
endif 
 
erase 
print,'Pick beam noise peaks with the mouse by clicking once for' 
print,'  one corner of the region to be zeroed, and once for the' 
test = string(format='("computing 2d FT . . . ",A)','') 
WIDGET_CONTROL, Uprompt, Bad_ID = badID, Set_Value = test 
 
nx=n_elements(array(*,0)) 
ny=n_elements(array(0,*)) 
ft=fft(array,-1) 
ft=shift(ft,nx/2,ny/2) 
ft_min=min(abs(ft)) 
ft_max=max(abs(ft)) 
tv, bytscl(alog(abs(ft)), min=alog(ft_min), max=alog(ft_max)) 
 
print,'  other corner.  Click outside the image when done.' 
test = string(format='("Select region to remove",/,/,"centre is low frequency",A)','') 
;	  		,//,"click outside image twice to complete",A)','') 
WIDGET_CONTROL, Uprompt, Bad_ID = badID, Set_Value = test 
; ------------- get limits and ensure they are inside image ---------- 
box,xv,yv,/device 
x_left=min(xv) 
x_right=max(xv) 
y_bottom=min(yv) 
y_top=max(yv) 
 while ( (x_left lt 0) or (x_left ge nx) or $ 
	(x_right lt 0) or (x_right ge nx) or $ 
	(y_bottom lt 0) or (y_bottom ge ny) or $ 
	(y_top lt 0) or (y_top ge ny) ) do begin 
	box,xv,yv,/device 
	x_left=min(xv) 
	x_right=max(xv) 
	y_bottom=min(yv) 
	y_top=max(yv) 
endwhile 
; --------- remove selected area ------------- 
print,' Averaging from (',strtrim(string(x_left,format='(i5)'),2),',', $ 
strtrim(string(y_bottom,format='(i5)'),2),') to (', $ 
strtrim(string(x_right,format='(i5)'),2),',', $ 
strtrim(string(y_top,format='(i5)'),2),').' 
value1 = ft(x_left-1,y_bottom:y_top) 
value2 = ft(x_left:x_right,y_top + 1) 
num1 = n_elements(value1) 
num2 = n_elements(value2) 
sum = 0 
for i = 0,num1-1 do sum = sum + value1(i) 
for j = 0,num2-1 do sum = sum + value2(j) 
avg = sum/(num1+num2) 
ft(x_left:x_right,y_bottom:y_top)=avg 
tv, bytscl(alog(abs(ft)), min=alog(ft_min), max=alog(ft_max)) 
ft=shift(ft,-nx/2,-ny/2) 
; ----------------- reverse FT to generate filtered image ----------- 
test = string(format='("computing 2d reverse FT . . . ",A)','') 
WIDGET_CONTROL, Uprompt, Bad_ID = badID, Set_Value = test 
array=fft(ft,1) 
tvscl,array 
WIDGET_CONTROL, Uprompt, Bad_ID = badID, Set_Value = '' 
return 
end 
