; Copyright (c) 1998-2009 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	LOAD_SSV
;
;LAST CHANGED: ----------------------------------- 14-Dec-09 (aph)
;
;PURPOSE:
;	This function reads in Twinmic format image or spectra from
; the  *.ssv file written by Twinmic for images (transmission, phase, dark, XRF-maps, etc)
;
;CATEGORY:
;	STAND ALONE: read in utility
;
;CALLING SEQUENCE:
;	Result = LOAD_SSV([file=file, NOF=nof, group=group,TYPE=TYPE, $
;                  VERBOSE=VERBOSE, SILENT=SILENT, PARAM=PARAM, _extra=e])
;
;CALLED FROM AXIS:
;	Read->Images->Twinmic
;
;INPUTS: none. All input parameters are passed as keywords.
;
;KEYWORDS:
;	FILE 	filename to read
;	NOF 	no filter (default filter = '*.axb')
;	GROUP	group leader (Axis_ID if called from aXis2000 and axis)
;	SILENT  if set, do not display feedback on progress
;	PARAM	if set, ask user for parameters of Twinmic image
;	TYPE	type of image (sum, X_mom, Y_mom) selected automatically
;	VERBOSE print log information about processing
;	EXTRA 	passed on parameters
;
;OUTPUTS:
;	RESULT = AXIS 1d or 2d structure
;
;COMMON BLOCKS:
;	@AXIS_COM	standard set of common blocks
;	@BSIF_COM   for NSLSL *.nc
;
;MODIFICATION HISTORY:
; (14-Dec-09 aph) adapted from AXB_LOAD
;-

Function load_ssv, file=file, NOF=nof, group=group, silent=silent, $
       type=type, PARAM=PARAM, verbose=verbose, _extra=e
@axis_com
@bsif_com
on_error,2

IF NOT keyword_set(file) then begin
	fltr='*.ssv'
	if keyword_set(nof) then fltr='*'
	file = pickfile2(/READ, FILTER=fltr, /LPATH, DEFPATH=defpath)
ENDIF
s = ''
if file EQ '' THEN RETURN, s  ; bail-out if no filename
check = findfile(file)		 ; bail-out if non-existent file
if strlen(check(0)) EQ 0 then begin
	axis_log, 'Cannot find ' + file
	return, s
endif
t = ax_name(file)					; file format: yymmdd_hhmmss_(type).ssv)
tu1 =  strpos(t(1),'_')
st = strmid(t(1),tu1 +1)            ; all after first '_'
tu2 = tu1 + strpos(st,'_')
fileshort = strmid(t(1), 0,tu2+1)
ssv_type =strmid(t(1),tu2+2)
filep = t(0) + ax_sep() + fileshort + '.prm'
if keyword_set(verbose) then begin
	print, 'fileshort = ', fileshort
	print, 'parameter file = ', filep
	print, 'type of image = ', ssv_type
endif

openr, lun, file, /get_lun		; open as text
ssvt = ''
readf,lun, ssvt
; print, ssvt
ssvz=long(parsestring(ssvt))		; convert to array of long integers
t = size(ssvz)
if t(0) EQ 1 then begin
	nx = t(1)						; assumes image file varies x-dimension most rapidly
endif else begin					; use ROTATE afterward to obtain correct sense of image
	axis_log, 'error on read of Twinmic file ' + fileshort
	return, s
endelse
ny = 1
dt = ssvz
while not  eof(lun) do begin
	readf,lun, ssvz
;	print, ssvz
	dt = [dt,ssvz]
	ny = ny + 1
endwhile
d = float(reform(dt,nx,ny))
d= rotate(d,4)
close, lun & free_lun, lun

;  ------------ convert to axis 2d structure
x = findgen(nx)
y = findgen(ny)
energy = 0.
dwell  = 0.
dl = fileshort		; default header

; extract parameters from the associated parameter file
openr, lun, filep, /get_lun		; open as text
; ------- Read header file into a single string (max = 32600 characters)  --------
line = ' '
all = ''
count = 0
while not  eof(lun) do begin
	readf,lun, line
;	if keyword_set(verbose) then print, line
	all = all + line
	count = count + 1			; the number of lines is used in spectral readin
endwhile
close, lun & free_lun, lun
on_ioerror,null

x_mid = float(ax_strmid(all, 'X center of image (micron) =', 'Y'))
y_mid = float(ax_strmid(all, 'Y center of image (micron) =', 'X'))
x_range = float(ax_strmid(all, 'X scan range (micron) =', 'Y'))
y_range = float(ax_strmid(all, 'Y scan range (micron) =', 'No'))
if x_range GT 0. then x_step = x_range/nx else x_step = 1
if y_range GT 0. then  y_step = y_range/ny else y_step = 1
x= x_mid - 0.5*x_range  + x*x_step
y= y_mid - 0.5*y_range  + y*y_step

dwell = float(ax_strmid(all, 'Dwell time (ms) =', 'Total'))

if keyword_set(param) then begin
	if keyword_set(group) then begin
		energy = get_num(prompt='Photon energy (eV)', val=energy, group=group)
;		dwell = get_num(prompt='Dwell (ms)', val=dwell, group=group)
		dl = get_text(prompt='Header', val=dl, group=group)
	endif else begin
		energy = get_num(prompt='Photon energy (eV)', val=energy)
;		dwell = get_num(prompt='Dwell (ms)', val=dwell)
		dl = get_text(prompt='Header', val=dl)
	endelse
 endif

if dwell GT 1000 then begin
	xl = string(FORMAT='("x (um)     E = ",f8.3," eV     dwell = ",f7.2," sec")', $
      energy, dwell/1000.)
endif else begin
    xl = string(FORMAT='("x (um)     E = ",f8.3," eV     dwell = ",f7.2," ms")', $
      energy, dwell)
endelse
yl = 'Twinmic image type: ' + ssv_type

text = string(format= $
  '(i4," x",i4," Twinmic image read from ",A,".   E =",f7.2," eV. Dwell =",g13.6," ms.")', $
   nx, ny, fileshort, energy, dwell)
if not keyword_set(silent) then axis_log, text

s = {t:'2d', x:x, y:y, d:d, e: energy, xl:xl, yl:yl, dl: dl}	; add E

return, s

END
