; Copyright (c) 1998-2009 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	STXM4_NCB
;
;LAST CHANGED: ----------------------------------- 30-Jul-09
;
; PURPOSE:
;	This procedure converts a series of NSLS STXM IV image files
; into a binary stack file. The STXM IV files are those listed in a '*.sl' file
;
; CATEGORY:
;	AXIS: stack analysis
;
; CALLING SEQUENCE:
;	RESULT = STXM4_NCM(group_leader=axis_ID)
;
; INPUTS:
;	none
;
; KEYWORD PARAMETERS:
;	GROUP_LEADER - identification of caller for modal control
;
; OUTPUTS:
;  FILE.NCB - stack binary file
;
; COMMON BLOCKS:
;	stack_process_com
;	axis_com
;	bsif_com
; 	volume_data
;
; RESTRICTIONS:
;	caveat emptor
;
; MODIFICATION HISTORY:
; (30-sep-01 aph) first developed
; (30-Jul-09 aph) changed analcom to stack_process_com (update to stack_process)
;-

function stxm4_ncb,group_leader=group_leade
@stack_process_com
@axis_com
@bsif_com
COMMON volume_data, image_stack

on_error,2

listfile = PICKFILE2(/Read, FILTER='*.sl', $
    title = 'Select NSLS STXM IV stack list file', /LPATH, DEFPATH=defpath)
stack_readlist, listfile, filename_list

; ------- remove alignment values if present
n_files = n_elements(filename_list)
for i = 0, n_files-1 do begin
	comma = strpos(filename_list(i),',')
	if comma NE -1 then filename_list(i) = strmid(filename_list(i),0, comma)
endfor

; -------- use GET_PATH to establish correct file names
; -------- check if file exists - this handles subdirectory changes
file1 = filename_list(0)
New_Path = get_path(file1)
if New_Path EQ '' then return,0	; let user cancel
for i = 0, n_files-1 do begin
	t = ax_name(filename_list(i))
	filename_list(i) = New_Path + t(1) + '.' + t(2)
endfor
;
; -------------- check back for user to see if filename list is valid---------
print, 'First file to process is ',filename_list(0)
; --- read first file to define stack dimensions
tmp = read_stxm4(file = filename_list(0),/silent)
nx = n_elements(tmp.x)
ny = n_elements(tmp.y)
npts = n_files
x = tmp.x
y = tmp.y
image_stack = fltarr(nx,ny,npts)
energy = fltarr(npts)
filename_ev_msec_list = strarr(npts)
; -------- read in images of stack and store in volume_data array
widget_control,hourglass=1
FOR i = 0,n_files-1 DO BEGIN
	tmp = read_stxm4(file = filename_list(i), /silent)
	image_stack(*,*,i) = tmp.d
	energy(i) = tmp.e
	dwell = sd.dwell_time
	filename_ev_msec_list(i) = 'stxmIV ' + $
		    string(energy(i), format = '(F7.2)') + '  ' +  string(dwell, format='(F5.2)')
ENDFOR
filename = pickfile2(filter = '*.ncb', title = 'binary stack file')
if strlen(filename) GT 0 then begin
	t = ax_name(filename)
	filename = t(0) + t(1) +'.ncb'
endif else return, filename
widget_control,hourglass=1
; define parameters needed for stack write out
x_start = x(0) & x_stop = x(nx-1)
y_start = y(0) & y_stop = y(ny-1)
ev = energy
stack_wb, filename
widget_control,hourglass=0
print, 'Binary stack file derived from STXM IV data written as ', filename
return, filename

END
