
; $Id: sf.pro,v 1.4 1999/03/03 06:14:10 idl Exp $
;
;+
; NAME:
;       SF
;
; LAST CHANGED: ------------------ aph ------------------------ 24-dec-00
;
; PURPOSE:
;       Function to calculate x-ray interaction properties of materials.
;
; CATEGORY:
;       X-ray optics
;
; CALLING SEQUENCE:
;       Result = SF(Compound [, Abscissa])
;
; INPUTS:
;       Compound:    A scalar string containing the compound molecular formula.
;                    The structure of the formula must be as follows:
;                      Name1 Amount1 Name2 Amount2 ...
;                    where Name1, Name2, etc. are the one or two letter
;                    standard abbreviations for elements 1 to 92, and
;                    Amount1, Amount2, etc. are the values describing the
;                    stoichiometry. The spaces are optional. Each element may
;                    appear only once in the formula.
;                    NOTES: 1. Formula names ARE case-sensitive
;                              (e.g., NI is NOT Ni).
;                           2. If an amount is not specified, it is assumed
;                              to be 1 (e.g., H2O is the same as H2 O1).
;
; OPTIONAL INPUTS:
;       Abscissa:    A vector of photon energies or wavelengths at which the
;                    result will be evaluated. The type and units of the
;                    abscissae are specified by the keyword parameters. The
;                    default is to assume the abscissae are photon energies
;                    in electron volts.
;
; KEYWORD PARAMETERS:
;       J:           If set, the abscissae, if supplied, are interpreted
;                    as x-ray photon energies in Joules.
;
;       keV:         If set, the abscissae, if supplied, are interpreted
;                    as x-ray photon energies in kiloelectron volts.
;
;       eV:          If set, the abscissae, if supplied, are interpreted
;                    as x-ray photon energies in electron volts.
;
;       m:           If set, the abscissae, if supplied, are interpreted
;                    as x-ray wavelengths in meters.
;
;       nm:          If set, the abscissae, if supplied, are interpreted
;                    as x-ray wavelengths in nanometers.
;
;       Angstrom:    If set, the abscissae, if supplied, are interpreted
;                    as x-ray wavelengths in Angstroms.
;
;       RESULT_TYPE: A string specifying what result to return. The following
;                    are the possible choices:
;                     "f1":      The compound f1 scattering factor for the
;                                molecule.
;
;                     "f2":      The compound f2 scattering factor for the
;                                molecule.
;
;                     "delta":   The real part of the complex refractive index.
;
;                     "beta":    The imaginary part of the complex refractive
;                                index.
;
;                     "n":       The complex refractive index.
;
;                     "mu":      The mass photoabsorption coefficient,
;                                in cm^2/g. This is the default result type.
;
;                     "mu_a":    The atomic photoabsorption coefficient,
;                                in barns/atom.
;
;                     "mu_l":    The linear photoabsorption coefficient,
;                                in cm^-1.
;
;                     "trans":   The transmissivity of the material.
;
;                     "reflect": The specular reflectivity of unpolarized light
;                                on an ideally smooth surface of the material.
;
;       DENSITY:     The density of the material, in g/cm^3. This is needed
;                    if the result type is "delta", "beta", "n", "mu_l",
;                    "trans", or "reflect". Otherwise, a default density
;                    of 1.0 g/cm^3 is used.
;
;       THICKNESS:   The thickness of the material in micrometers. This is
;                    needed if the result type is "trans". Otherwise, a
;                    default thickness of 1.0 micron is used.
;
;       ANGLE:       The angle of incidence in milliradians. This is needed
;                    if the result type is "reflect". Otherwise, a default
;                    angle of incidence of 10.0 mr is used.
;
;       ENERGY:      A variable to receive the x-ray photon energies (in eV)
;                    at which the result was evaluated.
;
;       MW:          A variable to receive the molecular weight (in g/mol) of
;                    the compound.
;
;       SF_DIR:      A string containing the name of the directory containing
;                    the SF data files.
;
; OUTPUTS:
;       Result:      The result specified by the RESULT_TYPE keyword.
;
; COMMON BLOCKS:
;       SF_COMMON
;
; SIDE EFFECTS:
;       SF data is stored into common block variables each time data is read
;       from the data files. This takes up memory to gain faster performance.
;       The memory can be freed using the SF_DESTROY routine.
;
; ROUTINES CONTAINED IN THIS MODULE:
;       SF_INIT           Procedure to initialize common block for module.
;       SF_DESTROY        Procedure to delete common block variables.
;       SF_GEN_EL_DATA    Procedure to generate data file for elements.
;       SF_READ_DATA      Procedure to read SF data for single elements.
;       SF_PARSE_COMPOUND Function to parse molecular formulas.
;       SF                Main function to compute materials x-ray properties.
;
; ROUTINES CALLED BY THIS MODULE:
;       IDL User Library routines:
;         UNIQ, INTERPOL
;       Bill Loo's IDL routines:
;         STRING2FILE, FILE2STRING, STR_COLUMNS, WVLEN2EN, EN2WVLEN
;
; DATA FILES:
;       This module requires the scattering factor data files compiled by
;       B.L. Henke, E.M. Gullikson, and J.C. Davis, available by anonymous
;       ftp at:
;         ftp://www-cxro.lbl.gov/pub/sf/sf.tar.Z
;
;       In addition, it requires a file containing the abbreviated element
;       names and atomic weights. Such a file can be generated manually
;       using the routine SF_GEN_EL_DATA. Just type "SF_GEN_EL_DATA" at the
;       IDL prompt, and enter the requested information.
;
; REFERENCE:
;       All formulas are published in:
;         B.L. Henke, E.M. Gullikson, and J.C. Davis, "X-ray interactions:
;         photoabsorption, scattering, transmission, and reflection at
;         E=50-30,000 eV, Z=1-92," Atomic Data and Nuclear Data Tables,
;         Vol. 54(2), pp. 181-342, July 1993.
;
; WRITTEN BY:
;       Billy W. Loo, Jr.
;       Bioengineering Graduate Group, UCB / UCSF
;       School of Medicine, UCD
;       Ernest Orlando Lawrence Berkeley National Lab
;       August, 1996
;-
;
; MODIFICATION HISTORY:
;
; $Log: sf.pro,v $
; Revision 1.4  1999/03/03 06:14:10  idl
; * Changed function names for more consistent naming.
; * Minor changes to documentation.
;
; Revision 1.3  1999/02/28 00:26:10  idl
; * Return scalar when result has only one element.
;
; Revision 1.2  1997/12/16 01:55:20  loo
; * Major revision of SF program module.

;=============================================================================
;=           Procedure to initialize common block for module                 =
;=============================================================================

PRO SF_INIT, sf_dir

  ;Declare common block for module.
  COMMON SF_COMMON, sf_dir_,   $ ;Directory containing SF data files.
                    sf_ext_,   $ ;File extension of SF data files.
                    el_names_, $ ;Array of element names.
                    at_wts_,   $ ;Array of element atomic weights.
                    energy_,   $ ;Array of pointers to energy for each element.
                    f1_,       $ ;Array of pointers to f1s for each element.
                    f2_          ;Array of pointers to f2s for each element.

  ;Initialize directory containing SF data files.
  IF (N_PARAMS() GT 0) THEN sf_dir_ = sf_dir ELSE sf_dir_ = ''

  ;Default file extension.
  sf_ext_ = '.nff'

  ;Try to load element data.
  IF (N_ELEMENTS(el_names_) EQ 0) THEN BEGIN
    ;Search for data file.
    fn = FINDFILE(sf_dir_ + 'elements.dat')

    IF (fn[0] NE '') THEN BEGIN
      data = FILE2STRING(fn[0])
      STR_COLUMNS, data

      el_names_ = data[0, *]
      at_wts_   = [[0], [FLOAT(data[1, 1:*])]]
    ENDIF $

    ELSE PRINT, 'Data file not found!' + STRING(7B)

    ;Initialize pointer arrays.
    energy_ = PTRARR(93)
    f1_     = PTRARR(93)
    f2_     = PTRARR(93)
  ENDIF

  RETURN
END ;sf_init
