;+
; NAME:
;	ZSTACK_SHIFT.PRO
; AUTHORS:
;	Carl G. Zimba (NIST), Chris Jacobsen (SUNY - Stonybrook)
; PURPOSE:
;	Shift of images abtained on an x-ray microscope.
; CATEGORY:
;	Data analysis.
; CALLING SEQUENCE:
;	zstack_shift,image,xshift,yshift,new_image
; INPUTS:
;	image = image to be shifted
;	xshift = x-axis shift
;	yshift = y-axis shift
; KEYWORD PARAMETERS:
;	edgefill = type of filling to use at edges of shifted images
;				(edges can contain no data due to shifting)
;		meanfill 		:	replace with mean of image
;		medianfill 	:	replace with median of image
;		maxfill 		:	replace with maximum of image
;		minfill 		:	replace with minimum of image
;		zero			:	replace with zero (default)
;		NaN			:	replace with NAN (!values.f_NaN)
;		infinity		:	replace with infinity (!values.f_infinity)
;	help = set to print how-to-use statement
; OUTPUTS:
;	new_image = image after applying x and y shifts
; COMMON BLOCKS:
;	NONE
; SIDE EFFECTS:
;
; RESTRICTIONS:
;
; PROCEDURE:
;   Complicated.
; EXAMPLE:
;
; MODIFICATION HISTORY:
; Adapted from sections of STACK_ANALYZE.PRO and STACK_ALIGN.PRO
;	where poly_2D is used to shift images
; Eliminates conditional based upon minimum shift of 0.02
;	This is now done in code prior to calling ZSTACK_SHIFT
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_shift,image,xshift,yshift,new_image,edgefill=edgefill,help=help
;print,'zstack_shift'
@bsif_common
IF (keyword_set(help) OR (n_params() eq 0)) THEN BEGIN
    print,'stack_shift,image,xshift,yshift,new_image'
    print,'  Constraints on shifted image: edgefill'
    return
ENDIF
idl_major_version_string = $
  strmid(!version.release,0,strpos(!version.release,'.'))
idl_major_version = fix(0)
reads,idl_major_version_string,idl_major_version
; Only shift the image if the user wants it
; Changed to always shift the image, CGZ
; Let the user decide after viewing movie of shifted images
; Changed to use edgefill parameter to replace out-of-bounds datapoints, CGZ
; 	edgefill = meanfill 		- replace with mean of image
;	edgefill = medianfill 	- replace with median of image
;	edgefill = maxfill 		- replace with maximum of image
;	edgefill = minfill 		- replace with minimum of image (default)
;	edgefill = zero			- replace with zero
;	edgefill = NaN			- replace with NAN (!values.f_NaN)
;	edgefill = infinity		- replace with infinity (!values.f_infinity)
p = [xshift,0.,1.,0.]
q = [yshift,1.,0.,0.]
; edgefill setting of infinity doesn't seem to work -
;	needed to use following in zstack_analyze_imgprep routine to properly display images
;		min(image(where(finite(image))))
;		max(image(where(finite(image))))
;edgefill = 'infinity'
; edgefill setting of NaN doesn't seem to work -
;	later on in conditional tests for Nan, it can't be found
;	(VAR EQ !values.f_nan) gives zero even if NaN values exist
CASE edgefill OF
	'meanfill' : BEGIN
;		print,'edgefill : meanfill'
		missing = total(image)/float(n_cols*n_rows)
		IF (idl_major_version LE 3) THEN BEGIN
			new_image = poly_2d(image,p,q,1,missing=missing)
		ENDIF ELSE BEGIN
			new_image = poly_2d(image,p,q,2,missing=missing,cubic=-0.5)
		ENDELSE
	END
	'medianfill' : BEGIN
;		print,'edgefill : medianfill'
		missing = median(image)
		IF (idl_major_version LE 3) THEN BEGIN
			new_image = poly_2d(image,p,q,1,missing=missing)
		ENDIF ELSE BEGIN
			new_image = poly_2d(image,p,q,2,missing=missing,cubic=-0.5)
		ENDELSE
	END
	'maxfill' : BEGIN
;		print,'edgefill : maxfill'
		missing = 1.01*max(image)
		IF (idl_major_version LE 3) THEN BEGIN
			new_image = poly_2d(image,p,q,1,missing=missing)
		ENDIF ELSE BEGIN
			new_image = poly_2d(image,p,q,2,missing=missing,cubic=-0.5)
		ENDELSE
	END
	'minfill' : BEGIN
;		print,'edgefill : minfill'
		missing = 0.99*min(image)	; needed 0.9 factor for proper display of image ratios
							; this is also useful in clipping images and spectra
							; since all data is still larger than clipped regions
		IF (idl_major_version LE 3) THEN BEGIN
			new_image = poly_2d(image,p,q,1,missing=missing)
		ENDIF ELSE BEGIN
			new_image = poly_2d(image,p,q,2,missing=missing,cubic=-0.5)
		ENDELSE
	END
	'zero' : BEGIN
;		print,'edgefill : zero'
		missing = 0
		IF (idl_major_version LE 3) THEN BEGIN
			new_image = poly_2d(image,p,q,1,missing=missing)
		ENDIF ELSE BEGIN
			new_image = poly_2d(image,p,q,2,missing=missing,cubic=-0.5)
		ENDELSE
	END
	'NaN' : BEGIN
;		print,'edgefill : NaN'
		missing = !values.f_NaN
		IF (idl_major_version LE 3) THEN BEGIN
			new_image = poly_2d(image,p,q,1,missing=missing)
		ENDIF ELSE BEGIN
			new_image = poly_2d(image,p,q,2,missing=missing,cubic=-0.5)
		ENDELSE
	END
	'infinity' : BEGIN
;		print,'edgefill : Infinity'
		missing = !values.f_infinity
		IF (idl_major_version LE 3) THEN BEGIN
			new_image = poly_2d(image,p,q,1,missing=missing)
		ENDIF ELSE BEGIN
			new_image = poly_2d(image,p,q,2,missing=missing,cubic=-0.5)
		ENDELSE
	END
	ELSE : BEGIN
;		print,'edgefill : else'
		missing = min(image)
		IF (idl_major_version LE 3) THEN BEGIN
			new_image = poly_2d(image,p,q,1,missing=missing)
		ENDIF ELSE BEGIN
			new_image = poly_2d(image,p,q,2,missing=missing,cubic=-0.5)
		ENDELSE
	END
ENDCASE

;			print,'shift:  p           =',p
;			print,'shift:  q           =',q
return
END