; Copyright (c) 1998-2000 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	ALS_SPECIN
;
;LAST CHANGED: ----------------------------------- 19-mar-02
;
; PURPOSE:
;	This function reads in an ALS BL7.0 stxm spectral data file
;	based on parameters generated by the ALS_SPECTRA widget
;
; CATEGORY:
;	AXIS: spectra analysis
;
; CALLING SEQUENCE:
;	Result =  ALS_SPECIN(in_file, REF=ref, COL=col, _extra=e)
;
; INPUTS:
;	IN_FILE  - name of ALS BL 7.0 STXM data file (transmission or yield signal)
;
; KEYWORDS:
;	REF - switch to indicate this is incident flux reference
;	COL=# - data column to read in
;	_EXTRA_E - in case one wants to pass plot paramaters
;
; OUTPUTS:
;	Result is an AXIS 1d data structure.
; The y data values are calculated according to the readin parameters
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;;
; PROCEDURE:
; uses current values of I_process, Io_process to determine what to do
; set KEYWORD REF to 1 to use Io_process
; I_process =
; 'all'  - return both dark and data values
; 'none' - ignore dark counts
; 'dark' - average dark, subtract
; 'OSA'  - ratio signal to OSA (without dark correction)
; 'both' -  ratio signal to OSA (with dark correction of each)
;
; -------- internal parameter, OPT, used to control processing -------
;default (no opt) is to read in E,col1,col2
; OPT = col1:   raw data - ASSUMES transmission detector in col(1); OSA in col(2)
; OPT = col2:  extract col(2)  - OSA (pre-98) or trans (after Dec-97; flipped here)
; OPT = col3;  extract col(3)  - TEY
; OPT = dark:  dark correct signals
; OPT = OSA;   divide by OSA signal (no dark correct)
; OPT = OSA-dark; dark correct, then divide by OSA signal
;
; ALS BL7.0 point spectra consist of ascii files (*.dat, *.xas)
; ------ FORMAT of *.DAT file (scan details)
;10	5	50000	100000		(#dark points) (#sections) (shutter delay - us) (dwell/dark - us)
;2.7500E+2	2.0000E-1	40	100000 Region 1:(first-E in eV) (step in eV) (number) (dwell)
;2.8300E+2	7.9710E-2	138	100000 Region 2:
;2.9400E+2	1.2000E-1	50	100000 Region 3:
;3.0000E+2	2.0000E-1	75	100000 Region 4:
;3.1500E+2	2.5000E-1	60	100000 Region 5:
; ------ FORMAT of *.xas file
;273.000	36	16
;273.200	37	17		dark signal levels before scan
; .....
;275.000	16774	51465	E, signal_1, signal_2 (as many defined in acquire)
;275.200	16820	52070
;275.400	16824	52848
; .....
;330.200	36	16
;330.400	37	17		dark signal levels after scan
; .....
;
; CAUTIONS
; the data format was modified several times in 1997
; this routine attempts to auto-detect the format of the file
;	but some errors do occur
;
; MODIFICATION HISTORY:
; (09-Dec-97 aph) multi-section files; single file to do all types of read-in of ALS STXM spectral files
; (15-feb-98 aph) cancel protect; read 0 dark_pt  files
; (04-mar-98 aph) still problems with 0 dark pts;
; (09-apr-98 aph) save img_shifts; correct spike; add col(3) = TEY
; (07-aug-98 aph) indicate extension (? not done since col indicated already)
; (02-jan-99 aph) derived from XAS1.pro
; (03-jan-99 aph) add ytitle (s.yl)
; (28-jan-99 aph) xlabel consistent with nsls readin; DefPath in pickfile2
; (18-sep-99 aph) correct to allow read-in with no dark points
; (30-dec-99 aph) AXIS standard documentation
; (19-mar-02 aph) adapt to new stxm701 which dies not use OSA (only 2 cols)
;-

function als_specin, in_file, COL=col, REF=ref, _extra=e

@axis_com
on_error,2

if n_elements(in_file) eq 0 then begin  ;popup file dialog box
   fltr='*.xas'
   if keyword_set(nof) then fltr='*'
   in_file=PICKFILE2(/Read, FILTER=fltr, /LPATH, DEFPATH=defpath)
endif

IF strlen(in_file) GT 0 THEN BEGIN

; adapt according to current values of I_process (Io_process if Ref set)
	Proc_opt = I_Process
	IF keyword_set(ref) then Proc_opt = Io_Process
	CASE Proc_opt of
		'all' : OPT = 'all'
		'none': OPT = 'raw'
		'dark': OPT = 'dark'
		'OSA' : OPT = 'OSA'
		'both': OPT = 'OSA-dark'
	ENDCASE

	file=strlowcase(in_file)                  ; assumes ALS 8.3 filename
	file=strmid(file,0,strlen(file)-4)
	dat=text_read1(file+'.dat')
	a=text_read1(file+'.xas')
    a_size = size(a)
	icst=a(0,*)		; Energy
	b=a(1,*)
	if a_size(1) GT 2 then begin
		c=a(2,*)		; Note this only works of there is actually an OSA in the file !
	endif else c = b
; ------------------ COMMENT ON FILE FORMATS ---------------
; pre-Dec-97
; c is 2nd y-column (transmitted)          b is 1st y-column (OSA signal)
; Since the subsequent data processing assumed these labels, post-Dec-97 read-in swaps col (1,2)

; AUTO-detect file type (aph - 6-dec-97)
	test = size(dat)
	nul = 0
	if test(1) EQ 3 then begin
		print, "Reading in ALS format spectral file in FORMAT A (pre Nov-20-97)'
		nseg = 1
		nul=dat(0,1)	; (format A) pre-Nov-97 - 1-section file
		step=dat(1,0)
		dw = dat(2,1)/1e3  ; convert dwell in usec to msec
		samp=dat(2,0)
		ien=dat(0,0)
		b = b/dw  &  c = c/dw  ; normalize to dwell in msec
	endif else begin		; post Nov-97 formats - multi-section files
; (format B)(Nov-20 to 1-Dec-97) - no dark channel info - deduce from file
	    if dat(1,0) NE (test(2)-1) then begin  ;(format B)
	    	print, "Reading in ALS format spectral file in FORMAT B (Nov-20 to 1-Dec-97)'
	        nseg = test(2)                 ; number of segments
	        samp = total(dat(2,*))          ; total number of data points
	        nul = (a_size(2) - samp)/2       ; number of dark points
	        first = 0 & last = nseg - 1
	    endif else begin
; (format C) (after 3-Dec-97) - 1st line of dat has dark channel info
; ALSO somewhere (perhaps Feb-98 ?) OSA and TRANS signals swapped
	    	t=b & b=c & c=t                ; so swap them back !!
;			print, "Reading in ALS format spectral file in FORMAT C (after 3-Dec-97)'
	    	IF keyword_set(col) then begin
;	    		print, 'col = ', strcompress(string(col))
	    		IF strcompress(string(col)) EQ ' 3' then begin ;only attempt col(3) readin if requested
				   	if a_size(1) GE 4 then begin    ; bail if no col(3)
						   d=a(3,*)
;						   print, 'read col3 from ', file, '  : ', $
;						   strcompress(string(t(1))), ' cols ', strcompress(string(t(2))), ' points'
				    endif else goto, no_tey
				ENDIF
		    ENDIF
			nseg = dat(1,0)               ;(format C)
			samp  = total(dat(2,1:*))
	    	nul = dat(0,0)
	    	first = 1 & last = nseg
	    endelse
; common to both post_Nov-97 formats
;	    print, '# of dark points = ', fix(nul),' # of sample points = ', fix(samp)
;		print, ' size of b array = ', n_elements(b)
	    nb = fix(nul)
		dwmax = 0
	    for i = first,last do begin    ; correct for different dwell for each section
	       dw = dat(3,i)/1e3		   ; convert data to CPS (dwell in ALS files is in microseconds)
	       if dw GT dwmax then dwmax = dw
	       nf = fix(nb + dat(2,i)-1)
;	       print,'section ',i,' dwell (ms) = ',dw, ' npts = ',nf-nb+1
	       b(nb:nf) = b(nb:nf)/dw
	       c(nb:nf) = c(nb:nf)/dw
	       if n_elements(d) NE 0 then d(nb:nf) = d(nb:nf)/dw
	       nb = nf + 1
	    endfor
		dw = dwmax	; set default dwell to longest value if different dwells used

; correct dark counts for dwell (NB can have error here for format B)
		if nul GT 1 then begin  ; don't if no dark counts
;
;  NB - (sep-99) found that ALS  files have only nul-1 dark points at the END !!!
;   this caused the spike and the no-dark read in error - UGH !!
;
; NB  - (mar-02, aph) found new stxm701 spectral files, the first point
;   after dark is about zero - turn on problem ?
;   kluge added to DROP on extra point  -  UGH !!!
;
			dw = dat(3,0)/1e3
			b(0:nul-1) = b(0:nul-1)/dw
			b(nul+samp:*)=b(nul+samp:*)/dw              ; 7-apr-98 increment
			c(0:nul-1) = c(0:nul-1)/dw                      ; to avoid spike
			c(nul+samp:*) = c(nul+samp:*)/dw
			IF n_elements(d) NE 0 then BEGIN
				d(0:nul-1) = d(0:nul-1)/dw
				d(nul+samp:*) = d(nul+samp:*)/dw
;				d(nul+samp+1:*) = d(nul+samp+1:*)/dw
			ENDIF
;			print, '#dark= ',fix(nul),' dwell dark (ms) ',dw,'   #data= ', samp

			IF keyword_set(COL) THEN BEGIN
				CASE STRCOMPRESS(string(col)) OF
				' 1' : BEGIN		; OSA readin
					b = c	;set data array to the OSA signal
;					print, 'Column ', col, ' read-in'
				END
				' 2' : BEGIN		; transmission readin (default)
;					print, 'Column ', col, ' read-in'
				; do nothing
				END
				' 3' : BEGIN		; OTHER (Tey) readin
					IF n_elements(d) GT 0 then BEGIN
						b = d
;						print, 'Column ', col, ' read-in'
					ENDIF ELSE goto, no_tey
				END
				ELSE:
				ENDCASE
			ENDIF

	 ; compute average of dark counts
			bi=b(0:nul-1)
			bf=b(nul+samp:*)
			ci=c(0:nul-1)
			cf=c(nul+samp:*)
			m1i=median(bi)
			m1f=median(bf)
			n1i=median(ci)
			n1f=median(cf)

			dark_avg = (m1f(0)+m1i(0)) / 2.
;			print, string(format='("pre-dark = ",f6.3, " post-dark = ",f6.3," average =",f6.3)', m1i,m1f,dark_avg)
		endif else begin
			nul = 0
			dark_avg = 0.0
		endelse
	endelse

	file=strmid(file,strlen(file)-8,strlen(file))
	bd=b(nul:nul+samp-1)	; strip out true spectral data
	cd=c(nul:nul+samp-1)
   	ytit = ' DET + dark (c/ms)'

	ics = icst
   	IF OPT NE 'all' then ics=icst(nul:nul+samp-1) ; only preserve E's for dark if choose 'dark + data'

	if OPT EQ 'raw' then begin
		b = bd & c = cd
		ytit = 'DET [no dark correction]  (c/ms)'
	endif

	if OPT EQ 'dark' and nul GT 0 then begin
		cia1=findgen(samp)*(n1f(0)-n1i(0))/samp+n1i(0)
		bia1=findgen(samp)*(m1f(0)-m1i(0))/samp+m1i(0)
		b=bd-bia1
		c=cd-cia1
		ytit = 'DET [dark corrected] (c/ms)'
	endif

	if OPT EQ 'OSA' then begin
		b1=b(nul:nul+samp-1)
		c1=c(nul:nul+samp-1)
	   	b = b1/c1
	   	c = c1
	   	ytit = ' (DET/OSA) [no dark correction]'
	endif

	if OPT EQ 'OSA-dark' and nul GT 0 then begin
		cia1=findgen(samp)*(n1f(0)-n1i(0))/samp+n1i(0)
		bia1=findgen(samp)*(m1f(0)-m1i(0))/samp+m1i(0)
		bd=bd-bia1
		cd=cd-cia1
		if min(cd) LT 0 then cd = cd - min(cd) + 1
		b = (bd/cd)
		c = cd
		ytit = '(DET/OSA) [dark corrected]'
	endif
	x_label = string(format='("Photon Energy,    dwell ",f6.0," ms")', dw)

; ------ 19-mar-02 (aph) KLUGE to remove first real data point
; to get around bad data - not yet put in

	s={t:'1d', d:b, nd: nul, nycol: a_size(1)-1, nsect: nseg, dwell: dw, $
	    drk: dark_avg, x:ics, xl:x_label, yl: ytit, dn:c, dl:file}

	if keyword_set(pl)then begin
	   	splot,s,ytitle=ytit,_extra=e,charsize=1
	  	if OPT EQ 'raw' then oplot,s.x,s.dn/max(s.dn)*max(s.d),color=129
	  	if OPT EQ 'dark' then oplot,s.x,s.dn/max(s.dn)*max(s.d),color=129
	endif

	return, s
endif
print, 'Filename of zero length (Cancel)'
s = 0
return, s
no_tey:
print, 'No col(3) signal in file', file
s = 0
return, s
end

