; Copyright (c) 1998-2016 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	ECHEM_PLOT
;
;LAST CHANGED: ----------------------------------- 18-Dec-16 (aph)
;
;PURPOSE:
;	This function generates a sequence of png files of a spectrum with
; a superimpose symbol and line to make a track of the voltage or curent (or ...) at
; which a STXM data was measured. Developed for in situ echem data display (Dec16)
;
;CATEGORY:
;	STAND ALONE: utility
;
;CALLING SEQUENCE:
;	Result = ECHEM_STXM(file=file)
;
;CALLED FROM AXIS:
;	standalone and in aXis2000
;
;INPUTS: none. All input parameters are passed as keywords.
;
;KEYWORDS:
;	FILE 	- filename to read
;
;OUTPUTS:
;	RESULT = set of png files to use to make a movie
;
;COMMON BLOCKS:
;	@AXIS_COM	standard set of common blocks
;
;PROCEDURE:
;	plots spectrum and superimposes a symbol at each data point sequentially
;  and writes a png of each
;
; EXAMPLE
;  Image sequence to match is:3638jod-time.ncb
;       (appended, aligned, ocnvert to OD of A161006036 and A161006038
;
; Spectrum - I(t) matched to the # images is OappP-i-t-s-matched.txt
; was generated by
;	(1) IviumSoft: exporting (t,V,I) from Ivium data files
;					(161006-VP036-O-Stripping-in-situ.idf, 161006-VP036-P-Stripping-in-situ.idf)
;   (2) aXis2000: read in I(t) for O and P
;	     	set start of the time scale for P to be shifted by no-collection, OCP time (511 s)
;			Append, clean-up, S5,  & save  OappP-i-t-s.txt
;			read in spectrum 3638jod-time-all.txt derived from 3638jod-time.ncb
;				(used to get right number of time points for annotated plot)
;			starting with 3638jod-time-all.txt; add~buffer (*1e-30) then select OappP-i-t.txt (scale*1)
;			write as OappP-i-t-s-matched.txt
;	(3) IDL (licenced) compile and run echem_plot.pro
;		this will write out a png filewith a symbol and line at each time point in 3638jod-time.ncb
;	(4) EasyGIFanimator
;		select the 98 (*.png) files; set to run cycles, with 20 multiplier (0.2s/image ~20 sec)
;		write as O-P-i-t-s-matched.gif
;	(5) embed in powerpoint (the multi-gif written by EasyGIFanimator only runs in powerpoint)
;
;MODIFICATION HISTORY:
; (13-Dec-16 aph) first version
; (17-Dec-16 aph) adapt for calling from aXis2000

Function  echem_plot, file=file, group = group
@axis_com
@bsif_com
on_error,2

if keyword_Set(file) then tmp = spc_load(file=file) else begin
	file=''
	file = pickfile2(prompt='current-time plot', value = file, filter='*.txt')
	tmp = spc_load(file=file)
endelse
t = ax_name(file)
del = 0.05*(max(tmp.d) - min(tmp.d))
!y.crange = [min(tmp.d)-del, max(tmp.d)+del]

if keyword_set(group) then begin
	print, 'called get_Text with group=aXis_ID'
	header = get_text(prompt='top label ', val=tmp.dl, group=group)
endif else begin
	header = get_text(prompt='top label ', val=tmp.dl)
endelse
tmp.dl = header
if keyword_set(group) then x_label = get_text(prompt='X-axis label ', val=tmp.xl, group=group) $
    else x_label = get_text(prompt='X-axis label ', val=tmp.xl)
tmp.xl = x_label
if keyword_set(group) then y_label = get_text(prompt='Y-axis label ', val=y_label, group=group) $
	else y_label = get_text(prompt='Y-axis label ', val=y_label)

tmp = {t:tmp.t, x:tmp.x, d:tmp.d, dn:tmp.dn, dl:tmp.dl, xl:tmp.xl, yl:y_label}
scr_siz=get_screen_size()
t1_scr = scr_siz(0)*0.65
t2_scr = n_elements(tmp.x)
x_scr_size=max(t1_scr,t2_scr)
y_scr_size=0.7*x_scr_size

window, /free, xsiz= x_scr_size, ysize=y_scr_size

for i = 0, n_elements(tmp.x)-2 do begin
	splot,tmp, thick=2
	plots, tmp.x(i), tmp.d(i), psym=6, symsize=1,thick=4, color=0
	xl=[tmp.x(i),tmp.x(i)]  & yl=[0,tmp.d(i)]
	oplot, xl, yl, thick = 2, color = 0
	imag= tvrd()
	file_out = t(0)+t(1)+'_'+strtrim(string(i),2)+'.png'
	write_png,file_out, imag
 endfor

wdelete, !D.WINDOW

return, tmp
end
