; Copyright (c) 1998-2010 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		READ_APS_XRF_MAPS.PRO
;
;LAST CHANGED: ----------------------------------- 	21-Jun-2010
;
; PURPOSE:
;	This function reads APS XRF maps from ascii files
;
; CATEGORY:
;	captive to aXis2000 (only works properly in modal mode
;	data read in from aXis200:  read~images~XRF~APS
;
; CALLING SEQUENCE:
;	Result = READ_APS_XRF_IMG(file=file, filter=filter, map_num=map_num, normalize=normalize, $
;                      group=group, store=store, silent=silent, verbose = verbose)
;
; KEYWORDS:
;	FILE:  name of file (*.pgm)
;	FILTER: extension (default is '*.pgm')
;	GROUP: group_ID of parent
;	NORMALIZE: if set, modify intensities to standard ring current (value of normalize parameter
;	SILENT: if set, do not print tracking comments
;	STORE: if set, write *.axb (aXis2000 format) binary file of image
;	TYPE:   data source (currently set up for APS or CLS SGM)
;	VERBOSE:  if set print all of the header
;
; OUTPUTS:
;	returns one or more maps into aXis2000 buffers using xrm_map_selector widget
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;
; MODIFICATION HISTORY:
; (21-Jun-10 aph) first version
;-

function READ_APS_XRF_MAPS, file=file, filter=filter, map_num=map_num, normalize=normalize, $
                      group=group, store=store, silent=silent, verbose = verbose
@axis_com
on_error,2

if not keyword_set(group) then group = aXis_ID	; this makes it captive to aXis2000

if n_elements(file) eq 0 then begin  ;popup file dialog box
	if not keyword_set(filter) then filter='*.*'
	file=PICKFILE2(/Read, FILTER=filter, /LPATH, DEFPATH=defpath)
endif

if strlen(file) GT 0 THEN BEGIN
	name_bits = ax_name(file)	; sort the name into its parts
	path = name_bits(0)
	fileshort = name_bits(1)
    ext = name_bits(2)
 endif else begin
	print, 'Filename ', file,' is not defined, or does not exist'
	return, 0
endelse

; ----------------- read APS XRF maps  ----------------

openr, unit, file, /get_lun
a=''

; ------- find photon energy
while strmid(a,0,8) NE '# Angle:' DO BEGIN
	readf, unit, a
	if keyword_set(verbose) then axis_log, a
endwhile
energy = float(strmid(a,strpos(a,'record: ')+8,8))
if keyword_set(verbose) then print, 'energy = ', energy

; ------- find list of headers
while a NE '# Column Headings:' DO BEGIN
	readf, unit, a
	if keyword_set(verbose) then print, a
endwhile
readf, unit, a
aps_label_string = strmid(a,1)
if keyword_set(verbose) then print, 'LABELS'
if keyword_set(verbose) then print, aps_label_string

; ------ read remaining parameters defining maps
while strmid(a,0,1) NE '*' DO BEGIN
	readf, unit, a
	if keyword_set(verbose) then axis_log, a
endwhile
a = strmid(a,1)
aps_par = intarr(5)
reads,a,aps_par

n_maps = aps_par(2)+2
n_rows = aps_par(4)
n_cols = aps_par(3)
if keyword_set(verbose) then begin
	axis_log, '# of maps: ' + string(n_maps)
	axis_log, '# of rows: ' + string(n_rows)
	axis_log, '# of cols: ' + string(n_cols)
endif
n_lines = n_rows*n_cols
aps_maps=fltarr(n_maps, n_lines)
; ------------------- read labels of the maps from aps_label_string
aps_label = strarr(n_maps)
reads, aps_label_string, aps_label, format='(90a21)'

readf, unit, aps_maps

close,unit & free_lun, unit


; ---------- select map -----------------------------------

maps = xrf_map_selector(list = aps_label, group_leader=group)
if keyword_set(verbose) then print, maps

; ---- write all maps selected into assigned buffers
; -----------------  generate x, y axes and other parameters
x_step = 1  & x_start = 0
y_step = 1  & y_start = 0.
x = findgen(n_cols)  & y = findgen(n_rows)
x = x_start + x*x_step
y = y_start + y*y_step
dwell = 1000.
xl = string(FORMAT='("x (pixels)     E = ",f8.3," eV     dwell = ",f6.1," ms")', $
      energy,  dwell)
yl = 'APS XRF map from file ' + fileshort

for i = 1,9 do begin
	if maps(i-1) GT 0 then begin
		axis_log, 'buffer ' + strtrim(string(i),2) + ' = ' + aps_label(maps(i-1)-1)
		image_data = aps_maps(maps(i-1)-1,*)
		image_data = reform(image_data, n_cols, n_rows)
		dl = aps_label(maps(i-1)-1)
		s={t:'2d', d:image_data, x:x, y:y, e: energy, xl:xl, yl: yl, dl:dl}
		Curbuf = i
		HANDLE_VALUE, Data(Curbuf), s, /SET
	 	Plotbuf, CurBuf
	endif
endfor

if not keyword_set(silent) then axis_log, 'read APS XRF maps from ' + fileshort

s = 1

return, s
end

; --- first version for single map read-in - park for now (21-Jun-2010)
;if not keyword_set(map_num)then begin
; -------- print labels --------
;	for i=0,n_maps-1 do $
;		axis_log, string(i, format='(I2)') + ' ' + strtrim(aps_label(i),2)
;	if keyword_set(group) then $
;		i_map = get_num(prompt='map number ? ', val = i_map, group = aXis_ID) $
;	else $
;		i_map = get_num(prompt='map number ? ', val = i_map)
;endif else i_map = map_num
