; Copyright (c) 1998-2022 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	READ_SGM_SPECTRA
;
;LAST CHANGED: ----------------------------------- 12 May 2022 (aph)
;
;PURPOSE:
;	This function reads  data  from an hdf5 file written by CLS SGM
; TEY, Io, clock, photodiode, and partial (or total FY can be read
; the SDD spectrum is displayed during read-in so the user can identify the limits for the peak of interest
;
;CATEGORY:
;	STAND ALONE or called in aXis2000:  spectral processing
;
;CALLING SEQUENCE:
; 	s = read_sgm(file=file,nof=nof, struct_only=struct_only, silent=silent)
;
;CALLED FROM AXIS:
;	Read~spectrum~SGM
;
;INPUTS: none
;
;KEYWORDS:
;   struct_only	= read & return structure
;	file		= name of file (if not in ncall, will launch dialog)
;	nof	= no name filter
;	silent		= do not print progress

;OUTPUTS:
;    returns spectrum
;
;COMMON BLOCKS:
;	axis_com
;
;MODIFICATION HISTORY:
; (26-Feb-21 aph) first version
; (02-Mar-21 aph) add read-in of user-selected FY spectra
; (02-May-21 aph) fix up on-the-fly sum (was returning NaN error) ==> code bad from 2-MArch to 2-May 2021 !!!!
; (18-Apr-22 aph) adapt to process any entry
; (12-May-22 aph) implement "A" option to read all TEY spectra and output as a spectral linescan
;-

FUNCTION read_sgm,file=file,nof=nof, struct_only=struct_only, silent=silent

@axis_com
@sgm_com

on_error,2

; get filename if not supplied in call
fltr='*.hdf5'
if keyword_set(nof) then fltr='*'
title = 'Select SGM hdf5 data file'
if not keyword_set(file) then file = pickfile2(/READ, FILTER=fltr, /LPATH, DEFPATH=defpath, TITLE = TITLE)
; print, 'File to open is ', file
if strlen(file) EQ 0 THEN RETURN, 0

; use h5_parse to get data as an IDL structure
t = ax_name(file)  & fileshort = t(1)
if NOT keyword_set(silent) then print, 'Reading from file ', fileshort

ax_wait

if keyword_set(struct_only) then a=h5_parse(file)  else a=h5_parse(file,/read_data)

ax_wait, /off

; ------- report the number of entries  (assumes all SGM hdf5 files will have same first 6 tages)
print, 'n_tags = ', n_tags(a)
n_entry = n_tags(a)-6
if n_tags(a) GT 7 then text = 'There are ' + strtrim(string(n_entry),2) + ' entries' else text = 'only 1 entry'
axis_log, text
if n_tags(a) EQ 7 then one_spectrum=1 else one_spectrum=0

; -- return the structure
IF keyword_set(struct_only) THEN RETURN, a

;  -----------  multiple spectra - select entry to read  TEY/Io
IF one_spectrum  EQ 0 then begin
	entry_num='A'
	if widget_info(/active) EQ 0 then $
		entry_num = get_text(prompt= 'Choose entry (#) or all (A) (CAUTION: all is slow!)', val=entry_num) else $
		     entry_num = get_text(prompt= 'Choose entry (#) or all (A) (CAUTION: all is slow!)', val=entry_num, group=axis_ID)
	entry_num = strtrim(strupcase(entry_num),2)
	if entry_num EQ 'A' then begin
;  --------- read all TEY spectra and output as a spectral linescan  (12 May 2022)
		a_tag_names=tag_names(a)
		ev = reform(a.(7).data.en._data)		; find size of array (nb may be variable !!)
		ev = ev(1:n_elements(ev)-1)
		emin = min(ev, max = emax)
		print, '0  ',  emin , emax, n_elements(ev)

		eva = fltarr(n_entry, 1.2*n_elements(ev))
		teyn = eva

; -------- read all the energies to find largest emin and smallest emax
		for i = 1,n_entry-1 do begin
			ev = reform(a.(i+6).data.en._data)		; get energy
			ev = ev(1:n_elements(ev)-1)
			t=min(ev, max = tm)
			emin = min(emin,t)
			emax = max(emax, tm)
;			print, i, emin , emax, n_elements(ev)
		endfor
; -------- define fixed , monotonic energy scale
		np=300
		if widget_info(/active) EQ 0 then $
		     np = get_num(prompt= 'Number of points', val=np) else $
		     np = get_num(prompt= 'Number of points', val=np, group=axis_ID)
		e_step = (emax-emin)/np
		E_final=emin + e_step*findgen(np)

; loop through each entry
		for i = 0,n_entry-1 do begin
			index = i+6
			print, 'entry label is ', a_tag_names(index)			; print the actual entry# in the file (they are out of order)
			ev = reform(a.(index).data.en._data)		; get energy
;			help, ev
;			help, eva
			eva(i,0:n_elements(ev)-2) = ev(1:n_elements(ev)-1)		; first point is NaN
			tey = reform(a.(index).data.tey._data)/reform(a.(index).data.I0._data)
			t = {t:'1d', d:tey, x:ev, xl: 'Energy (eV)', yl:'y', dn: tey, dl: 'v'}
			t = ax_sort_mono(t, /no_dup)		; force monotonic
			tey = t.d
			ev = t.x
			tey2d=fltarr(2, n_elements(ev))
			for i = 0, n_elements(ev)-1 do begin
				tey2d(0,*)=ev
				tey2d(1,*)=tey
			endfor
			; ------- interpolate the irregularly gridded data (ev, tey) to a regular grid (E_final)
;			tey_interp = interpolate(tey2d, E_final)
			tey_interp = ax_interp(t, mesh= E_final)

; 	get (x,y) positions for this spectrum
			xp = a.(index).sample.positioner.xp._data
			yp = a.(index).sample.positioner.yp._data
			print, a_tag_names(index), '     X= ', xp, '     Y= ', yp
	; -------- construct spectrun  as aXis2000 1d structure
			dl = fileshort +' ' + a_tag_names(index) + ' tey / Io'
			yl = string(xp, format='(F7.4)')
			yl = yl  + '     '  +  string(yp, format='((F7.4))')
			teyn_out = {t:'1d', d:tey_interp, x:E_final, xl: 'Energy (eV)', yl:yl, dn: tey_interp, dl: dl}

	; ------- write to disk with auto-naming
			t = ax_name(file)
			file_out = t(0) + t(1) + '_'+ strtrim(string(i),2) + '.txt'
			t = spc_save(teyn_out, file = file_out, /noLbl)
			axis_log, 'SGM TEY spectrum ' + a_tag_names(index) + ' read from file ' + fileshort
		endfor
		return, teyn

	endif else begin
	;  --------- read one TEY spectrum (12 May 2022)
		print, 'reading TEY/Io from ',entry_num
	    if  n_entry GE 10 AND entry_num GT 1 then begin
	    	entry_n= fix(entry_num) + 7
	    endif else entry_n= fix(entry_num) + 5
	    if entry_num EQ 10 then entry_n = 7			;	kluge ! because SGM tags do not use 0# for 1-9
; --------- so far, this will only work for multi-spectra with less than 19 samples
		a_tag_names=tag_names(a)
		print, 'tag number for entry_n = ', entry_num,' is ', entry_n
		print, 'entry label is ', a_tag_names(entry_n)

	; ---------- get energies for entry_n      NB first point reads as NaN so remove it
		ev = reform(a.(entry_n).data.en._data)
		ev = ev(1:n_elements(ev)-1)

	; ----------- read TEY/Io for entry_num using the tag number
		teyn = reform(a.(entry_n).data.tey._data)/reform(a.(entry_n).data.I0._data)
		teyn = teyn(1:n_elements(teyn)-1)		; first point is NaN
	; 	get (x,y) positions for this spectrum
		xp = a.(entry_n).sample.positioner.xp._data
		yp = a.(entry_n).sample.positioner.yp._data

	; -------- construct spectral file
		dl = fileshort + entry_num + ' tey / Io'
		yl = string(xp, format='(F7.4)')
		yl = yl  + '     '  +  string(yp, format='((F7.4))')
		teyn_out = {t:'1d', d:teyn, x: ev, xl: 'Energy (eV)', yl:yl, dn: teyn, dl: dl }
		teyn_out = ax_sort_mono(teyn_out)		; force monotonic

		axis_log, 'SGM TEY spectrum # ' + strtrim(entry_num,2) + ' read from file ' + fileshort
		return, teyn_out
	endelse
ENDIF

;  PRE-Mar-22 code which assumes only 1 entry (hdf5 files sent by Jay Dynes after Feb21 run)
; ------- CHOOSE TYPE OF DATA to read
if n_elements(sgm_type) EQ 0 then sgm_type = 'T'	;	default to TEY

if widget_info(/active) EQ 0 then $
	sgm_type = get_text(prompt = ' <T>ey, <I>o, <C>lock,<P>hotodiode, <F>luorescece yield', val=sgm_type) $
	else $
		sgm_type = get_text(prompt = ' <T>ey, <I>o, <C>lock,<P>hotodiode, <F>luorescece yield', val=sgm_type, group = aXis_ID)
sgm_type=strmid(sgm_type,0,1)
sgm_type =strupcase(sgm_type)


CASE sgm_type OF

'T': BEGIN
	;  --------- get the TEY data
	tey = reform(a.entry1.data.tey._data)
	tey = tey(1:n_elements(tey)-1)
	; -------- construct spectral file
	dl = fileshort + ' tey'
	tey={t:'1d', d:tey, x: ev, xl: 'Energy (eV)', dn: tey, dl: dl }
	axis_log, 'SGM TEY spectrum read from ' + fileshort

	return, tey
	END

'I': BEGIN
	;  --------- get the I0  data
	tey = reform(a.entry1.data.I0._data)
	tey = tey(1:n_elements(tey)-1)
	; -------- construct spectral file
	dl = fileshort + ' I0'
	io={t:'1d', d:tey, x: ev, xl: 'Energy (eV)', dn: tey, dl: dl }
	axis_log, 'SGM I0 spectrum read from ' + fileshort

	return, io
	END

'C': BEGIN
	;  --------- get the CLOCK data
	tey = reform(a.entry1.data.clock._data)
	tey = tey(1:n_elements(tey)-1)
	; -------- construct spectral file
	dl = fileshort + ' clock (s)'
	clock={t:'1d', d:tey, x: ev, xl: 'Energy (eV)', dn: tey, dl: dl }
	axis_log, 'SGM CLOCK signal read from ' + fileshort

	return, clock
	END

'P': BEGIN
	;  --------- get the Photodiode data
	tey = reform(a.entry1.data.PD._data)
	tey = tey(1:n_elements(tey)-1)
	; -------- construct spectral file
	dl = fileshort + ' photodiode '
	pd={t:'1d', d:tey, x: ev, xl: 'Energy (eV)', dn: tey, dl: dl }
	axis_log, 'SGM photodiode signal read from ' + fileshort

	return, pd
	END


'F': BEGIN

	if n_elements(ssd_ch) EQ 0 then ssd_ch = 'A'
	if widget_info(/active) EQ 0 then $
	ssd_ch = get_text(prompt = 'SDD<1>,SDD<2>,SDD<3>,SDD<4> or <A>vg', val=ssd_ch) $
	else $
		ssd_ch = get_text(prompt = 'SDD<1>,SDD<2>,SDD<3>,SDD<4> or <A>vg', val=ssd_ch, group = aXis_ID)
	ssd_ch=strmid(ssd_ch,0,1)

;	----------- read average SDD spectrum for requested SDD
	CASE ssd_ch OF

	'1':	sdd = a.entry1.data.sdd1._data
	'2':	sdd = a.entry1.data.sdd2._data
	'3':	sdd = a.entry1.data.sdd3._data
	'4':	sdd = a.entry1.data.sdd4._data
	ELSE: BEGIN
			sdd = a.entry1.data.sdd1._data
			sdd = sdd + a.entry1.data.sdd2._data
			sdd = sdd + a.entry1.data.sdd3._data
			sdd = sdd + a.entry1.data.sdd4._data
			sdd = sdd/4
		  END
	ENDCASE

	t = size(sdd)
	n_x = t(1)
	n_e = t(2)
	x = 10*findgen(n_x)			; convert from channel # to X-ray emission energy (10 eV/channel)
	sum = fltarr(n_x)

; -------- replace NaN values with 0
	sdd_nan = where(~FINITE(sdd), count)
	if count GT 0 then sdd(sdd_nan)=0.

; -------- integrate SDD spectrum recorded at each photon energy
	for i = 1, n_e - 1 do begin
		sum = sum + sdd(*,i)
	endfor

	; -------- construct SDD X-ray fluorescence file
	dl = fileshort + ' SDD ' + ssd_ch
	f={t:'1d', d:sum, x: x, xl: 'channel', dn: sum, dl: dl }
;	axis_log, 'SDD '+ ssd_ch + ' average fluorescence  read from ' + fileshort

;	return,f
	if widget_info(/active) EQ 0 then begin
		splot, f
	endif else begin
		CurBuf_tmp = CurBuf
		HANDLE_VALUE, Data(0), f, /SET
		Label(0) = f.dl
		PlotBuf, 0
		CurBuf = CurBuf_tmp
	endelse

; ---------  plot and ask user to select X-ray emission energy integration limits
	if n_elements(ch_lo) EQ 0 then E_lo  = 10 else E_lo=ch_lo*10
	if widget_info(/active) EQ 0 then $
		E_lo = fix(get_num(prompt='X-ray E: lower limit', val=E_lo)) $
		else  E_lo = fix(get_num(prompt='X-ray E: lower limit', val=E_lo, group = axis_ID))
	if n_elements(ch_hi) EQ 0 then ch_hi= 10*(n_x - 1) else E_hi=ch_hi*10
	if widget_info(/active) EQ 0 then $
		E_hi = fix(get_num(prompt='X-ray E: upper limit', val=E_hi)) $
			else E_hi = fix(get_num(prompt='X-ray E: upper limit', val=E_hi, group = axis_ID))
	ch_lo = E_lo / 10		; convert back to channel #
	ch_hi = E_hi / 10

; --------- integrate over user selected limits
	print, 'Integrating SDD signal from ', E_lo,' to ', E_hi
	fy = fltarr(n_e-1)
	for i = 0, n_e-2 do begin
		fy(i) = reform(total(sdd(ch_lo:ch_hi,i)))
	endfor

; -------- construct spectral file
	if ssd_ch EQ 'A' then tl = 'Average' else tl = ssd_ch
	dl = fileshort + ' PFY from SDD ' + tl + '  I: ' + string(10*ch_lo, format='(I4)') + ' - ' + string(10*ch_hi, format='(I4)')
	t= reform(a.entry1.data.en._data)
	ev = t(1:n_e-2)
	fy = fy(1:n_elements(fy)-1)
	tmp = {t:'1d', d: fy, x: ev, xl: 'Photon energy (eV)', dn: fy, dl: dl }

	axis_log, 'SDD '+ tl + ' partial fluorescence yield read from ' + fileshort

	return,tmp

	END


ELSE: print, 'invalid data identifier'


ENDCASE

END
