; Copyright (c) 1998-2005 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	READ_UVIEW
;
;LAST CHANGED: ----------------------------------- 01-Nov-05
;
;PURPOSE:
;	This function reads in an image from (*.dat) binary format
; of UVIEW2002 camera program from Elmitec PEEM-III
; Format is described in the UVIEW2002 manual
; returns image as an aXis2000 2d structure
;
;CATEGORY:
;	AXIS: image utility (also runs stand alone)
;
;CALLING SEQUENCE:
;	Result = READ_UVIEW(file = file, skip=skip, verbose=verbose)
;
;CALLED FROM AXIS:
;	->Image->Elmitec->{dat}
;
;INPUTS:
;	All input parameters are passed as keywords.
;
;KEYWORDS:
; FILE  - name of file; if supplied, and exists, reads silently
; SKIP - do not read header; value of skip is image size (square)
; VERBOSE - if set, print all of file and image headers
;
;OUTPUTS: none
;
;COMMON BLOCKS:
;	@AXIS_COM	standard set of common blocks
;
; DETAILS
; Format of Uview binary mage files (*.dat)
; has 3 parts
;  Fileheader  (104 bytes)
;  Imageheader (48 bytes)
;  Data (width x height x 2 bytes)
;
; struct UKFileHeader{		 size of(UKFileHeader):104
;	20 char id[20];
;	 2 short size;
;	 2 short version;
;	 2 short BitsPerPixel; (= 16 for Sensicam,  storage, not 12-bit acquisition)
;    6  [not advetised 6-byte 'bye' for getting long to proper spot ]
;        // 6 bytes inserted to get to next 8 byte boundary.
;           LONGLONG seems to need to start at 8 byte boundary
;  	 8 LONGLONG starttime;
;	 2 short ImageWidth;
;	 2 short ImageHeight;
;	 2 short NrImages;
;	 2 short spareShort;
;	56 BYTE spare[56];
; };
;
; struct UKImageHeader   size of(UKImageHeader):48
;  2	short size;
;  2	short version;
;  4    // 4 bytes inserted to get to next 8 byte boundary
;  8	LONGLONG imagetime;
;  4	long LEEMdata1_source;
;  4	float LEEMdata1_data;
;  2	short spin;
;  2	short spareShort;
;  4	float LEEMdata2_data;
; 16	BYTE spare[16];
;	};

;
; ISSUES
; conversion from 8 bytes to a long_long time
; imagetime =   80  34 169 121 151 231 193   1
; may be same as
; imagetime =   2041127504 (ulong)
;
; NEW FORMAT WITH UVIEW2002   version 1.3.10 (October 2005)
; Format of Uview binary mage files (*.dat)
; has 3 parts
;  Fileheader  (104 bytes)
;  Imageheader (48 bytes - if Version <5; 288 bytes if version >=5)
;  Data (width x height x 2 bytes)
;
;
;
;
;MODIFICATION HISTORY:
; (29-apr-02 aph) first written
; (03-may-02 aph)  get all of header using clues from Uwe
; (09-jun-04 aph) remove tag line in image time (under verbose); check that E is added
; (01-Nov-05 aph) adapt for new data format (1.3.10 - Oct-05); NB skip not working
;-

FUNCTION Read_uview, file = file, skip=skip, verbose=verbose
@axis_com
on_error,2

s = 0

IF NOT keyword_set(file) then begin
	fltr='*.dat'
	file = pickfile2(/READ, FILTER=fltr, /LPATH, DEFPATH=defpath)
ENDIF
if strlen(file) EQ 0 THEN RETURN, s ; bail-out if no filename
WIDGET_CONTROL, Bad_Id = BadID, /Hourglass
t = ax_name(file)
fileshort = t(1)
energy = 5.
dwell = 85.   ; defalt dwell = 85 ms
; energy = get_num(prompt='Photon energy', val = energy, group = axis_ID)
; ---------- read File Header -------------
openu,lun, file,/get_lun

ID = '                    '
readu, lun, ID
if keyword_set(verbose) then print, 'ID = ', ID,'|'
filesize=1
readu, lun, filesize
if keyword_set(verbose) then  print, 'fileheader size = ', filesize
version = 1
readu, lun, version
if keyword_set(verbose) then  print, 'version = ', version
skip_file_hdr = byte(1)
if keyword_set(skip) then begin
	for i= 0, 79 do readu, lun, skip_file_hdr
	tmp=uintarr(skip,skip)
	readu, lun, tmp
	goto, skip_file_header
endif
BitsPerPixel = 1
readu, lun, BitsPerPixel
if keyword_set(verbose) then print, 'BitsPerPixel = ', BitsPerPixel
for i = 0,5 do readu, lun, skip_file_hdr
starttime = bytarr(8)
readu, lun, starttime
start_t = ulong(0)
for i = 0, 7 do start_t = start_t + starttime(7-i)*256^(7-i)
if keyword_set(verbose) then print, 'starttime = ', start_t  ; ,' from ', starttime
ImageWidth  = 1
ImageHeight = 1
readu, lun, ImageWidth
readu, lun, ImageHeight
print, 'UView2002 image:', fileshort, ' size  = ', ImageWidth,' x ', ImageHeight
NrImages = 1
readu, lun, NrImages
if keyword_set(verbose) then  print, 'NrImages = ', NrImages
spareShort = 1
readu, lun, spareShort ; & print, 'spareShort = ', spareShort
spare = byte(1)
for i = 0, 55 do readu, lun, spare

skip_file_header:

; --- read image header
if version LT 5 then begin			; Image_header is 48 bytes
	if keyword_set(skip) then begin
		for i= 0, 47 do readu, lun, skip_file_hdr
		tmp=uintarr(skip,skip)
		readu, lun, tmp
		goto, skip_image_header
	endif
	filesize2 = 1
	readu, lun, filesize2
	if keyword_set(verbose) then print, 'Image header size = ', filesize2
	version_img = 1
	readu, lun, version_img
	if keyword_set(verbose) then  print, 'image version = ', version_img
	for i = 0, 3 do readu, lun, skip_file_hdr
	imagetime = bytarr(8)
	readu, lun, imagetime
	image_t = ulong(0)
	for i = 0, 7 do image_t = image_t + imagetime(7-i)*256^(7-i)
	if keyword_set(verbose) then print, 'imagetime = ', image_t  ; ,' from ', imagetime
	LEEM1 = long(1)
	readu, lun, LEEM1
	if keyword_set(verbose) then print, 'LEEM1 = ', LEEM1
	LEEM1_data = 1.
	readu, lun, LEEM1_data
	if keyword_set(verbose) then print, 'LEEM1_data = ', LEEM1_data
	spin = 1
	readu, lun, spin
	if keyword_set(verbose) then  print, 'spin = ', spin
	spareShort = 1
	readu, lun, spareShort
	if keyword_set(verbose) then print, 'spareShort = ', spareShort
	LEEM2_data = 1.
	readu, lun, LEEM2_data
	if keyword_set(verbose) then print, 'LEEM2_data = ', LEEM2_data
	for i = 0, 15 do readu, lun, spare
endif else begin
; ------- adapt to post version 5 format
;short size; // 2 Bytes
;short version; // 2 Bytes
;short ColorScaleLow,ColorScaleHigh; // from 1.3.10
;
;LONGLONG imagetime; // 8 Bytes
;short MaskXShift,MaskYShift; // from 1.3.10
;// before 1.3.10 4 filler Bytes
;BYTE spare[4]; // 4 Bytes
;short spin; // 2 Bytes
;short LEEMdataVersion; // 2 Bytes
;unsigned char LEEMdata[256]; // 256 Bytes Overlay data
;// 4 Bytes filler

	if keyword_set(skip) then begin
		for i= 0, 283 do readu, lun, skip_file_hdr
;		tmp=uintarr(skip,skip)
		help, tmp
		readu, lun, tmp
		goto, skip_image_header
	endif
	filesize2 = 1
	readu, lun, filesize2
	if keyword_set(verbose) then print, 'Image header size = ', filesize2
	version_img = 1
	readu, lun, version_img
	if keyword_set(verbose) then  print, 'image version = ', version_img
	ColorScaleLow= 0 &  ColorScaleHigh = 255
	readu, lun, ColorScaleLow,ColorScaleHigh
	if keyword_set(verbose) then  print, 'ColorScaleLow, ColorScaleHigh ', ColorScaleLow,ColorScaleHigh
	imagetime = bytarr(8)
	readu, lun, imagetime
	image_t = ulong(0)
	for i = 0, 7 do image_t = image_t + imagetime(7-i)*256^(7-i)
	if keyword_set(verbose) then print, 'imagetime = ', image_t  ; ,' from ', imagetime
	MaskXShift=0 & MaskYShift=0
	readu, lun, MaskXShift, MaskYShift
	if keyword_set(verbose) then  print, 'MaskXShift, MaskYShift ',  MaskXShift, MaskYShift
	spin = 1
	readu, lun, spin
	if keyword_set(verbose) then  print, 'spin = ', spin
	LEEM2_dataVersion = 1.
	readu, lun, LEEM2_dataVersion
	if keyword_set(verbose) then print, 'LEEM2_dataVersion = ', LEEM2_dataVersion
	for i = 0, 259 do readu, lun, spare
endelse

; ------ read image data - CHECK ALLOC as faster method
tmp = uintarr(ImageWidth, ImageHeight)
readu, lun, tmp
skip_image_header:
close, lun
free_lun, lun

ts = size(tmp)
if ts(0) EQ 2 then begin
nx = ts(1) &  ny = ts(2)
endif else begin
nx = ts(2) &  ny = ts(3)	; adapt for multi-image files
endelse
x = findgen(nx)
y = findgen(ny)
xl = string(FORMAT='("x (um)     E = ",f8.3," eV     dwell = ",f7.2," ms")', $
      energy, dwell)

s = {t:'2d', x:x, y:y, E: energy, d:float(tmp), $
     xl:xl, yl:'Elmitec-PEEM     y (um)', dl: fileshort}
axis_log, 'read Elmitec file ' + fileshort
return, s
end